#include "UG_IO_LIB.h"

INT_ ug_io_read_grid_file
 (char Grid_File_Name[],
  INT_ Message_Flag,
  INT_ M_BL_Thickness,
  INT_ M_Initial_Normal_Spacing,
  INT_ M_Surf_Grid_BC_Flag,
  INT_ M_Surf_ID_Flag,
  INT_ M_Surf_Reconnection_Flag,
  INT_ M_Vol_ID_Flag,
  INT_ Bnd_Only_Flag,
  INT_ No_Quad_Flag,
  INT_ Renumber_ID_Flag,
  INT_ Separate_by_ID_Flag,
  INT_ *Number_of_BL_Vol_Tets,
  INT_ *Number_of_Nodes,
  INT_ *Number_of_Bnd_Edges,
  INT_ *Number_of_Surf_Quads,
  INT_ *Number_of_Surf_Trias,
  INT_ *Number_of_Vol_Hexs,
  INT_ *Number_of_Vol_Pents_5,
  INT_ *Number_of_Vol_Pents_6,
  INT_ *Number_of_Vol_Tets,
  INT_1D ** Bnd_Edge_ID_Flag,
  INT_2D ** Bnd_Edge_Connectivity,
  INT_1D ** Surf_Grid_BC_Flag,
  INT_1D ** Surf_ID_Flag,
  INT_1D ** Surf_Reconnection_Flag,
  INT_4D ** Surf_Quad_Connectivity,
  INT_3D ** Surf_Tria_Connectivity,
  INT_8D ** Vol_Hex_Connectivity,
  INT_1D ** Vol_ID_Flag,
  INT_5D ** Vol_Pent_5_Connectivity,
  INT_6D ** Vol_Pent_6_Connectivity,
  INT_4D ** Vol_Tet_Connectivity,
  DOUBLE_3D ** Coordinates,
  DOUBLE_1D ** Initial_Normal_Spacing,
  DOUBLE_1D ** BL_Thickness)

{

/*
 * Read grid data from a standard UG surface or volume grid file.
 * 
 * UG_IO LIB : Unstructured Grid - Input/Output Routine Library
 * $Id: ug_io_read_grid_file.c,v 1.152 2024/11/02 06:01:09 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  FILE *Grid_File;

  CHAR_UG_MAX Case_Name, File_Name, Name_Text, Text;
  CHAR_21 File_Compression_Suffix_, File_Format_Suffix_, File_Mode_Suffix_,
          File_Type_Suffix_, Mode;

  INT_1D *Map = NULL;
  INT_1D *Surf_Error_Flag = NULL;
  INT_1D *Quad_Tria_Flag = NULL;

  INT_ Close_Flag, Error_Flag, File_Data_Type_, File_Format_, File_Mode_,
       File_Name_Length, File_Type_Format_Flag_, Max_ID, Min_ID,
       Number_of_Bnd_Nodes, Number_of_IDs, Number_of_Nodes_Dim,
       Number_of_Surf_Faces, Read_Task_Flag, Vol_ID_Max, Vol_ID_Min;
  INT_ Number_of_Vol_Elems = 0;

  double dc0;

  dc0 = 0.0;

  // if reorder and separate by surface ID option is on
  // then the no quad option must also be on

  if (Separate_by_ID_Flag)
    No_Quad_Flag = 1;

  // get file info

  Error_Flag = ug_io_file_info (Grid_File_Name, Case_Name, 
                                File_Compression_Suffix_, File_Format_Suffix_,
                                File_Mode_Suffix_, File_Type_Suffix_,
                                0, 3, &File_Data_Type_,
                                &File_Format_, &File_Mode_,
                                &File_Type_Format_Flag_);

  // exit for error

  if (Error_Flag) 
    return (Error_Flag);

  // exit if not a surface or volume grid file

  if (File_Data_Type_ != UG_IO_SURFACE_GRID && File_Data_Type_ != UG_IO_VOLUME_GRID)
    return (-1);

  // use abs of file format

  File_Format_ = ug_abs (File_Format_);

  // output header

  if (Message_Flag >= 1)
  {
    ug_message (" ");
    ug_message ("UG_IO    : INPUT GRID");
    ug_message (" ");
    ug_message ("UG_IO    : Reading Data");
    ug_message (" ");

    ug_cpu_message ("");
  }

  // set file name without compression suffix

  File_Name_Length = (INT_) strlen (Grid_File_Name)
                   - (INT_) strlen (File_Compression_Suffix_);

  strcpy (File_Name, "");
  strncat (File_Name, Grid_File_Name, File_Name_Length);

  // uncompress file

  Error_Flag = ug_uncompress_file (Grid_File_Name);

  if (Error_Flag)
  {
    ug_error_message ("*** ERROR 609 : unable to uncompress grid file ***");
    return (609);
  }

  // zero boundary only option for tmp files and surface grids

  if (File_Mode_ == UG_FIO_TMP_FILE_MODE || File_Data_Type_ == UG_IO_SURFACE_GRID)
    Bnd_Only_Flag = 0;

  // initialize grid dimensions

  *Number_of_BL_Vol_Tets = 0;
  *Number_of_Nodes = 0;
  *Number_of_Bnd_Edges = 0;
  *Number_of_Surf_Quads = 0;
  *Number_of_Surf_Trias = 0;
  *Number_of_Vol_Hexs = 0;
  *Number_of_Vol_Pents_5 = 0;
  *Number_of_Vol_Pents_6 = 0;
  *Number_of_Vol_Tets = 0;

  if (Bnd_Only_Flag)
    Number_of_Bnd_Nodes = -1;
  else
    Number_of_Bnd_Nodes = 0;

  // make two passes

  // the first pass reads the dimensions and the second reads the data

  Read_Task_Flag = 1;

  do
  {
    // open file and set close flag

  // open file and set close flag

    if (File_Mode_ == UG_FIO_TMP_FILE_MODE)
      strcpy (Mode, "r_tmp");
    else
      strcpy (Mode, "r");

    if (strcmp (File_Type_Suffix_, ".cfd++") == 0 ||
        strcmp (File_Type_Suffix_, ".cgns") == 0 ||
        strcmp (File_Type_Suffix_, ".face") == 0 ||
        strcmp (File_Type_Suffix_, ".mesh") == 0 ||
        strcmp (File_Type_Suffix_, ".meshb") == 0 ||
        strcmp (File_Type_Suffix_, ".foam") == 0)
      Close_Flag = 0;

    else {
      Grid_File = ug_fopen (File_Name, Mode);

      if (Grid_File == NULL)
      {
        ug_error_message ("*** ERROR 609 : unable to open grid file ***");
        Error_Flag = 609;
      }

      Close_Flag = 1;
    }

    // read grid file

    if (Error_Flag == 0)
    {
      if (strcmp (File_Type_Suffix_, ".face") == 0)
        Error_Flag = ug_io_read_tetgen (Case_Name, Read_Task_Flag,
                                        Number_of_Nodes, 
                                        Number_of_Surf_Trias,
                                        Number_of_Vol_Tets,
                                        *Surf_ID_Flag,
                                        *Surf_Tria_Connectivity,
                                        *Vol_Tet_Connectivity,
                                        *Coordinates);

      else if (strcmp (File_Type_Suffix_, ".fgrid") == 0 ||
               strcmp (File_Type_Suffix_, ".ufast") == 0)
      {
        Error_Flag = ug_io_read_fgrid (Grid_File, File_Format_,
                                       Read_Task_Flag,
                                       &Number_of_Bnd_Nodes, Number_of_Nodes, 
                                       Number_of_Surf_Trias, Number_of_Vol_Tets,
                                       *Surf_ID_Flag, *Surf_Tria_Connectivity,
                                       *Vol_Tet_Connectivity,
                                       *Coordinates);

        if (Error_Flag == 0 && Read_Task_Flag == 2 &&
            strcmp (File_Type_Suffix_, ".ufast") != 0)
          ug3_read_gdata (Grid_File, File_Format_,
                          *Number_of_Nodes,
                          *Number_of_Surf_Quads, *Number_of_Surf_Trias,
                          *Surf_Grid_BC_Flag, *Surf_Reconnection_Flag,
                          *Initial_Normal_Spacing, *BL_Thickness);
      }

      else if (strcmp (File_Type_Suffix_, ".mesh") == 0 ||
               strcmp (File_Type_Suffix_, ".meshb") == 0)
        Error_Flag = ug3_read_mesh (File_Name,
                                    Read_Task_Flag,
                                    &Number_of_Bnd_Nodes,
                                    Number_of_Nodes,
                                    Number_of_Bnd_Edges,
                                    Number_of_Surf_Quads,
                                    Number_of_Surf_Trias,
                                    Number_of_Vol_Hexs,
                                    Number_of_Vol_Pents_5,
                                    Number_of_Vol_Pents_6,
                                    Number_of_Vol_Tets,
                                    *Bnd_Edge_ID_Flag,
                                    *Bnd_Edge_Connectivity,
                                    *Surf_ID_Flag,
                                    *Surf_Quad_Connectivity,
                                    *Surf_Tria_Connectivity,
                                    *Vol_Hex_Connectivity,
                                    *Vol_ID_Flag,
                                    *Vol_Pent_5_Connectivity,
                                    *Vol_Pent_6_Connectivity,
                                    *Vol_Tet_Connectivity,
                                    *Coordinates);

      else if (strcmp (File_Type_Suffix_, ".bdf") == 0 || strcmp (File_Type_Suffix_, ".nas") == 0)
        Error_Flag = ug_io_read_nas (Grid_File, Read_Task_Flag,
                                     Number_of_BL_Vol_Tets,
                                     &Number_of_Bnd_Nodes, Number_of_Nodes, 
                                     Number_of_Surf_Quads, Number_of_Surf_Trias,
                                     Number_of_Vol_Hexs,
                                     Number_of_Vol_Pents_5, Number_of_Vol_Pents_6,
                                     Number_of_Vol_Tets,
                                     *Surf_Grid_BC_Flag, *Surf_ID_Flag,
                                     *Surf_Quad_Connectivity, *Surf_Reconnection_Flag,
                                     *Surf_Tria_Connectivity,
                                     *Vol_Hex_Connectivity,
                                     *Vol_ID_Flag, 
                                     *Vol_Pent_5_Connectivity,
                                     *Vol_Pent_6_Connectivity,
                                     *Vol_Tet_Connectivity,
                                     *Coordinates,
                                     *Initial_Normal_Spacing, *BL_Thickness);

      else if (strcmp (File_Type_Suffix_, ".mcell") == 0)
        Error_Flag = ug_io_read_nsu3d (Grid_File, 
                                       File_Format_, Read_Task_Flag,
                                       &Number_of_Bnd_Nodes, Number_of_Nodes, 
                                       Number_of_Surf_Quads, Number_of_Surf_Trias,
                                       Number_of_Vol_Hexs,
                                       Number_of_Vol_Pents_5, Number_of_Vol_Pents_6,
                                       Number_of_Vol_Tets,
                                       *Surf_ID_Flag,
                                       *Surf_Quad_Connectivity,
                                       *Surf_Tria_Connectivity,
                                       *Vol_Hex_Connectivity,
                                       *Vol_Pent_5_Connectivity,
                                       *Vol_Pent_6_Connectivity,
                                       *Vol_Tet_Connectivity,
                                       *Coordinates);

      else if (strcmp (File_Type_Suffix_, ".stl") == 0)
        Error_Flag = ug_io_read_stl (Grid_File, File_Format_,
                                     Message_Flag,
                                     Read_Task_Flag,
                                     Number_of_Nodes,
                                     Number_of_Surf_Trias,
                                     *Surf_ID_Flag,
                                     *Surf_Tria_Connectivity,
                                     *Coordinates);

      else if (strcmp (File_Type_Suffix_, ".surf") == 0)
        Error_Flag = ug3_read_surf (Grid_File, Read_Task_Flag,
                                    Number_of_Nodes,
                                    Number_of_Surf_Quads, Number_of_Surf_Trias,
                                    *Surf_Grid_BC_Flag, *Surf_ID_Flag,
                                    *Surf_Quad_Connectivity,
                                    *Surf_Reconnection_Flag,
                                    *Surf_Tria_Connectivity,
                                    *Coordinates,
                                    *Initial_Normal_Spacing, *BL_Thickness);

      else if (strcmp (File_Type_Suffix_, ".tri") == 0)
        Error_Flag = ug_io_read_cart3d (Grid_File, File_Format_,
                                        Read_Task_Flag,
                                        Number_of_Nodes,
                                        Number_of_Surf_Trias,
                                        *Surf_ID_Flag,
                                        *Surf_Tria_Connectivity,
                                        *Coordinates);

      else if (strcmp (File_Type_Suffix_, ".ugrid") == 0)
      {
        Error_Flag = ug3_read_ugrid (Grid_File, File_Format_, Read_Task_Flag,
                                     &Number_of_Bnd_Nodes,
                                     Number_of_Nodes,
                                     Number_of_Surf_Quads,
                                     Number_of_Surf_Trias,
                                     Number_of_Vol_Hexs,
                                     Number_of_Vol_Pents_5,
                                     Number_of_Vol_Pents_6,
                                     Number_of_Vol_Tets,
                                     *Surf_ID_Flag,
                                     *Surf_Quad_Connectivity,
                                     *Surf_Tria_Connectivity,
                                     *Vol_Hex_Connectivity,
                                     *Vol_ID_Flag,
                                     *Vol_Pent_5_Connectivity,
                                     *Vol_Pent_6_Connectivity,
                                     *Vol_Tet_Connectivity,
                                     *Coordinates);

        if (Error_Flag == 0 && Read_Task_Flag == 2 && Number_of_Vol_Elems == 0)
          ug3_read_gdata (Grid_File, File_Format_,
                          *Number_of_Nodes,
                          *Number_of_Surf_Quads, *Number_of_Surf_Trias,
                          *Surf_Grid_BC_Flag, *Surf_Reconnection_Flag,
                          *Initial_Normal_Spacing, *BL_Thickness);
      }

      else if (strcmp (File_Type_Suffix_, ".cogsg") == 0)
        Error_Flag = ug_io_read_vgrid (Grid_File, Case_Name, 
                                       Read_Task_Flag,
                                       &Number_of_Bnd_Nodes, Number_of_Nodes, 
                                       Number_of_Surf_Trias, Number_of_Vol_Tets,
                                       Number_of_BL_Vol_Tets,
                                       *Surf_ID_Flag, *Surf_Tria_Connectivity,
                                       *Vol_Tet_Connectivity,
                                       *Coordinates);

      else if (strcmp (File_Type_Suffix_, ".cgns") == 0)
        Error_Flag = ug_io_read_cgns_grid (File_Name, Read_Task_Flag,
                                           Number_of_BL_Vol_Tets,
                                           &Number_of_Bnd_Nodes, Number_of_Nodes, 
                                           Number_of_Surf_Quads,
                                           Number_of_Surf_Trias,
                                           Number_of_Vol_Hexs,
                                           Number_of_Vol_Pents_5,
                                           Number_of_Vol_Pents_6,
                                           Number_of_Vol_Tets,
                                           *Surf_Grid_BC_Flag, *Surf_ID_Flag, 
                                           *Surf_Quad_Connectivity,
                                           *Surf_Reconnection_Flag,
                                           *Surf_Tria_Connectivity,
                                           *Vol_Hex_Connectivity,
                                           *Vol_ID_Flag,
                                           *Vol_Pent_5_Connectivity,
                                           *Vol_Pent_6_Connectivity,
                                           *Vol_Tet_Connectivity,
                                           *Coordinates,
                                           *Initial_Normal_Spacing, *BL_Thickness);

      else
      {
        ug_error_message ("*** ERROR 609 : unknown input grid file type ***");
        Error_Flag = 609;
      }
    }

    // close file

    if (Close_Flag)
      Close_Flag = ug_fclose (Grid_File);

    // zero volume grid dimensions if option to read boundary only is set

    if (Bnd_Only_Flag)
    {
      *Number_of_BL_Vol_Tets = 0;
      *Number_of_Vol_Hexs = 0;
      *Number_of_Vol_Pents_5 = 0;
      *Number_of_Vol_Pents_6 = 0;
      *Number_of_Vol_Tets = 0;
    }

    // set total surface faces and volume elements

    Number_of_Surf_Faces = (*Number_of_Surf_Trias) + (*Number_of_Surf_Quads);

    Number_of_Vol_Elems = (*Number_of_Vol_Hexs) + (*Number_of_Vol_Pents_5)
                        + (*Number_of_Vol_Pents_6) + (*Number_of_Vol_Tets);

    // output grid information

    if (Error_Flag == 0 && Read_Task_Flag == 1 && Message_Flag >= 1)
    {
      strcpy (Name_Text, "");
      strncat (Name_Text, Grid_File_Name, ((INT_) strlen (Name_Text))-32);
      snprintf (Text, sizeof(Text), "UG_IO    : Grid File Name    = %s", Name_Text);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Quad Surface Faces=%10i", (int) *Number_of_Surf_Quads);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Tria Surface Faces=%10i", (int) *Number_of_Surf_Trias);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Nodes             =%10i", (int) *Number_of_Nodes);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Total Elements    =%10i", (int) Number_of_Vol_Elems);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Hex Elements      =%10i", (int) *Number_of_Vol_Hexs);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Pent_5 Elements   =%10i", (int) *Number_of_Vol_Pents_5);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Pent_6 Elements   =%10i", (int) *Number_of_Vol_Pents_6);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Tet Elements      =%10i", (int) *Number_of_Vol_Tets);
      ug_message (Text);
    }

    // allocate grid data

    if (Number_of_Vol_Elems == 0)
       M_Vol_ID_Flag = 0;

    if (Error_Flag == 0 && Read_Task_Flag == 1)
      Error_Flag = ug_io_malloc_grid (M_BL_Thickness,
                                      M_Initial_Normal_Spacing,
                                      M_Surf_Grid_BC_Flag,
                                      M_Surf_ID_Flag,
                                      M_Surf_Reconnection_Flag,
                                      M_Vol_ID_Flag,
                                      *Number_of_Nodes,
                                      *Number_of_Bnd_Edges,
                                      *Number_of_Surf_Quads,
                                      *Number_of_Surf_Trias,
                                      *Number_of_Vol_Hexs,
                                      *Number_of_Vol_Pents_5,
                                      *Number_of_Vol_Pents_6,
                                      *Number_of_Vol_Tets,
                                      Bnd_Edge_ID_Flag,
                                      Bnd_Edge_Connectivity,
                                      Surf_Grid_BC_Flag,
                                      Surf_ID_Flag,
                                      Surf_Reconnection_Flag,
                                      Surf_Quad_Connectivity, 
                                      Surf_Tria_Connectivity,
                                      Vol_Hex_Connectivity,
                                      Vol_ID_Flag,
                                      Vol_Pent_5_Connectivity,
                                      Vol_Pent_6_Connectivity,
                                      Vol_Tet_Connectivity,
                                      Coordinates,
                                      Initial_Normal_Spacing,
                                      BL_Thickness);

    // set initial values for boundary data and volume ID

    if (Error_Flag == 0 && Read_Task_Flag == 1)
    {
      if (M_BL_Thickness)
        ug_set_double (1, *Number_of_Nodes, dc0, *BL_Thickness);

      if (M_Initial_Normal_Spacing)
        ug_set_double (1, *Number_of_Nodes, dc0, *Initial_Normal_Spacing);

      if (M_Surf_Grid_BC_Flag)
        ug_set_int (1, Number_of_Surf_Faces, 1, *Surf_Grid_BC_Flag);

      if (M_Surf_Reconnection_Flag)
        ug_set_int (1, Number_of_Surf_Faces, 0, *Surf_Reconnection_Flag);

      if (M_Vol_ID_Flag)
        ug_set_int (1, Number_of_Vol_Elems, 0, *Vol_ID_Flag);
    }

    ++Read_Task_Flag;
  }
  while (Read_Task_Flag <= 2 && Error_Flag == 0);

  // set error flag

  if (Error_Flag > 0)
    Error_Flag = 609;

  // compress file

  ug_compress_file (1, File_Compression_Suffix_, File_Name);

  // remove interior nodes from grid data if option to read boundary only is set
  // and if more nodes were saved than just the boundary nodes

  if (Error_Flag == 0 && Bnd_Only_Flag && *Number_of_Nodes > Number_of_Bnd_Nodes)
  {
    Error_Flag = ug3_bnd_nodes (1, &Number_of_Bnd_Nodes, *Number_of_Nodes,
                                *Number_of_Surf_Trias, *Number_of_Surf_Quads,
                                *Surf_Tria_Connectivity,
                                *Surf_Quad_Connectivity,
                                &Map,
                                *BL_Thickness, *Initial_Normal_Spacing,
                                *Coordinates);

    if (Error_Flag == 0)
    {
      *Number_of_Nodes = Number_of_Bnd_Nodes;

      if (Message_Flag >= 1)
      {
        snprintf (Text, sizeof(Text), "UG_IO    : Nodes             =%10i", (int) *Number_of_Nodes);
        ug_message (Text);
      }
    }
  }

  // Convert surface grid quads to trias.

  if (Error_Flag == 0 && No_Quad_Flag && *Number_of_Surf_Quads && Number_of_Vol_Elems == 0)
  {
    Error_Flag = ug3_qtria (0, 0,
                            Number_of_Surf_Trias, Number_of_Surf_Quads,
                            Surf_Grid_BC_Flag, Surf_ID_Flag,
                            &Surf_Error_Flag,
                            Surf_Tria_Connectivity, *Surf_Quad_Connectivity,
                            &Quad_Tria_Flag, Surf_Reconnection_Flag,
                            *Coordinates);

    if (Error_Flag == 0 && Message_Flag >= 1)
    {
      snprintf (Text, sizeof(Text), "UG_IO    : Quad Surface Faces=%10i", (int) *Number_of_Surf_Quads);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Tria Surface Faces=%10i", (int) *Number_of_Surf_Trias);
      ug_message (Text);
    }
  }

  // renumber boundary surface IDs consecutively

  if (Error_Flag == 0 && Renumber_ID_Flag)
  {
    Number_of_IDs = 0;

    Error_Flag = ug_renumber_int_values (Number_of_Surf_Faces,
                                         &Number_of_IDs, &Max_ID, &Min_ID,
                                         *Surf_ID_Flag, *Surf_ID_Flag,
                                         &Map);

    ug_free (Map);
  }

  // reorder nodes so that boundary nodes are first followed by interior nodes

  // increment all connectivity node indices by one if the connectivity maps
  // use a node index value of 0

  if (Error_Flag == 0 && Number_of_Vol_Elems)
    Error_Flag = ug3_bnd_vol_nodes (&Number_of_Bnd_Nodes, *Number_of_Nodes,
                                    *Number_of_Surf_Trias,
                                    *Number_of_Vol_Tets,
                                    *Number_of_Vol_Pents_5,
                                    *Number_of_Vol_Pents_6,
                                    *Number_of_Vol_Hexs,
                                    *Number_of_Surf_Quads,
                                    *Surf_Tria_Connectivity,
                                    *Vol_Tet_Connectivity,
                                    *Vol_Pent_5_Connectivity,
                                    *Vol_Pent_6_Connectivity,
                                    *Vol_Hex_Connectivity,
                                    *Surf_Quad_Connectivity,
                                    *BL_Thickness, *Initial_Normal_Spacing,
                                    *Coordinates);

  // reorder faces so that they are grouped by separate surface ID and
  // duplicate nodes on their boundaries so that each surface ID patch has
  // unique node indexes

  if (Error_Flag == 0 && Separate_by_ID_Flag && Number_of_Vol_Elems == 0)
  {
    Number_of_Nodes_Dim = *Number_of_Nodes;

    Error_Flag = ug3_reorderid (*Number_of_Surf_Trias,
                                Number_of_Nodes, &Number_of_Nodes_Dim,
                                *Surf_Grid_BC_Flag, *Surf_ID_Flag,
                                *Surf_Tria_Connectivity,
                                *Surf_Reconnection_Flag,
                                BL_Thickness, Initial_Normal_Spacing,
                                Coordinates);
  }

  // output BL tet element and vol ID information

  if (Error_Flag == 0 && Message_Flag >= 1)
  {
    if (Number_of_Vol_Elems && M_Vol_ID_Flag)
    {
      Vol_ID_Max = ug_max_int (1, Number_of_Vol_Elems, *Vol_ID_Flag);
      Vol_ID_Min = ug_min_int (1, Number_of_Vol_Elems, *Vol_ID_Flag);

      snprintf (Text, sizeof(Text), "UG_IO    : Volume ID Min/Max =%10i%10i", (int) Vol_ID_Min,
                                                               (int) Vol_ID_Max);
      ug_message (Text);
    }

    ug_message (" ");
    ug_cpu_message ("UG_IO    :");
  }

  return (Error_Flag);

}
