#include "UG_IO_LIB.h"

INT_ ug_io_read_func_file
 (char Func_File_Name[],
  INT_ Message_Flag,
  INT_ NDim,
  INT_ *Number_of_Nodes,
  INT_ *Number_of_U_Scalars,
  INT_ *Number_of_U_Vectors,
  INT_ *Number_of_U_Matrixes,
  INT_ *Number_of_U_Metrics,
  CHAR_21 ** U_Scalar_Labels,
  CHAR_21 ** U_Vector_Labels,
  CHAR_21 ** U_Matrix_Labels,
  CHAR_21 ** U_Metric_Labels,
  DOUBLE_1D ** U_Scalars,
  DOUBLE_1D ** U_Vectors,
  DOUBLE_1D ** U_Matrixes,
  DOUBLE_1D ** U_Metrics)
{

/*
 * Read function data from a standard UG function file.
 * 
 * UG_IO LIB : Unstructured Grid - Input/Output Routine Library
 * $Id: ug_io_read_func_file.c,v 1.48 2022/11/21 00:41:36 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  FILE * Func_File;

  CHAR_UG_MAX Case_Name, File_Name, Name_Text, Text;
  CHAR_21 File_Compression_Suffix_, File_Format_Suffix_, File_Mode_Suffix_, File_Type_Suffix_;

  INT_ Close_Flag, Error_Flag, File_Data_Type_, File_Format_, File_Mode_,
       File_Name_Length, File_Type_Format_Flag_, Index, Read_Task_Flag;
  INT_ SFUNC_Flag = 1;

  // get file info

  Error_Flag = ug_io_file_info (Func_File_Name, Case_Name, 
                                File_Compression_Suffix_, File_Format_Suffix_,
                                File_Mode_Suffix_, File_Type_Suffix_,
                                0, NDim, &File_Data_Type_,
                                &File_Format_, &File_Mode_,
                                &File_Type_Format_Flag_);

  // exit for error

  if (Error_Flag) 
    return (Error_Flag);

  // exit if not a function file

  if ((NDim == 2 && File_Data_Type_ != UG_IO_2D_FUNCTION_DATA) ||
      (NDim == 3 && File_Data_Type_ != UG_IO_FUNCTION_DATA))
    return (-1);

  // use abs of file format

  File_Format_ = ug_abs (File_Format_);

  // output header

  if (Message_Flag >= 1)
  {
    ug_message (" ");
    ug_message ("UG_IO    : INPUT FUNCTION FIELD");
    ug_message (" ");
    ug_message ("UG_IO    : Reading data");
    ug_message (" ");

    ug_cpu_message ("");
  }

  // uncompress file

  File_Name_Length = (INT_) strlen (Func_File_Name)
                   - (INT_) strlen (File_Compression_Suffix_);

  strcpy (File_Name, "");
  strncat (File_Name, Func_File_Name, File_Name_Length);

  Error_Flag = ug_uncompress_file (Func_File_Name);

  if (Error_Flag != 0)
  {
    ug_error_message ("*** ERROR 608 : unable to uncompress function file ***");
    return (608);
  }

  // initialize number of function types

  *Number_of_U_Scalars = 0;
  *Number_of_U_Vectors = 0;
  *Number_of_U_Matrixes = 0;
  *Number_of_U_Metrics = 0;

  // make two passes

  // the first pass reads the dimensions
  // the second reads the labels and the third reads the data

  Read_Task_Flag = 1;

  do
  {
    // open file and set close flag

    if (File_Mode_ == UG_FIO_TMP_FILE_MODE)
    {
      Func_File = ug_fopen (File_Name, "r_tmp");

      Close_Flag = 1;
    }
    else
    {
      Func_File = ug_fopen (File_Name, "r");

      Close_Flag = 0;
    }

    if (Func_File == NULL)
    {
      ug_error_message ("*** ERROR 608 : unable to open function file ***");
      Error_Flag = 608;
    }

    // allocate function flag and label arrays

    if (Error_Flag == 0 && Read_Task_Flag == 2)
      Error_Flag = ug_io_malloc_func_flag (*Number_of_U_Scalars, 
                                           *Number_of_U_Vectors,
                                           *Number_of_U_Matrixes,
                                           *Number_of_U_Metrics,
                                           U_Scalar_Labels, U_Vector_Labels,
                                           U_Matrix_Labels, U_Metric_Labels);

    // initialize function labels

    if (Error_Flag == 0 && Read_Task_Flag == 2)
    {
      if (*U_Scalar_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Scalars; ++Index)
        {
          strcpy ((*U_Scalar_Labels)[Index], "                    ");
        }
      }

      if (*U_Vector_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Vectors; ++Index)
        {
          strcpy ((*U_Vector_Labels)[Index], "                    ");
        }
      }

      if (*U_Matrix_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Matrixes; ++Index)
        {
          strcpy ((*U_Matrix_Labels)[Index], "                    ");
        }
      }

      if (*U_Metric_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Metrics; ++Index)
        {
          strcpy ((*U_Metric_Labels)[Index], "                    ");
        }
      }
    }

    // malloc function data arrays

    if (Error_Flag == 0 && Read_Task_Flag == 3)
      Error_Flag = ug_io_malloc_func (NDim,
                                      *Number_of_Nodes,
                                      *Number_of_U_Scalars, 
                                      *Number_of_U_Vectors,
                                      *Number_of_U_Matrixes,
                                      *Number_of_U_Metrics,
                                      U_Scalars,
                                      (DOUBLE_1D **) U_Vectors,
                                      (DOUBLE_1D **) U_Matrixes,
                                      (DOUBLE_1D **) U_Metrics);

    // read function file

    if (Error_Flag == 0)
    {
       if (strcmp (File_Type_Suffix_, ".ufunc") == 0)
         SFUNC_Flag = 0;

      if (strcmp (File_Type_Suffix_, ".ufunc") == 0 ||
          strcmp (File_Type_Suffix_, ".sfunc") == 0)
        Error_Flag = ug3_read_sfunc (Func_File, File_Format_,
                                     Read_Task_Flag, NDim, Number_of_Nodes,
                                     Number_of_U_Scalars, Number_of_U_Vectors,
                                     Number_of_U_Matrixes, Number_of_U_Metrics,
                                     SFUNC_Flag,
                                     *U_Scalar_Labels, *U_Vector_Labels,
                                     *U_Matrix_Labels, *U_Metric_Labels,
                                     *U_Scalars, *U_Vectors,
                                     *U_Matrixes, *U_Metrics);

      else if (strcmp (File_Type_Suffix_, ".sol") == 0 ||
               strcmp (File_Type_Suffix_, ".solb") == 0)
      {
        Close_Flag = ug_fclose (Func_File);

        Error_Flag = ug3_read_sol (Func_File_Name, 
                                   Read_Task_Flag, NDim, Number_of_Nodes,
                                   Number_of_U_Scalars, Number_of_U_Vectors,
                                   Number_of_U_Matrixes, Number_of_U_Metrics,
                                   *U_Scalar_Labels, *U_Vector_Labels,
                                   *U_Matrix_Labels, *U_Metric_Labels,
                                   *U_Scalars, *U_Vectors,
                                   *U_Matrixes, *U_Metrics);

        Close_Flag = 0;
      }

      else
      { 
        ug_error_message ("*** ERROR 608 : unknown input function file type ***");
        Error_Flag = 608;
      }
    }

    // close file

    if (Close_Flag == 1)
      Close_Flag = ug_fclose (Func_File);

    // output function data information

    if (Error_Flag == 0 && Read_Task_Flag == 1 && Message_Flag >= 1)
    {
      strcpy (Name_Text, "");
      strncat (Name_Text, Func_File_Name, ((INT_) strlen (Name_Text))-32);
      snprintf (Text, sizeof(Text), "UG_IO    : Function File Name= %s", Name_Text);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG_IO    : Nodes             =%10i", (int) *Number_of_Nodes);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG_IO    : Scalar Variables  =%10i", (int) *Number_of_U_Scalars);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG_IO    : Vector Variables  =%10i", (int) *Number_of_U_Vectors);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG_IO    : Matrix Variables  =%10i", (int) *Number_of_U_Matrixes);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG_IO    : Metric Variables  =%10i", (int) *Number_of_U_Metrics);
      ug_message (Text);
    }

    ++Read_Task_Flag;
  }
  while (Read_Task_Flag <= 3 && Error_Flag == 0);

  // set error flag

  if (Error_Flag > 0)
    Error_Flag = 609;

  // compress file

  ug_compress_file (1, File_Compression_Suffix_, File_Name);

  // output function labels

  if (Error_Flag == 0 && Message_Flag >= 1)
  {
    if (*U_Scalar_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Scalars; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG_IO    : Scalar Label %-4i = %s",
                 (int) Index, (*U_Scalar_Labels)[Index]);

        ug_message (Text);
      }
    }

    if (*U_Vector_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Vectors; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG_IO    : Vector Label %-4i = %s",
                 (int) Index, (*U_Vector_Labels)[Index]);

        ug_message (Text);
      }
    }

    if (*U_Matrix_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Matrixes; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG_IO    : Matrix Label %-4i = %s",
                 (int) Index, (*U_Matrix_Labels)[Index]);

        ug_message (Text);
      }
    }

    if (*U_Metric_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Metrics; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG_IO    : Metric Label %-4i = %s",
                 (int) Index, (*U_Metric_Labels)[Index]);

        ug_message (Text);
      }
    }

    ug_message (" ");
    ug_cpu_message ("UG_IO    :");
  }

  return (Error_Flag);

}
