#include "UG_IO_LIB.h"

INT_ ug_io_read_2d_grid_file
 (char Grid_File_Name[],
  INT_ Message_Flag,
  INT_ M_BL_Thickness,
  INT_ M_Initial_Normal_Spacing,
  INT_ M_Bnd_Edge_Grid_BC_Flag,
  INT_ M_Bnd_Edge_ID_Flag,
  INT_ Bnd_Only_Flag,
  INT_ No_Quad_Flag,
  INT_ *Number_of_Bnd_Edges,
  INT_ *Number_of_Nodes,
  INT_ *Number_of_Quads,
  INT_ *Number_of_Trias,
  INT_2D ** Bnd_Edge_Connectivity,
  INT_1D ** Bnd_Edge_Grid_BC_Flag,
  INT_1D ** Bnd_Edge_ID_Flag,
  INT_4D ** Quad_Connectivity,
  INT_3D ** Tria_Connectivity,
  DOUBLE_2D ** Coordinates,
  DOUBLE_1D ** Initial_Normal_Spacing,
  DOUBLE_1D ** BL_Thickness)

{

/*
 * Read 2D grid data from a standard UG 2D grid file.
 * 
 * UG_IO LIB : Unstructured Grid - Input/Output Routine Library
 * $Id: ug_io_read_2d_grid_file.c,v 1.62 2022/11/21 00:41:36 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  FILE *Grid_File;

  CHAR_UG_MAX Case_Name, File_Name, Name_Text, Text;
  CHAR_21 File_Compression_Suffix_, File_Format_Suffix_, File_Mode_Suffix_,
          File_Type_Suffix_;

  INT_1D *Face_ID_Flag = NULL;
  INT_1D *Map = NULL;

  INT_ Close_Flag, Error_Flag, File_Data_Type_, File_Format_, File_Mode_,
       File_Name_Length, File_Type_Format_Flag_, Number_of_Bnd_Nodes,
       Read_Task_Flag;

  double dc0;

  dc0 = 0.0;

  // check input grid file name and set case name

  Error_Flag = ug_io_file_info (Grid_File_Name, Case_Name, 
                                File_Compression_Suffix_, File_Format_Suffix_,
                                File_Mode_Suffix_, File_Type_Suffix_,
                                0, 2, &File_Data_Type_,
                                &File_Format_, &File_Mode_,
                                &File_Type_Format_Flag_);

  // exit for error

  if (Error_Flag) 
    return (Error_Flag);

  // exit if not a 2D boundary edge or 2D grid file

  if (File_Data_Type_ != UG_IO_2D_EDGE_GRID && File_Data_Type_ != UG_IO_2D_GRID)
    return (-1);

  // use abs of file format

  File_Format_ = ug_abs (File_Format_);

  // output header

  if (Message_Flag >= 1)
  {
    ug_message (" ");
    ug_message ("UG_IO    : INPUT GRID");
    ug_message (" ");
    ug_message ("UG_IO    : Reading Data");
    ug_message (" ");

    ug_cpu_message ("");
  }

  // set file name without compression suffix

  File_Name_Length = (INT_) strlen (Grid_File_Name)
                   - (INT_) strlen (File_Compression_Suffix_);

  strcpy (File_Name, "");
  strncat (File_Name, Grid_File_Name, File_Name_Length);

  // uncompress file

  Error_Flag = ug_uncompress_file (Grid_File_Name);

  if (Error_Flag)
  {
    ug_error_message ("*** ERROR 606 : unable to uncompress grid file ***");
    return (606);
  }

  // zero boundary only option for boundary edge grids

  if (File_Data_Type_ == UG_IO_2D_EDGE_GRID)
    Bnd_Only_Flag = 0;

  // initialize grid dimensions

  *Number_of_Nodes = 0;
  *Number_of_Quads = 0;
  *Number_of_Trias = 0;

  if (Bnd_Only_Flag)
    Number_of_Bnd_Nodes = -1;
  else
    Number_of_Bnd_Nodes = 0;

  // make two passes

  // the first pass reads the dimensions and the second reads the data

  Read_Task_Flag = 1;

  do
  {
    // open file and set close flag

    Grid_File = ug_fopen (File_Name, "r");

    Close_Flag = 1;

    if (Grid_File == NULL)
    {
      ug_error_message ("*** ERROR 606 : unable to open grid file ***");
      Error_Flag = 606;
    }

    // read grid file

    if (Error_Flag == 0)
    {
      if (strcmp (File_Type_Suffix_, ".bedge") == 0)
        Error_Flag = ug_io_read_2d_bedge (Grid_File, 
                                          Read_Task_Flag,
                                          Number_of_Bnd_Edges, Number_of_Nodes,
                                          *Bnd_Edge_Connectivity,
                                          *Bnd_Edge_Grid_BC_Flag,
                                          *Bnd_Edge_ID_Flag,
                                          *Coordinates,
                                          *Initial_Normal_Spacing,
                                          *BL_Thickness);

      else if (strcmp (File_Type_Suffix_, ".fgrid") == 0)
      {
        Error_Flag = ug_io_read_2d_fgrid (Grid_File, 
                                          File_Format_, Read_Task_Flag,
                                          Number_of_Bnd_Nodes, Number_of_Nodes,
                                          Number_of_Trias,
                                          *Tria_Connectivity,
                                          *Coordinates);

        if (Error_Flag == 0)
          Error_Flag = ug2_read_gdata (Grid_File, 
                                       File_Format_, Read_Task_Flag,
                                       Number_of_Bnd_Edges,
                                       &Number_of_Bnd_Nodes, Number_of_Nodes,
                                       Number_of_Quads, Number_of_Trias,
                                       *Bnd_Edge_Connectivity,
                                       *Bnd_Edge_Grid_BC_Flag,
                                       *Bnd_Edge_ID_Flag,
                                       *Initial_Normal_Spacing,
                                       *BL_Thickness);
      }

      else if (strcmp (File_Type_Suffix_, ".mesh") == 0 ||
               strcmp (File_Type_Suffix_, ".meshb") == 0)
      {
        Close_Flag = ug_fclose (Grid_File);

        Error_Flag = ug2_read_mesh (File_Name, 
                                    Read_Task_Flag,
                                    &Number_of_Bnd_Nodes,
                                    Number_of_Nodes,
                                    Number_of_Bnd_Edges,
                                    Number_of_Quads,
                                    Number_of_Trias,
                                    *Bnd_Edge_ID_Flag,
                                    *Bnd_Edge_Connectivity,
                                    Face_ID_Flag,
                                    *Quad_Connectivity,
                                    *Tria_Connectivity,
                                    *Coordinates);

        Close_Flag = 0;
      }

      else if (strcmp (File_Type_Suffix_, ".ugrid") == 0)
      {
        Error_Flag = ug2_read_ugrid (Grid_File, 
                                     File_Format_, Read_Task_Flag,
                                     Number_of_Bnd_Nodes, Number_of_Nodes, 
                                     Number_of_Quads, Number_of_Trias,
                                     *Quad_Connectivity, *Tria_Connectivity,
                                     *Coordinates);

        if (Error_Flag == 0)
          Error_Flag = ug2_read_gdata (Grid_File, 
                                       File_Format_, Read_Task_Flag,
                                       Number_of_Bnd_Edges,
                                       &Number_of_Bnd_Nodes, Number_of_Nodes,
                                       Number_of_Quads, Number_of_Trias,
                                       *Bnd_Edge_Connectivity,
                                       *Bnd_Edge_Grid_BC_Flag,
                                       *Bnd_Edge_ID_Flag,
                                       *Initial_Normal_Spacing,
                                       *BL_Thickness);
      }

      else
      {
        Read_Task_Flag = 0;
        ug_error_message ("*** ERROR 606 : unknown input grid file type ***");
        return (606);
      }
    }

    // close file

    if (Close_Flag)
      Close_Flag = ug_fclose (Grid_File);

    // zero volume grid dimensions if option to read boundary only is set

    if (Bnd_Only_Flag)
    {
      *Number_of_Quads = 0;
      *Number_of_Trias = 0;
    }

    // output grid information

    if (Error_Flag == 0 && Read_Task_Flag == 1 && Message_Flag >= 1)
    {
      strcpy (Name_Text, "");
      strncat (Name_Text, Grid_File_Name, ((INT_) strlen (Name_Text))-32);
      snprintf (Text, sizeof(Text), "UG_IO    : Grid File Name    = %s", Name_Text);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Boundary Edges    =%10i", (int) *Number_of_Bnd_Edges);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Nodes             =%10i", (int) *Number_of_Nodes);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Quad Elements     =%10i", (int) *Number_of_Quads);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Tria Elements     =%10i", (int) *Number_of_Trias);
      ug_message (Text);
    }

    // allocate grid data

    if (Error_Flag == 0 && Read_Task_Flag == 1)
      Error_Flag = ug_io_malloc_2d_grid (M_BL_Thickness,
                                         M_Initial_Normal_Spacing,
                                         M_Bnd_Edge_Grid_BC_Flag,
                                         M_Bnd_Edge_ID_Flag,
                                         *Number_of_Bnd_Edges,
                                         *Number_of_Nodes,
                                         *Number_of_Quads,
                                         *Number_of_Trias,
                                         Bnd_Edge_Connectivity,
                                         Bnd_Edge_Grid_BC_Flag,
                                         Bnd_Edge_ID_Flag,
                                         Quad_Connectivity,
                                         Tria_Connectivity,
                                         Coordinates,
                                         Initial_Normal_Spacing,
                                         BL_Thickness);

    // set initial values for boundary data

    if (Error_Flag == 0 && Read_Task_Flag == 1)
    {
      if (M_BL_Thickness)
        ug_set_double (1, *Number_of_Nodes, dc0, *BL_Thickness);

      if (M_Initial_Normal_Spacing)
        ug_set_double (1, *Number_of_Nodes, dc0, *Initial_Normal_Spacing);

      if (M_Bnd_Edge_Grid_BC_Flag)
        ug_set_int (1, *Number_of_Bnd_Edges, -1, *Bnd_Edge_Grid_BC_Flag);
    }

    ++Read_Task_Flag;
  }
  while (Read_Task_Flag <= 2 && Error_Flag == 0);

  // set error flag

  if (Error_Flag > 0)
    Error_Flag = 606;

  // compress file

  ug_compress_file (1, File_Compression_Suffix_, File_Name);

  // remove interior nodes from grid data if option to read boundary only is set

  if (Error_Flag == 0 && Bnd_Only_Flag)
  {
    Error_Flag = ug2_bnd_nodes (1, &Number_of_Bnd_Nodes, *Number_of_Nodes,
                                *Number_of_Bnd_Edges,
                                *Bnd_Edge_Connectivity, &Map,
                                *BL_Thickness, *Initial_Normal_Spacing,
                                *Coordinates);

    ug_free (Map);

    if (Error_Flag == 0)
    {
      *Number_of_Nodes = Number_of_Bnd_Nodes;

      if (Message_Flag >= 1)
      {
        snprintf (Text, sizeof(Text), "UG_IO    : Nodes             =%10i", (int) *Number_of_Nodes);
        ug_message (Text);
      }
    }
  }

  // Convert surface grid quads to trias.

  if (Error_Flag == 0 && No_Quad_Flag && *Number_of_Quads)
  {
    Error_Flag = ug2_qtria (Number_of_Trias, Number_of_Quads,
                            Tria_Connectivity, *Quad_Connectivity,
                            *Coordinates);

    if (Error_Flag == 0 && Message_Flag >= 1)
    {
      snprintf (Text, sizeof(Text), "UG_IO    : Quad Elements     =%10i", (int) *Number_of_Quads);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG_IO    : Tria Elements     =%10i", (int) *Number_of_Trias);
      ug_message (Text);
    }
  }

  // output tailer

  if (Error_Flag == 0 && Message_Flag >= 1)
  {
    ug_message (" ");
    ug_cpu_message ("UG_IO    :");
  }

  return (Error_Flag);

}
