#include "UG_IO_LIB.h"

INT_ ug_io_p2_write_grid_file
 (char Grid_File_Name[],
  INT_ Message_Flag,
  INT_ Number_of_BL_Vol_Tets,
  INT_ Number_of_Nodes,
  INT_ Number_of_Surf_Quads,
  INT_ Number_of_Surf_Trias,
  INT_ Number_of_Vol_Hexs,
  INT_ Number_of_Vol_Pents_5,
  INT_ Number_of_Vol_Pents_6,
  INT_ Number_of_Vol_Tets,
  INT_ P_Order,
  INT_1D * Surf_Grid_BC_Flag,
  INT_1D * Surf_ID_Flag,
  INT_1D * Surf_Reconnection_Flag,
  INT_4D * Surf_Quad_Connectivity,
  INT_3D * Surf_Tria_Connectivity,
  INT_8D * Vol_Hex_Connectivity,
  INT_1D * Vol_ID_Flag,
  INT_5D * Vol_Pent_5_Connectivity,
  INT_6D * Vol_Pent_6_Connectivity,
  INT_4D * Vol_Tet_Connectivity,
  INT_5D * P2_Surf_Quad_Connectivity,
  INT_3D * P2_Surf_Tria_Connectivity,
  INT_19D * P2_Vol_Hex_Connectivity,
  INT_9D * P2_Vol_Pent_5_Connectivity,
  INT_12D * P2_Vol_Pent_6_Connectivity,
  INT_6D * P2_Vol_Tet_Connectivity,
  DOUBLE_3D * Coordinates,
  DOUBLE_1D * Initial_Normal_Spacing,
  DOUBLE_1D * BL_Thickness)

{

/*
 * Write grid data to a standard UG surface or volume grid file.
 * 
 * UG_IO LIB : Unstructured Grid - Input/Output Routine Library
 * $Id: ug_io_p2_write_grid_file.c,v 1.8 2022/11/21 00:41:36 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  FILE *Grid_File;

  CHAR_UG_MAX Case_Name, File_Name, Name_Text, Text;
  CHAR_21 File_Compression_Suffix_, File_Format_Suffix_, File_Mode_Suffix_,
          File_Type_Suffix_;

  INT_1D *Map = NULL;
  INT_1D *New_Surf_ID_Flag = NULL;

  INT_ Close_Flag, Error_Flag, File_Data_Type_, File_Format_, File_Mode_,
       File_Name_Length, File_Type_Format_Flag_, Max_ID, Min_ID,
       Number_of_IDs, Number_of_Int_Quads, Number_of_Int_Trias,
       Number_of_Surf_Faces, Number_of_Vol_Elems, Vol_ID_Max, Vol_ID_Min;
  INT_ Number_of_Bnd_Nodes = 0;

  // get file info

  Error_Flag = ug_io_file_info (Grid_File_Name, Case_Name, 
                                File_Compression_Suffix_, File_Format_Suffix_,
                                File_Mode_Suffix_, File_Type_Suffix_,
                                0, 3, &File_Data_Type_,
                                &File_Format_, &File_Mode_,
                                &File_Type_Format_Flag_);

  // exit for error

  if (Error_Flag) 
    return (Error_Flag);

  // exit if not a surface or volume grid file
  
  if (File_Data_Type_ != UG_IO_SURFACE_GRID && File_Data_Type_ != UG_IO_VOLUME_GRID)
    return (-1);

  // use abs of file format

  File_Format_ = ug_abs (File_Format_);

  // exit if file format is not appropriate for file type

  if (strcmp (File_Type_Suffix_, ".stl") == 0 &&
      File_Format_ != UG_FIO_FORMATTED && File_Format_ != UG_FIO_BINARY_SINGLE)
  {
    ug_error_message ("*** ERROR 615 : STL surface grid files must use ASCII formatted or binary float file format ***");
    return (615);
  }

  // exit if file format is not appropriate for p2-faces or elements

  if (P_Order == 2 && (strcmp (File_Type_Suffix_, ".mesh") == 0 ||
                       strcmp (File_Type_Suffix_, ".meshb") == 0))
  {
    ug_error_message ("*** ERROR 615 : P2-faces and P2-elements are only supported by MESH and MESHB file types ***");
    return (615);
  }

  // output header

  if (Message_Flag >= 1)
  {
    ug_message (" ");
    ug_message ("UG_IO    : OUTPUT GRID");
    ug_message (" ");
    ug_message ("UG_IO    : Writing Data");
    ug_message (" ");

    ug_cpu_message ("");
  }

  // backup file

  if (File_Mode_ == UG_FIO_STANDARD_FILE_MODE)
     ug_backup_file (Grid_File_Name);

  // set file name without compression suffix

  File_Name_Length = (INT_) strlen (Grid_File_Name)
                   - (INT_) strlen (File_Compression_Suffix_);

  strcpy (File_Name, "");
  strncat (File_Name, Grid_File_Name, File_Name_Length);

  // open file and set close flag

  if (File_Mode_ == UG_FIO_TMP_FILE_MODE)
    Grid_File = ug_fopen (File_Name, "tmp");
  else
    Grid_File = ug_fopen (File_Name, "w");

  if (Grid_File == NULL)
  {
    ug_error_message ("*** ERROR 615 : unable to open output grid file ***");
    return (615);
  }

  Close_Flag = 1;

  // set number of volume elements

  Number_of_Vol_Elems = Number_of_Vol_Hexs + Number_of_Vol_Pents_5
                      + Number_of_Vol_Pents_6 + Number_of_Vol_Tets;

  // write grid file

  if (strcmp (File_Type_Suffix_, ".cfd++") == 0)
  {
    Close_Flag = ug_fclose (Grid_File);

    Error_Flag = ug_io_write_cfdpp (Number_of_Nodes,
                                    Number_of_Surf_Quads,
                                    Number_of_Surf_Trias,
                                    Number_of_Vol_Hexs,
                                    Number_of_Vol_Pents_5,
                                    Number_of_Vol_Pents_6,
                                    Number_of_Vol_Tets,
                                    Surf_ID_Flag, 
                                    Surf_Quad_Connectivity,
                                    Surf_Tria_Connectivity,
                                    Vol_Hex_Connectivity,
                                    Vol_Pent_5_Connectivity,
                                    Vol_Pent_6_Connectivity,
                                    Vol_Tet_Connectivity,
                                    Coordinates);

    Close_Flag = 0;
  }

  else if (strcmp (File_Type_Suffix_, ".cobalt.grd") == 0)
    Error_Flag = ug_io_write_cobalt (Grid_File, 
                                     Number_of_Nodes,
                                     Number_of_Surf_Quads,
                                     Number_of_Surf_Trias,
                                     Number_of_Vol_Hexs,
                                     Number_of_Vol_Pents_5,
                                     Number_of_Vol_Pents_6,
                                     Number_of_Vol_Tets,
                                     Surf_ID_Flag, 
                                     Surf_Quad_Connectivity,
                                     Surf_Tria_Connectivity,
                                     Vol_Hex_Connectivity,
                                     Vol_Pent_5_Connectivity,
                                     Vol_Pent_6_Connectivity,
                                     Vol_Tet_Connectivity,
                                     Coordinates);

  else if (strcmp (File_Type_Suffix_, ".egrid") == 0)
    Error_Flag = ug_io_write_egrid (Grid_File, Case_Name, 
                                    Number_of_Nodes,
                                    Number_of_Surf_Quads,
                                    Number_of_Surf_Trias,
                                    Number_of_Vol_Hexs,
                                    Number_of_Vol_Pents_5,
                                    Number_of_Vol_Pents_6,
                                    Number_of_Vol_Tets,
                                    Surf_ID_Flag, 
                                    Surf_Quad_Connectivity,
                                    Surf_Tria_Connectivity,
                                    Vol_Hex_Connectivity,
                                    Vol_Pent_5_Connectivity,
                                    Vol_Pent_6_Connectivity,
                                    Vol_Tet_Connectivity,
                                    Coordinates);

  else if (strcmp (File_Type_Suffix_, ".fluent.cas") == 0)
    Error_Flag = ug_io_write_fluent (Grid_File, Case_Name, 
                                     Number_of_Nodes,
                                     Number_of_Surf_Quads,
                                     Number_of_Surf_Trias,
                                     Number_of_Vol_Hexs,
                                     Number_of_Vol_Pents_5,
                                     Number_of_Vol_Pents_6,
                                     Number_of_Vol_Tets,
                                     Surf_Grid_BC_Flag, Surf_ID_Flag, 
                                     Surf_Quad_Connectivity,
                                     Surf_Tria_Connectivity,
                                     Vol_Hex_Connectivity,
                                     Vol_Pent_5_Connectivity,
                                     Vol_Pent_6_Connectivity,
                                     Vol_Tet_Connectivity,
                                     Coordinates);

  else if (strcmp (File_Type_Suffix_, ".fgrid") == 0)
  {
    Number_of_Surf_Quads = 0;
    Number_of_Vol_Hexs = 0;
    Number_of_Vol_Pents_5 = 0;
    Number_of_Vol_Pents_6 = 0;

    Error_Flag = ug_io_write_fgrid (Grid_File, File_Format_,
                                    Number_of_Nodes, Number_of_Surf_Trias,
                                    Number_of_Vol_Tets,
                                    Surf_ID_Flag, Surf_Tria_Connectivity,
                                    Vol_Tet_Connectivity, Coordinates);

    if (Error_Flag == 0 && Number_of_Vol_Tets == 0)
      Error_Flag = ug3_write_gdata (Grid_File, File_Format_,
                                    Number_of_Nodes,
                                    Number_of_Surf_Quads,
                                    Number_of_Surf_Trias,
                                    Surf_Grid_BC_Flag, Surf_Reconnection_Flag,
                                    Initial_Normal_Spacing, BL_Thickness);

    if (Error_Flag)
      ug_error_message ("*** ERROR : error writing grid file ***");
  }

  else if (strcmp (File_Type_Suffix_, ".fvuns") == 0)
    Error_Flag = ug_io_write_fvuns (Grid_File, File_Format_,
                                    Number_of_Nodes,
                                    Number_of_Surf_Quads, Number_of_Surf_Trias,
                                    Number_of_Vol_Hexs,
                                    Number_of_Vol_Pents_5,
                                    Number_of_Vol_Pents_6,
                                    Number_of_Vol_Tets,
                                    Surf_ID_Flag,
                                    Surf_Quad_Connectivity, 
                                    Surf_Tria_Connectivity,
                                    Vol_Hex_Connectivity,
                                    Vol_Pent_5_Connectivity,
                                    Vol_Pent_6_Connectivity,
                                    Vol_Tet_Connectivity,
                                    Coordinates);

  else if (strcmp (File_Type_Suffix_, ".gambit") == 0)
    Error_Flag = ug_io_write_gambit (Grid_File, 
                                     Number_of_Nodes,
                                     Number_of_Surf_Quads, Number_of_Surf_Trias,
                                     Number_of_Vol_Hexs,
                                     Number_of_Vol_Pents_5,
                                     Number_of_Vol_Pents_6,
                                     Number_of_Vol_Tets,
                                     Surf_ID_Flag, 
                                     Surf_Quad_Connectivity,
                                     Surf_Tria_Connectivity,
                                     Vol_Hex_Connectivity,
                                     Vol_Pent_5_Connectivity,
                                     Vol_Pent_6_Connectivity,
                                     Vol_Tet_Connectivity,
                                     Coordinates);

  else if (strcmp (File_Type_Suffix_, ".mesh") == 0 ||
           strcmp (File_Type_Suffix_, ".meshb") == 0)
  {
    Close_Flag = ug_fclose (Grid_File);

    Error_Flag = ug3_p2_write_mesh (File_Name, 1,
                                    Number_of_Nodes,
                                    Number_of_Surf_Quads,
                                    Number_of_Surf_Trias,
                                    Number_of_Vol_Hexs,
                                    Number_of_Vol_Pents_5,
                                    Number_of_Vol_Pents_6,
                                    Number_of_Vol_Tets,
                                    P_Order,
                                    Surf_ID_Flag,
                                    Surf_Quad_Connectivity,
                                    Surf_Tria_Connectivity,
                                    Vol_Hex_Connectivity,
                                    Vol_ID_Flag,
                                    Vol_Pent_5_Connectivity,
                                    Vol_Pent_6_Connectivity,
                                    Vol_Tet_Connectivity,
                                    P2_Surf_Quad_Connectivity,
                                    P2_Surf_Tria_Connectivity,
                                    P2_Vol_Hex_Connectivity,
                                    P2_Vol_Pent_5_Connectivity,
                                    P2_Vol_Pent_6_Connectivity,
                                    P2_Vol_Tet_Connectivity,
                                    Coordinates);

    Close_Flag = 0;

    if (Error_Flag)
      return (Error_Flag);
  }

  else if (strcmp (File_Type_Suffix_, ".bdf") == 0 || strcmp (File_Type_Suffix_, ".nas") == 0)
    Error_Flag = ug_io_write_nas (Grid_File, 
                                  Number_of_BL_Vol_Tets,
                                  Number_of_Nodes,
                                  Number_of_Surf_Quads, Number_of_Surf_Trias,
                                  Number_of_Vol_Hexs,
                                  Number_of_Vol_Pents_5,
                                  Number_of_Vol_Pents_6,
                                  Number_of_Vol_Tets,
                                  Surf_Grid_BC_Flag, Surf_ID_Flag, 
                                  Surf_Quad_Connectivity,
                                  Surf_Reconnection_Flag,
                                  Surf_Tria_Connectivity,
                                  Vol_Hex_Connectivity,
                                  Vol_ID_Flag, 
                                  Vol_Pent_5_Connectivity,
                                  Vol_Pent_6_Connectivity,
                                  Vol_Tet_Connectivity,
                                  Coordinates,
                                  Initial_Normal_Spacing,
                                  BL_Thickness);

  else if (strcmp (File_Type_Suffix_, ".mcell") == 0)
  {
    // renumber boundary surface IDs consecutively for NSU3D

    Number_of_Surf_Faces = Number_of_Surf_Trias + Number_of_Surf_Quads;

    Error_Flag = 0;

    New_Surf_ID_Flag = (INT_1D *) ug_malloc (&Error_Flag,
                                             (Number_of_Surf_Faces+1)
                                           * (INT_) sizeof (INT_1D));

    if (Error_Flag > 0)
    {
      ug_error_message ("*** ERROR 100602 : unable to malloc I/O work array ***");
      return (100602);
    }

    Number_of_IDs = 0;

    Error_Flag = ug_renumber_int_values (Number_of_Surf_Faces,
                                         &Number_of_IDs, &Max_ID, &Min_ID,
                                         Surf_ID_Flag, New_Surf_ID_Flag,
                                         &Map);

    ug_free (Map);

    if (Error_Flag == 0)
      Error_Flag = ug_io_write_nsu3d (Grid_File, File_Format_,
                                      Number_of_Nodes,
                                      Number_of_Surf_Quads, Number_of_Surf_Trias,
                                      Number_of_Vol_Hexs,
                                      Number_of_Vol_Pents_5,
                                      Number_of_Vol_Pents_6,
                                      Number_of_Vol_Tets,
                                      New_Surf_ID_Flag, 
                                      Surf_Quad_Connectivity,
                                      Surf_Tria_Connectivity,
                                      Vol_Hex_Connectivity,
                                      Vol_Pent_5_Connectivity,
                                      Vol_Pent_6_Connectivity,
                                      Vol_Tet_Connectivity,
                                      Coordinates);

    ug_free (New_Surf_ID_Flag);
  }

  else if(strcmp (File_Type_Suffix_, ".foam") == 0)
  {
    Close_Flag = ug_fclose (Grid_File);

    Error_Flag = ug_io_write_OpenFOAM
      (Number_of_Nodes,
       Number_of_Surf_Quads,
       Number_of_Surf_Trias,
       Number_of_Vol_Hexs,
       Number_of_Vol_Pents_5,
       Number_of_Vol_Pents_6,
       Number_of_Vol_Tets,
       Surf_ID_Flag,
       Surf_Quad_Connectivity,
       Surf_Tria_Connectivity,
       Vol_Hex_Connectivity,
       Vol_Pent_5_Connectivity,
       Vol_Pent_6_Connectivity,
       Vol_Tet_Connectivity,
       Coordinates);

    Close_Flag = 0;
  }

  else if (strcmp (File_Type_Suffix_, ".poly") == 0)
  {
    Number_of_BL_Vol_Tets = 0;
    Number_of_Vol_Hexs = 0;
    Number_of_Vol_Pents_5 = 0;
    Number_of_Vol_Pents_6 = 0;
    Number_of_Vol_Tets = 0;

    Error_Flag = ug_io_write_poly (Grid_File, 
                                   &Number_of_Bnd_Nodes, Number_of_Nodes,
                                   Number_of_Surf_Quads, Number_of_Surf_Trias,
                                   Surf_ID_Flag, 
                                   Surf_Quad_Connectivity,
                                   Surf_Tria_Connectivity,
                                   Coordinates);
  }

  else if (strcmp (File_Type_Suffix_, ".stl") == 0)
  {
    Number_of_BL_Vol_Tets = 0;
    Number_of_Vol_Hexs = 0;
    Number_of_Vol_Pents_5 = 0;
    Number_of_Vol_Pents_6 = 0;
    Number_of_Vol_Tets = 0;

    Error_Flag = ug_io_write_stl (Grid_File, File_Format_,
                                  &Number_of_Bnd_Nodes, Number_of_Nodes,
                                  Number_of_Surf_Quads, Number_of_Surf_Trias,
                                  Surf_ID_Flag,
                                  Surf_Quad_Connectivity, 
                                  Surf_Tria_Connectivity,
                                  Coordinates);
  } 

  else if (strcmp (File_Type_Suffix_, ".surf") == 0)
  {
    Number_of_BL_Vol_Tets = 0;
    Number_of_Vol_Hexs = 0;
    Number_of_Vol_Pents_5 = 0;
    Number_of_Vol_Pents_6 = 0;
    Number_of_Vol_Tets = 0;

    Error_Flag = ug3_write_surf (Grid_File, 
                                 &Number_of_Bnd_Nodes, Number_of_Nodes,
                                 Number_of_Surf_Quads, Number_of_Surf_Trias,
                                 Surf_Grid_BC_Flag, Surf_ID_Flag,
                                 Surf_Quad_Connectivity, 
                                 Surf_Reconnection_Flag,
                                 Surf_Tria_Connectivity,
                                 Coordinates,
                                 Initial_Normal_Spacing, BL_Thickness);
  } 

  else if (strcmp (File_Type_Suffix_, ".tri") == 0)
  {
    Number_of_BL_Vol_Tets = 0;
    Number_of_Surf_Quads = 0;
    Number_of_Vol_Hexs = 0;
    Number_of_Vol_Pents_5 = 0;
    Number_of_Vol_Pents_6 = 0;
    Number_of_Vol_Tets = 0;

    Error_Flag = ug_io_write_cart3d (Grid_File, File_Format_,
                                     &Number_of_Bnd_Nodes, Number_of_Nodes,
                                     Number_of_Surf_Trias,
                                     Surf_ID_Flag,
                                     Surf_Tria_Connectivity,
                                     Coordinates);
  } 

  else if (strcmp (File_Type_Suffix_, ".ugrid") == 0)
  {
    Error_Flag = ug3_write_ugrid (Grid_File, File_Format_, 1,
                                  Number_of_BL_Vol_Tets,
                                  Number_of_Nodes,
                                  Number_of_Surf_Quads,
                                  Number_of_Surf_Trias,
                                  Number_of_Vol_Hexs,
                                  Number_of_Vol_Pents_5,
                                  Number_of_Vol_Pents_6,
                                  Number_of_Vol_Tets,
                                  Surf_ID_Flag,
                                  Surf_Quad_Connectivity,
                                  Surf_Tria_Connectivity,
                                  Vol_Hex_Connectivity,
                                  Vol_ID_Flag,
                                  Vol_Pent_5_Connectivity,
                                  Vol_Pent_6_Connectivity,
                                  Vol_Tet_Connectivity,
                                  Coordinates);

    if (Error_Flag == 0 && Number_of_Vol_Elems == 0)
      Error_Flag = ug3_write_gdata (Grid_File, File_Format_,
                                    Number_of_Nodes,
                                    Number_of_Surf_Quads,
                                    Number_of_Surf_Trias,
                                    Surf_Grid_BC_Flag, Surf_Reconnection_Flag,
                                    Initial_Normal_Spacing, BL_Thickness);

    if (Error_Flag)
    {
      Close_Flag = ug_fclose (Grid_File);
      return (Error_Flag);
    }
  }

  else if (strcmp (File_Type_Suffix_, ".cogsg") == 0)
  {
    Number_of_Surf_Quads = 0;
    Number_of_Vol_Hexs = 0;
    Number_of_Vol_Pents_5 = 0;
    Number_of_Vol_Pents_6 = 0;

    Error_Flag = ug_io_write_vgrid (Grid_File, Case_Name, 
                                    Number_of_Nodes,
                                    Number_of_Surf_Trias,
                                    Number_of_Vol_Tets,
                                    Number_of_BL_Vol_Tets,
                                    Surf_ID_Flag,
                                    Surf_Tria_Connectivity,
                                    Vol_Tet_Connectivity,
                                    Coordinates);
  }

  else if (strcmp (File_Type_Suffix_, ".cgns") == 0)
  {
    Close_Flag = ug_fclose (Grid_File);

    Error_Flag = ug_io_write_cgns_grid (Case_Name, File_Name, 
                                        Number_of_BL_Vol_Tets,
                                        Number_of_Nodes,
                                        Number_of_Surf_Quads,
                                        Number_of_Surf_Trias,
                                        Number_of_Vol_Hexs,
                                        Number_of_Vol_Pents_5,
                                        Number_of_Vol_Pents_6,
                                        Number_of_Vol_Tets,
                                        Surf_Grid_BC_Flag, Surf_ID_Flag, 
                                        Surf_Quad_Connectivity,
                                        Surf_Reconnection_Flag,
                                        Surf_Tria_Connectivity,
                                        Vol_Hex_Connectivity,
                                        Vol_ID_Flag, 
                                        Vol_Pent_5_Connectivity,
                                        Vol_Pent_6_Connectivity,
                                        Vol_Tet_Connectivity,
                                        Coordinates,
                                        Initial_Normal_Spacing, BL_Thickness);

    Close_Flag = 0;
  } 

  else
  { 
    Error_Flag = 615;
    ug_error_message ("*** ERROR 615 : unknown output grid file type ***");
  }

  // close file

  if (Close_Flag == 1)
    Close_Flag = ug_fclose (Grid_File);

  // exit on error

  if (Error_Flag > 0)
    return (615);

  // output grid information

  if (Message_Flag >= 1)
  {
    Number_of_Int_Trias = (Number_of_Vol_Tets*4 + Number_of_Vol_Pents_5*4
                         + Number_of_Vol_Pents_6*2 - Number_of_Surf_Trias) / 2;
    Number_of_Int_Quads = (Number_of_Vol_Pents_5 + Number_of_Vol_Pents_6*3
                         + Number_of_Vol_Hexs*6 - Number_of_Surf_Quads) / 2;

    Number_of_Int_Trias = MAX (Number_of_Int_Trias, 0);
    Number_of_Int_Quads = MAX (Number_of_Int_Quads, 0);

    strcpy (Name_Text, "");
    strncat (Name_Text, Grid_File_Name, ((INT_) strlen (Name_Text))-32);
    snprintf (Text, sizeof(Text), "UG_IO    : Grid File Name    = %s", Name_Text);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Quad Surface Faces=%10i", (int) Number_of_Surf_Quads);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Tria Surface Faces=%10i", (int) Number_of_Surf_Trias);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Quad Int. Faces   =%10i", (int) Number_of_Int_Quads);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Tria Int. Faces   =%10i", (int) Number_of_Int_Trias);
    ug_message (Text);
    if (Number_of_Bnd_Nodes > 0)
      snprintf (Text, sizeof(Text), "UG_IO    : Nodes             =%10i", (int) Number_of_Bnd_Nodes);
    else
      snprintf (Text, sizeof(Text), "UG_IO    : Nodes             =%10i", (int) Number_of_Nodes);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Total Elements    =%10i", (int) Number_of_Vol_Elems);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Hex Elements      =%10i", (int) Number_of_Vol_Hexs);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Pent_5 Elements   =%10i", (int) Number_of_Vol_Pents_5);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Pent_6 Elements   =%10i", (int) Number_of_Vol_Pents_6);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG_IO    : Tet Elements      =%10i", (int) Number_of_Vol_Tets);
    ug_message (Text);

    if (Vol_ID_Flag && Number_of_Vol_Elems > 0)
    {
      Vol_ID_Max = ug_max_int (1, Number_of_Vol_Elems, Vol_ID_Flag);
      Vol_ID_Min = ug_min_int (1, Number_of_Vol_Elems, Vol_ID_Flag);

      snprintf (Text, sizeof(Text), "UG_IO    : Volume ID Min/Max =%10i%10i", (int) Vol_ID_Min, (int) Vol_ID_Max);
      ug_message (Text);
    }

    ug_message (" ");
    ug_cpu_message ("UG_IO    :");
  }

  // compress file

  ug_compress_file (1, File_Compression_Suffix_, File_Name);

  return (0);

}
