#include "UG3_LIB.h"

static INT_ ug3_write_sfunc_func
 (FILE * Func_File,
  INT_ File_Format,
  INT_ n,
  INT_ Number_of_Nodes,
  INT_ Number_of_U_Funcs,
  DOUBLE_1D * U_Funcs);

static INT_ ug3_write_sfunc_label
 (FILE * Func_File,
  INT_ File_Format,
  INT_ Number_of_U_Funcs,
  CHAR_21 * U_Func_Labels);

INT_ ug3_write_sfunc
 (FILE * Func_File,
  INT_ File_Format,
  INT_ NDim,
  INT_ Number_of_Nodes,
  INT_ Number_of_U_Scalars,
  INT_ Number_of_U_Vectors,
  INT_ Number_of_U_Matrixes,
  INT_ Number_of_U_Metrics,
  INT_ SFUNC_Flag,
  CHAR_21 * U_Scalar_Labels,
  CHAR_21 * U_Vector_Labels,
  CHAR_21 * U_Matrix_Labels,
  CHAR_21 * U_Metric_Labels,
  DOUBLE_1D * U_Scalars,
  DOUBLE_1D * U_Vectors,
  DOUBLE_1D * U_Matrixes,
  DOUBLE_1D * U_Metrics)

{

/*
 * Write function data to a SFUNC function file or
 * write function data to a UFUNC function file if SFUNC_Flag=0.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_write_sfunc.c,v 1.8 2022/11/21 00:25:14 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_ n, ierr, Number_of_Bytes,
       Number_of_Write_Items, Special_Write_Items, Write_Flag;

  if (U_Scalars == NULL) Number_of_U_Scalars = 0;
  if (U_Vectors == NULL) Number_of_U_Vectors = 0;
  if (U_Matrixes == NULL) Number_of_U_Matrixes = 0;
  if (U_Metrics == NULL) Number_of_U_Metrics = 0;

  if (File_Format == UG_FIO_FORMATTED)
  {
    Write_Flag = fprintf (Func_File, "%i %i %i",
                          (int) Number_of_Nodes,
                          (int) Number_of_U_Scalars,
                          (int) Number_of_U_Vectors);

    if (SFUNC_Flag)
      Write_Flag = fprintf (Func_File, " %i %i\n",
                            (int) Number_of_U_Matrixes,
                            (int) Number_of_U_Metrics);
    else
      Write_Flag = fprintf (Func_File, "\n");

    if (Write_Flag < 0)
    {
      ug_error_message ("*** ERROR : error writing SFUNC function file ***");
      return (1);
    }
  }
  else
  {
    Special_Write_Items = ((File_Format == UG_FIO_UNFORMATTED_DOUBLE ||
                            File_Format == UG_FIO_UNFORMATTED_SINGLE) ? 1 : 0);

    Number_of_Bytes = (3 + (SFUNC_Flag * 2)) * ((INT_) (sizeof (INT_)));

    Write_Flag = 0;
    
    if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
      Write_Flag = Write_Flag + ug_fwrite (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

    Write_Flag = Write_Flag + ug_fwrite (&Number_of_Nodes, sizeof (INT_), 1, Func_File);
    Write_Flag = Write_Flag + ug_fwrite (&Number_of_U_Scalars, sizeof (INT_), 1, Func_File);
    Write_Flag = Write_Flag + ug_fwrite (&Number_of_U_Vectors, sizeof (INT_), 1, Func_File);

    if (SFUNC_Flag)
    {
      Write_Flag = Write_Flag + ug_fwrite (&Number_of_U_Matrixes, sizeof (INT_), 1, Func_File);
      Write_Flag = Write_Flag + ug_fwrite (&Number_of_U_Metrics, sizeof (INT_), 1, Func_File);
    }

    if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
      Write_Flag = Write_Flag + ug_fwrite (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

    Number_of_Write_Items = 3 + (SFUNC_Flag * 2) + Special_Write_Items + Special_Write_Items;

    if (Write_Flag != Number_of_Write_Items)
    {
      ug_error_message ("*** ERROR : error writing SFUNC function file ***");
      return (1);
    }
  }

  ierr = ug3_write_sfunc_label (Func_File, File_Format, Number_of_U_Scalars,
                                U_Scalar_Labels);

  if (ierr == 0)
    ierr = ug3_write_sfunc_label (Func_File, File_Format, Number_of_U_Vectors,
                                  U_Vector_Labels);

  if (SFUNC_Flag)
  {
    if (ierr == 0)
      ierr = ug3_write_sfunc_label (Func_File, File_Format, Number_of_U_Matrixes,
                                    U_Matrix_Labels);

    if (ierr == 0)
      ierr = ug3_write_sfunc_label (Func_File, File_Format, Number_of_U_Metrics,
                                    U_Metric_Labels);
  }

  n = 1;

  ierr = ug3_write_sfunc_func (Func_File, File_Format, n,
                               Number_of_Nodes, Number_of_U_Scalars,
                               U_Scalars);

  n = NDim;

  if (ierr == 0)
    ierr = ug3_write_sfunc_func (Func_File, File_Format, n,
                                 Number_of_Nodes, Number_of_U_Vectors,
                                 U_Vectors);

  if (SFUNC_Flag)
  {
    n = NDim*NDim;

    if (ierr == 0)
      ierr = ug3_write_sfunc_func (Func_File, File_Format, n,
                                   Number_of_Nodes, Number_of_U_Matrixes,
                                   U_Matrixes);

    n = 3*(NDim-1);

    if (ierr == 0)
      ierr = ug3_write_sfunc_func (Func_File, File_Format, n,
                                   Number_of_Nodes, Number_of_U_Metrics,
                                   U_Metrics);
  }

  if (ierr)
    return (ierr);

  return (0);

}

INT_ ug3_write_sfunc_func
 (FILE * Func_File,
  INT_ File_Format,
  INT_ n,
  INT_ Number_of_Nodes,
  INT_ Number_of_U_Funcs,
  DOUBLE_1D * U_Funcs)

{

/*
 * Write a single function.
 */

  DOUBLE_1D * U_Func;

  INT_ i, Index, Number_of_Bytes, Number_of_Write_Items, Special_Write_Items,
       Write_Flag, U_Func_Index, U_Func_Location;

  float TMP_float;

  if (Number_of_U_Funcs == 0)
    return (0);

  if (File_Format == UG_FIO_FORMATTED)
  {
    U_Func_Location = 0;

    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      U_Func = &U_Funcs[U_Func_Location];

      for (Index = 1; Index <= Number_of_Nodes; ++Index)
      {
        for (i = 0; i < n; ++i)
        {
          Write_Flag = fprintf (Func_File, " %.15g", U_Func[i+Index*n]);
        }

        Write_Flag = fprintf (Func_File, "\n");
      }

      U_Func_Location = U_Func_Location + n*(Number_of_Nodes+1);

      if (Write_Flag < 0)
      {
        ug_error_message ("*** ERROR : error writing SFUNC function file ***");
        return (1);
      }
    }
  }

  else
  {
    Special_Write_Items = ((File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE) ? 1 : 0);

    if (File_Format == UG_FIO_UNFORMATTED_SINGLE)
      Number_of_Bytes = n * Number_of_Nodes * ((INT_) (sizeof (float)));
    else if (File_Format == UG_FIO_UNFORMATTED_DOUBLE)
      Number_of_Bytes = n * Number_of_Nodes * ((INT_) (sizeof (double)));

    U_Func_Location = 0;

    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      Write_Flag = 0;

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
        Write_Flag = Write_Flag + ug_fwrite (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

      U_Func = &U_Funcs[U_Func_Location];

      if (File_Format == UG_FIO_UNFORMATTED_SINGLE || File_Format == UG_FIO_BINARY_SINGLE)
      {
        for (Index = 1; Index <= Number_of_Nodes; ++Index)
        {
          for (i = 0; i < n; ++i)
          {
            TMP_float = (float) U_Func[i+Index*n];

            Write_Flag = Write_Flag + ug_fwrite (&TMP_float, sizeof (float), 1, Func_File);
          }
        }
      }
      else
        Write_Flag = Write_Flag + ug_fwrite (&U_Func[n], sizeof (double), Number_of_Nodes*n, Func_File);

      U_Func_Location = U_Func_Location + n*(Number_of_Nodes+1);

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
        Write_Flag = Write_Flag + ug_fwrite (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

      Number_of_Write_Items = Number_of_Nodes*n + Special_Write_Items + Special_Write_Items;

      if (Write_Flag != Number_of_Write_Items)
      {
        ug_error_message ("*** ERROR : error writing SFUNC function file ***");
        return (1);
      }
    }
  }

  return (0);

}

INT_ ug3_write_sfunc_label
 (FILE * Func_File,
  INT_ File_Format,
  INT_ Number_of_U_Funcs,
  CHAR_21 * U_Func_Labels)

{

/*
 * Write a single function.
 */

  CHAR_21 * U_Func_Label;

  INT_ Number_of_Bytes, Number_of_Write_Items, Special_Write_Items,
       Write_Flag, U_Func_Index; 

  if (Number_of_U_Funcs == 0)
    return (0);

  if (File_Format == UG_FIO_FORMATTED)
  {
    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      if (U_Func_Labels)
        Write_Flag = fprintf (Func_File, "%s\n", U_Func_Labels[U_Func_Index]);
      else
        Write_Flag = fprintf (Func_File, "Func-%i\n", (int) U_Func_Index);

      if (Write_Flag < 0)
      {
        ug_error_message ("*** ERROR : error writing SFUNC function file ***");
        return (1);
      }
    }
  }
  else
  {
    Special_Write_Items = ((File_Format == UG_FIO_UNFORMATTED_DOUBLE ||
                            File_Format == UG_FIO_UNFORMATTED_SINGLE) ? 1 : 0);

    Number_of_Bytes = ((INT_) (sizeof (CHAR_21)));

    Number_of_Write_Items = 21 + Special_Write_Items + Special_Write_Items;

    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      if (U_Func_Labels)
        U_Func_Label = &U_Func_Labels[U_Func_Index];
      else
        snprintf ((char *) U_Func_Label, sizeof(CHAR_21), "Func-%-4i         ", (int) U_Func_Index);

      Write_Flag = 0;

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE) 
        Write_Flag = Write_Flag + ug_fwrite (&Number_of_Bytes, sizeof (INT_), 1, Func_File); 

      Write_Flag = Write_Flag + ug_fwrite (U_Func_Label, sizeof (char), 21, Func_File);

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
        Write_Flag = Write_Flag + ug_fwrite (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

      if (Write_Flag != Number_of_Write_Items)
      {
        ug_error_message ("*** ERROR : error writing SFUNC function file ***");
        return (1);
      }
    }
  }

  return (0);

}
