#include "UG3_LIB.h"

/*
 * Routines to compute and check tet, prism, pyramid, and/or hex element volume.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_vol.c,v 1.18 2022/11/21 00:25:14 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

double ug3_vol (double x11,
                double x12,
                double x13,
                double x21,
                double x22,
                double x23,
                double x31,
                double x32,
                double x33,
                double x41,
                double x42,
                double x43)
{

  // Compute tet element volume.

  double dx211, dx212, dx213, dx311, dx312, dx313, dx411, dx412, dx413, vol;

  dx211 = x21 - x11;
  dx212 = x22 - x12;
  dx213 = x23 - x13;
  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx411 = x41 - x11;
  dx412 = x42 - x12;
  dx413 = x43 - x13;

  vol = dx211 * (dx312 * dx413 - dx313 * dx412)
      + dx212 * (dx313 * dx411 - dx311 * dx413)
      + dx213 * (dx311 * dx412 - dx312 * dx411);

  return (vol);
}

double ug3_tet_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  DOUBLE_3D * x)

{

  // Compute tet element volume.

  double vol, x11, x12, x13, x21, x22, x23, x31, x32, x33, x41, x42, x43;

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];

  vol = ug3_vol (x11, x12, x13, x21, x22, x23, x31, x32, x33, x41, x42, x43);

  return (vol);
}

double ug3_pyramid_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  INT_ inode5,
  DOUBLE_3D * x)

{

  // Compute pyramid element volume.

  double dx211, dx212, dx213,
         dx311, dx312, dx313,
         dx411, dx412, dx413,
         dx511, dx512, dx513,
         vol,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53;

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];

  dx211 = x21 - x11;
  dx212 = x22 - x12;
  dx213 = x23 - x13;
  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx411 = x41 - x11;
  dx412 = x42 - x12;
  dx413 = x43 - x13;
  dx511 = x51 - x11;
  dx512 = x52 - x12;
  dx513 = x53 - x13;

  vol = dx511 * (dx212 * dx313 - dx213 * dx312)
      + dx512 * (dx213 * dx311 - dx211 * dx313)
      + dx513 * (dx211 * dx312 - dx212 * dx311)
      + dx411 * (dx512 * dx313 - dx513 * dx312)
      + dx412 * (dx513 * dx311 - dx511 * dx313)
      + dx413 * (dx511 * dx312 - dx512 * dx311);

  return (vol);

}

double ug3_prism_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  INT_ inode5,
  INT_ inode6,
  DOUBLE_3D * x)

{

  // Compute prism element volume.

  double dx211, dx212, dx213,
         dx311, dx312, dx313,
         dx411, dx412, dx413,
         dx511, dx512, dx513,
         dx611, dx612, dx613,
         vol,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53,
         x61, x62, x63;

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];
  x61 = x[inode6][0];
  x62 = x[inode6][1];
  x63 = x[inode6][2];

  dx211 = x21 - x11;
  dx212 = x22 - x12;
  dx213 = x23 - x13;
  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx411 = x41 - x11;
  dx412 = x42 - x12;
  dx413 = x43 - x13;
  dx511 = x51 - x11;
  dx512 = x52 - x12;
  dx513 = x53 - x13;
  dx611 = x61 - x11;
  dx612 = x62 - x12;
  dx613 = x63 - x13;

  vol = dx611 * (dx212 * dx313 - dx213 * dx312)
      + dx612 * (dx213 * dx311 - dx211 * dx313)
      + dx613 * (dx211 * dx312 - dx212 * dx311)
      + dx511 * (dx212 * dx613 - dx213 * dx612)
      + dx512 * (dx213 * dx611 - dx211 * dx613)
      + dx513 * (dx211 * dx612 - dx212 * dx611)
      + dx411 * (dx512 * dx613 - dx513 * dx612)
      + dx412 * (dx513 * dx611 - dx511 * dx613)
      + dx413 * (dx511 * dx612 - dx512 * dx611);

  return (vol);

}

double ug3_hex_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  INT_ inode5,
  INT_ inode6,
  INT_ inode7,
  INT_ inode8,
  DOUBLE_3D * x)

{

  // Compute hex element volume.

  double dx211, dx212, dx213,
         dx311, dx312, dx313,
         dx411, dx412, dx413,
         dx511, dx512, dx513,
         dx611, dx612, dx613,
         dx711, dx712, dx713,
         dx811, dx812, dx813,
         vol,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53,
         x61, x62, x63,
         x71, x72, x73,
         x81, x82, x83;

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];
  x61 = x[inode6][0];
  x62 = x[inode6][1];
  x63 = x[inode6][2];
  x71 = x[inode7][0];
  x72 = x[inode7][1];
  x73 = x[inode7][2];
  x81 = x[inode8][0];
  x82 = x[inode8][1];
  x83 = x[inode8][2];

  dx211 = x21 - x11;
  dx212 = x22 - x12;
  dx213 = x23 - x13;
  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx411 = x41 - x11;
  dx412 = x42 - x12;
  dx413 = x43 - x13;
  dx511 = x51 - x11;
  dx512 = x52 - x12;
  dx513 = x53 - x13;
  dx611 = x61 - x11;
  dx612 = x62 - x12;
  dx613 = x63 - x13;
  dx711 = x71 - x11;
  dx712 = x72 - x12;
  dx713 = x73 - x13;
  dx811 = x81 - x11;
  dx812 = x82 - x12;
  dx813 = x83 - x13;

  vol = dx711 * (dx212 * dx313 - dx213 * dx312)
      + dx712 * (dx213 * dx311 - dx211 * dx313)
      + dx713 * (dx211 * dx312 - dx212 * dx311)
      + dx611 * (dx212 * dx713 - dx213 * dx712)
      + dx612 * (dx213 * dx711 - dx211 * dx713)
      + dx613 * (dx211 * dx712 - dx212 * dx711)
      + dx511 * (dx612 * dx713 - dx613 * dx712)
      + dx512 * (dx613 * dx711 - dx611 * dx713)
      + dx513 * (dx611 * dx712 - dx612 * dx711)
      + dx711 * (dx312 * dx413 - dx313 * dx412)
      + dx712 * (dx313 * dx411 - dx311 * dx413)
      + dx713 * (dx311 * dx412 - dx312 * dx411)
      + dx811 * (dx712 * dx413 - dx713 * dx412)
      + dx812 * (dx713 * dx411 - dx711 * dx413)
      + dx813 * (dx711 * dx412 - dx712 * dx411)
      + dx511 * (dx712 * dx813 - dx713 * dx812)
      + dx512 * (dx713 * dx811 - dx711 * dx813)
      + dx513 * (dx711 * dx812 - dx712 * dx811);

  return (vol);

}

INT_ ug3_chk_tet_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  double tol,
  double *vol,
  DOUBLE_3D * x)

{

  // Compute and check tet element volume.

  double dc0, dx, dx1, dx2, dx3,
         voltol,
         x1max, x1min, x2max, x2min, x3max, x3min,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43;

  dc0 = 0.0;

  *vol = ug3_tet_vol (inode1, inode2, inode3, inode4, x);

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];

  x1max = MAX (x11, x21);
  x1max = MAX (x31, x1max);
  x1max = MAX (x41, x1max);
  x2max = MAX (x12, x22);
  x2max = MAX (x32, x2max);
  x2max = MAX (x42, x2max);
  x3max = MAX (x13, x23);
  x3max = MAX (x33, x3max);
  x3max = MAX (x43, x3max);

  x1min = MIN (x11, x21);
  x1min = MIN (x31, x1min);
  x1min = MIN (x41, x1min);
  x2min = MIN (x12, x22);
  x2min = MIN (x32, x2min);
  x2min = MIN (x42, x2min);
  x3min = MIN (x13, x23);
  x3min = MIN (x33, x3min);
  x3min = MIN (x43, x3min);

  dx1 = x1max - x1min;
  dx2 = x2max - x2min;
  dx3 = x3max - x3min;

  dx = MAX (dx1, dx2);
  dx = MAX (dx3, dx);

  voltol = tol * dx * dx * dx;

  if (*vol < voltol)
    return (1);
  else if (*vol < dc0)
    return (2);

  return (0);

}

INT_ ug3_chk_pyramid_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  INT_ inode5,
  double tol,
  double *vol,
  DOUBLE_3D * x)

{

  // Compute and check pyramid element volume.

  double dc0, dx, dx1, dx2, dx3,
         voltol,
         x1max, x1min, x2max, x2min, x3max, x3min,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53;

  dc0 = 0.0;

  *vol = ug3_pyramid_vol (inode1, inode2, inode3, inode4, inode5, x);

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];

  x1max = MAX (x11, x21);
  x1max = MAX (x31, x1max);
  x1max = MAX (x41, x1max);
  x1max = MAX (x51, x1max);
  x2max = MAX (x12, x22);
  x2max = MAX (x32, x2max);
  x2max = MAX (x42, x2max);
  x2max = MAX (x52, x2max);
  x3max = MAX (x13, x23);
  x3max = MAX (x33, x3max);
  x3max = MAX (x43, x3max);
  x3max = MAX (x53, x3max);

  x1min = MIN (x11, x21);
  x1min = MIN (x31, x1min);
  x1min = MIN (x41, x1min);
  x1min = MIN (x51, x1min);
  x2min = MIN (x12, x22);
  x2min = MIN (x32, x2min);
  x2min = MIN (x42, x2min);
  x2min = MIN (x52, x2min);
  x3min = MIN (x13, x23);
  x3min = MIN (x33, x3min);
  x3min = MIN (x43, x3min);
  x3min = MIN (x53, x3min);

  dx1 = x1max - x1min;
  dx2 = x2max - x2min;
  dx3 = x3max - x3min;

  dx = MAX (dx1, dx2);
  dx = MAX (dx3, dx);

  voltol = tol * dx * dx * dx;

  if (*vol < voltol)
    return (1);
  else if (*vol < dc0)
    return (2);

  return (0);

}

INT_ ug3_chk_prism_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  INT_ inode5,
  INT_ inode6,
  double tol,
  double *vol,
  DOUBLE_3D * x)

{

  // Compute and check prism element volume.

  double dc0, dx, dx1, dx2, dx3,
         voltol,
         x1max, x1min, x2max, x2min, x3max, x3min,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53,
         x61, x62, x63;

  dc0 = 0.0;

  *vol = ug3_prism_vol (inode1, inode2, inode3, inode4, inode5, inode6, x);

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];
  x61 = x[inode6][0];
  x62 = x[inode6][1];
  x63 = x[inode6][2];

  x1max = MAX (x11, x21);
  x1max = MAX (x31, x1max);
  x1max = MAX (x41, x1max);
  x1max = MAX (x51, x1max);
  x1max = MAX (x61, x1max);
  x2max = MAX (x12, x22);
  x2max = MAX (x32, x2max);
  x2max = MAX (x42, x2max);
  x2max = MAX (x52, x2max);
  x2max = MAX (x62, x2max);
  x3max = MAX (x13, x23);
  x3max = MAX (x33, x3max);
  x3max = MAX (x43, x3max);
  x3max = MAX (x53, x3max);
  x3max = MAX (x63, x3max);

  x1min = MIN (x11, x21);
  x1min = MIN (x31, x1min);
  x1min = MIN (x41, x1min);
  x1min = MIN (x51, x1min);
  x1min = MIN (x61, x1min);
  x2min = MIN (x12, x22);
  x2min = MIN (x32, x2min);
  x2min = MIN (x42, x2min);
  x2min = MIN (x52, x2min);
  x2min = MIN (x62, x2min);
  x3min = MIN (x13, x23);
  x3min = MIN (x33, x3min);
  x3min = MIN (x43, x3min);
  x3min = MIN (x53, x3min);
  x3min = MIN (x63, x3min);

  dx1 = x1max - x1min;
  dx2 = x2max - x2min;
  dx3 = x3max - x3min;

  dx = MAX (dx1, dx2);
  dx = MAX (dx3, dx);

  voltol = tol * dx * dx * dx;

  if (*vol < voltol)
    return (1);
  else if (*vol < dc0)
    return (2);

  return (0);

}

INT_ ug3_chk_hex_vol
 (INT_ inode1,
  INT_ inode2,
  INT_ inode3,
  INT_ inode4,
  INT_ inode5,
  INT_ inode6,
  INT_ inode7,
  INT_ inode8,
  double tol,
  double *vol,
  DOUBLE_3D * x)

{

  // Compute and check hex element volume.

  double dc0, dx, dx1, dx2, dx3,
         voltol,
         x1max, x1min, x2max, x2min, x3max, x3min,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53,
         x61, x62, x63,
         x71, x72, x73,
         x81, x82, x83;

  dc0 = 0.0;

  *vol = ug3_hex_vol (inode1, inode2, inode3, inode4,
                      inode5, inode6, inode7, inode8, x);

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];
  x61 = x[inode6][0];
  x62 = x[inode6][1];
  x63 = x[inode6][2];
  x71 = x[inode7][0];
  x72 = x[inode7][1];
  x73 = x[inode7][2];
  x81 = x[inode8][0];
  x82 = x[inode8][1];
  x83 = x[inode8][2];

  x1max = MAX (x11, x21);
  x1max = MAX (x31, x1max);
  x1max = MAX (x41, x1max);
  x1max = MAX (x51, x1max);
  x1max = MAX (x61, x1max);
  x1max = MAX (x71, x1max);
  x1max = MAX (x81, x1max);
  x2max = MAX (x12, x22);
  x2max = MAX (x32, x2max);
  x2max = MAX (x42, x2max);
  x2max = MAX (x52, x2max);
  x2max = MAX (x62, x2max);
  x2max = MAX (x72, x2max);
  x2max = MAX (x82, x2max);
  x3max = MAX (x13, x23);
  x3max = MAX (x33, x3max);
  x3max = MAX (x43, x3max);
  x3max = MAX (x53, x3max);
  x3max = MAX (x63, x3max);
  x3max = MAX (x73, x3max);
  x3max = MAX (x83, x3max);

  x1min = MIN (x11, x21);
  x1min = MIN (x31, x1min);
  x1min = MIN (x41, x1min);
  x1min = MIN (x51, x1min);
  x1min = MIN (x61, x1min);
  x1min = MIN (x71, x1min);
  x1min = MIN (x81, x1min);
  x2min = MIN (x12, x22);
  x2min = MIN (x32, x2min);
  x2min = MIN (x42, x2min);
  x2min = MIN (x52, x2min);
  x2min = MIN (x62, x2min);
  x2min = MIN (x72, x2min);
  x2min = MIN (x82, x2min);
  x3min = MIN (x13, x23);
  x3min = MIN (x33, x3min);
  x3min = MIN (x43, x3min);
  x3min = MIN (x53, x3min);
  x3min = MIN (x63, x3min);
  x3min = MIN (x73, x3min);
  x3min = MIN (x83, x3min);

  dx1 = x1max - x1min;
  dx2 = x2max - x2min;
  dx3 = x3max - x3min;

  dx = MAX (dx1, dx2);
  dx = MAX (dx3, dx);

  voltol = tol * dx * dx * dx;

  if (*vol < voltol)
    return (1);
  else if (*vol < dc0)
    return (2);

  return (0);

}

INT_ ug3_chkvol
 (INT_ mmsg,
  INT_ nelem,
  INT_4D * iniel,
  double tol,
  DOUBLE_3D * x)

{

  // Check tet element volumes.

  INT_ ierr;

  ierr = ug3_chkvolm (mmsg, nelem, 0, 0, 0, iniel, NULL, NULL, NULL, tol, x);

  return (ierr);

}

INT_ ug3_chkvolm
 (INT_ mmsg,
  INT_ nelem,
  INT_ nelemc5,
  INT_ nelemc6,
  INT_ nelemc8,
  INT_4D * iniel,
  INT_5D * inielc5,
  INT_6D * inielc6,
  INT_8D * inielc8,
  double tol,
  DOUBLE_3D * x)

{

  // Check mixed element volumes.

  CHAR_133 Text;

  INT_ ielem, ierr,
       inode1, inode2, inode3, inode4, inode5, inode6, inode7, inode8,
       jerr, nsmall;

  double vol;

  ierr = 0;

  nsmall = 0;

  if (mmsg == 2 && nelem > 0)
  {
    snprintf (Text, sizeof(Text), "Checking Tets      : Elements          =%10i", (int) nelem);
    ug_message (Text);
  }

  for (ielem = 1; ielem <= nelem; ++ielem)
  {
    inode1 = iniel[ielem][0];
    inode2 = iniel[ielem][1];
    inode3 = iniel[ielem][2];
    inode4 = iniel[ielem][3];

    jerr = ug3_chk_tet_vol (inode1, inode2, inode3, inode4, tol, &vol, x);

    if (jerr == -1)
    {
      ++nsmall;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Tet Vol<Tol       @%-10i", (int) ielem);
        ug_message (Text);
      }
    }

    else if (jerr == 1)
    {
      ierr = 1;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Tet Vol<0         @%-10i", (int) ielem);
        ug_message (Text);
      }
    }
  }

  if (mmsg == 2 && nelemc5 > 0)
  {
    snprintf (Text, sizeof(Text), "Checking Pyramids  : Elements          =%10i", (int) nelemc5);
    ug_message (Text);
  }

  for (ielem = 1; ielem <= nelemc5; ++ielem)
  {
    inode1 = inielc5[ielem][0];
    inode2 = inielc5[ielem][1];
    inode3 = inielc5[ielem][2];
    inode4 = inielc5[ielem][3];
    inode5 = inielc5[ielem][4];

    jerr = ug3_chk_pyramid_vol (inode1, inode2, inode3, inode4, inode5,
                                tol, &vol, x);

    if (jerr == -1)
    {
      ++nsmall;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Pyramid Vol<Tol   @%-10i", (int) ielem);
        ug_message (Text);
      }
    }

    else if (jerr == 1)
    {
      ierr = 1;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Pyramid Vol<0     @%-10i", (int) ielem);
        ug_message (Text);
      }
    }
  }

  if (mmsg == 2 && nelemc6 > 0)
  {
    snprintf (Text, sizeof(Text), "Checking Prisms    : Elements          =%10i", (int) nelemc6);
    ug_message (Text);
  }

  for (ielem = 1; ielem <= nelemc6; ++ielem)
  {
    inode1 = inielc6[ielem][0];
    inode2 = inielc6[ielem][1];
    inode3 = inielc6[ielem][2];
    inode4 = inielc6[ielem][3];
    inode5 = inielc6[ielem][4];
    inode6 = inielc6[ielem][5];

    jerr = ug3_chk_prism_vol (inode1, inode2, inode3, inode4, inode5, inode6,
                              tol, &vol, x);

    if (jerr == -1)
    {
      ++nsmall;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Prism Vol<Tol     @%-10i", (int) ielem);
        ug_message (Text);
      }
    }

    else if (jerr == 1)
    {
      ierr = 1;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Prism Vol<0       @%-10i", (int) ielem);
        ug_message (Text);
      }
    }
  }

  if (mmsg == 2 && nelemc8 > 0)
  {
    snprintf (Text, sizeof(Text), "Checking Hexs      : Elements          =%10i", (int) nelemc8);
    ug_message (Text);
  }

  for (ielem = 1; ielem <= nelemc8; ++ielem)
  {
    inode1 = inielc8[ielem][0];
    inode2 = inielc8[ielem][1];
    inode3 = inielc8[ielem][2];
    inode4 = inielc8[ielem][3];
    inode5 = inielc8[ielem][4];
    inode6 = inielc8[ielem][5];
    inode7 = inielc8[ielem][6];
    inode8 = inielc8[ielem][7];

    jerr = ug3_chk_hex_vol (inode1, inode2, inode3, inode4,
                            inode5, inode6, inode7, inode8, tol, &vol, x);

    if (jerr == -1)
    {
      ++nsmall;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Hex Vol<Tol       @%-10i", (int) ielem);
        ug_message (Text);
      }
    }

    else if (jerr == 1)
    {
      ierr = 1;

      if (mmsg >= 2)
      {
        snprintf (Text, sizeof(Text), "VOL CHECK          : Hex Vol<0         @%-10i", (int) ielem);
        ug_message (Text);
      }
    }
  }

  if (ierr)
  {
    if (mmsg >= 2)
      ug_error_message ("*** ERROR 325 : found element with negative volume ***");
    return (325);
  }

  if (mmsg == 2)
  {
    if (nsmall == 0)
      ug_message ("Checking           : Volume OK");

    else
    {
      snprintf (Text, sizeof(Text), "Checking           : Small Volumes     =%10i", (int) nsmall);
      ug_message (Text);
    }
  }

  ierr = (nsmall) ? -nsmall: 0;

  return (ierr);

}

double ug3_tvol
 (INT_ nelem,
  INT_4D * iniel,
  DOUBLE_3D * x)

{

  // Compute total tet element volume.

  double tvol;

  tvol = ug3_tvolm (nelem, 0, 0, 0, iniel, NULL, NULL, NULL, x);

  return (tvol);

}

double ug3_tvolm
 (INT_ nelem,
  INT_ nelemc5,
  INT_ nelemc6,
  INT_ nelemc8,
  INT_4D * iniel,
  INT_5D * inielc5,
  INT_6D * inielc6,
  INT_8D * inielc8,
  DOUBLE_3D * x)

{

  // Computed total mixed element volume.

  INT_ ielem, inode1, inode2, inode3, inode4, inode5, inode6, inode7, inode8;

  double vol;
  double tvol = 0;

  for (ielem = 1; ielem <= nelem; ++ielem)
  {
    inode1 = iniel[ielem][0];
    inode2 = iniel[ielem][1];
    inode3 = iniel[ielem][2];
    inode4 = iniel[ielem][3];

    vol = ug3_tet_vol (inode1, inode2, inode3, inode4, x);

    tvol = tvol + vol;
  }

  for (ielem = 1; ielem <= nelemc5; ++ielem)
  {
    inode1 = inielc5[ielem][0];
    inode2 = inielc5[ielem][1];
    inode3 = inielc5[ielem][2];
    inode4 = inielc5[ielem][3];
    inode5 = inielc5[ielem][4];

    vol = ug3_pyramid_vol (inode1, inode2, inode3, inode4, inode5, x);

    tvol = tvol + vol;
  }

  for (ielem = 1; ielem <= nelemc6; ++ielem)
  {
    inode1 = inielc6[ielem][0];
    inode2 = inielc6[ielem][1];
    inode3 = inielc6[ielem][2];
    inode4 = inielc6[ielem][3];
    inode5 = inielc6[ielem][4];
    inode6 = inielc6[ielem][5];

    vol = ug3_prism_vol (inode1, inode2, inode3, inode4, inode5, inode6, x);

    tvol = tvol + vol;
  }

  for (ielem = 1; ielem <= nelemc8; ++ielem)
  {
    inode1 = inielc8[ielem][0];
    inode2 = inielc8[ielem][1];
    inode3 = inielc8[ielem][2];
    inode4 = inielc8[ielem][3];
    inode5 = inielc8[ielem][4];
    inode6 = inielc8[ielem][5];
    inode7 = inielc8[ielem][6];
    inode8 = inielc8[ielem][7];

    vol = ug3_hex_vol (inode1, inode2, inode3, inode4,
                       inode5, inode6, inode7, inode8, x);

    tvol = tvol + vol;
  }

  return (tvol);

}

INT_ ug3_volm
 (INT_ nelem,
  INT_ nelemc5,
  INT_ nelemc6,
  INT_ nelemc8,
  INT_4D * iniel,
  INT_5D * inielc5,
  INT_6D * inielc6,
  INT_8D * inielc8,
  DOUBLE_1D ** vol,
  DOUBLE_3D * x)

{

  // Computed mixed element volumes.

  INT_ ielem, inode1, inode2, inode3, inode4, inode5, inode6, inode7, inode8;
  INT_ ierr = 0;
  INT_ jelem = 0;

  if (*vol == NULL)
  {
    *vol = (DOUBLE_1D *) ug_malloc (&ierr, (nelem+nelemc5+nelemc6+nelemc8+1) * sizeof (DOUBLE_1D));

    if (ierr)
    {
      ug_error_message ("*** ERROR 100345 : unable to allocate required memory ***");
      return (100345);
    }
  }

  for (ielem = 1; ielem <= nelem; ++ielem)
  {
    inode1 = iniel[ielem][0];
    inode2 = iniel[ielem][1];
    inode3 = iniel[ielem][2];
    inode4 = iniel[ielem][3];

    ++jelem;

    (*vol)[jelem] = ug3_tet_vol (inode1, inode2, inode3, inode4, x);
  }

  for (ielem = 1; ielem <= nelemc5; ++ielem)
  {
    inode1 = inielc5[ielem][0];
    inode2 = inielc5[ielem][1];
    inode3 = inielc5[ielem][2];
    inode4 = inielc5[ielem][3];
    inode5 = inielc5[ielem][4];

    ++jelem;

    (*vol)[jelem] = ug3_pyramid_vol (inode1, inode2, inode3, inode4, inode5, x);
  }

  for (ielem = 1; ielem <= nelemc6; ++ielem)
  {
    inode1 = inielc6[ielem][0];
    inode2 = inielc6[ielem][1];
    inode3 = inielc6[ielem][2];
    inode4 = inielc6[ielem][3];
    inode5 = inielc6[ielem][4];
    inode6 = inielc6[ielem][5];

    ++jelem;

    (*vol)[jelem] = ug3_prism_vol (inode1, inode2, inode3,
                                   inode4, inode5, inode6, x);
  }

  for (ielem = 1; ielem <= nelemc8; ++ielem)
  {
    inode1 = inielc8[ielem][0];
    inode2 = inielc8[ielem][1];
    inode3 = inielc8[ielem][2];
    inode4 = inielc8[ielem][3];
    inode5 = inielc8[ielem][4];
    inode6 = inielc8[ielem][5];
    inode7 = inielc8[ielem][6];
    inode8 = inielc8[ielem][7];

    ++jelem;

    (*vol)[jelem] = ug3_hex_vol (inode1, inode2, inode3, inode4,
                                 inode5, inode6, inode7, inode8, x);
  }

  return (0);
}
