#include "UG3_LIB.h"

static INT_ ug3_tbfsplit
 (INT_ mmsg,
  INT_ *nbface,
  INT_ *nnode,
  INT_1D ** ibcibf_ptr,
  INT_1D ** idibf_ptr,
  INT_1D ** ierribf_ptr,
  INT_3D ** inibf_ptr,
  INT_1D ** iqibf_ptr,
  INT_1D ** irfibf_ptr,
  DOUBLE_1D ** del_ptr,
  DOUBLE_1D ** ds_ptr,
  DOUBLE_3D ** x_ptr);

static INT_ ug3_tbfd0
 (INT_ mmsg,
  INT_ *nbface,
  INT_ *nnode,
  INT_1D ** ibcibf_ptr,
  INT_1D ** idibf_ptr,
  INT_1D ** ierribf_ptr,
  INT_3D ** inibf_ptr,
  INT_1D ** iqibf_ptr,
  INT_1D ** irfibf_ptr,
  DOUBLE_1D ** del_ptr,
  DOUBLE_1D ** ds_ptr,
  DOUBLE_3D ** x_ptr);

INT_ ug3_tbfd
 (INT_ mmsg,
  INT_ *mtbf,
  INT_ *nbface,
  INT_ *nnode,
  INT_1D ** ibcibf,
  INT_1D ** idibf,
  INT_1D ** ierribf,
  INT_3D ** inibf,
  INT_1D ** iqibf,
  INT_1D ** irfibf,
  DOUBLE_1D ** del,
  DOUBLE_1D ** ds,
  DOUBLE_3D ** x)

{

/*
 * Create duplicate faces and nodes for transparent boundary surface faces.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_tbfd.c,v 1.25 2022/11/21 00:25:14 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_ ierr;

  ierr = ug3_tbfsplit (mmsg, nbface, nnode,
                       ibcibf, idibf, ierribf, inibf, iqibf, irfibf,
                       del, ds, x);

  if (ierr == 0)
    ierr = ug3_tbfd0 (mmsg, nbface, nnode, 
                      ibcibf, idibf, ierribf, inibf, iqibf, irfibf, 
                      del, ds, x);

  if (ierr == -1)
  {
    *mtbf = 0;

    ierr = 0;
  }
  else
    *mtbf = 1;

  return (ierr);

}

INT_ ug3_tbfsplit
 (INT_ mmsg,
  INT_ *nbface,
  INT_ *nnode,
  INT_1D ** ibcibf,
  INT_1D ** idibf,
  INT_1D ** ierribf,
  INT_3D ** inibf,
  INT_1D ** iqibf,
  INT_1D ** irfibf,
  DOUBLE_1D ** del,
  DOUBLE_1D ** ds,
  DOUBLE_3D ** x)

{

/*
 * Split transparent boundary surface faces that are isolated at a corner.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_tbfd.c,v 1.25 2022/11/21 00:25:14 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_1D *ierribf_ = NULL;
  INT_1D *iqibf_ = NULL;
  INT_1D *irfibf_ = NULL;

  DOUBLE_1D *del_ = NULL;
  DOUBLE_1D *ds_ = NULL;

  INT_3D *ibfibf = NULL;
  INT_1D *ibfichk = NULL;
  INT_1D *ibfin = NULL;
  INT_1D *libfin = NULL;
  INT_1D *mchkbf = NULL;

  INT_ ibface, ibface1, ibface2, ibface3, ierr, 
       jbface, jbface1, jbface2, jnode1, jnode2, jnode3, jnode4,
       kbface, knode, merr,
       nbfacei, nbfpnt, nbfpntd, nnodei, nsplit;
  INT_ mmultc = 1;
  INT_ mreorder = 0;

  double dc1d2;

  dc1d2 = 0.5;

  ierribf_ = (ierribf) ? *ierribf: NULL;
  iqibf_ = (iqibf) ? *iqibf: NULL;
  irfibf_ = (irfibf) ? *irfibf: NULL;

  del_ = (del) ? *del: NULL;
  ds_ = (ds) ? *ds: NULL;

  merr = (ierribf_) ? 1: 0;

  kbface = 0;

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
      ++kbface;
  }

  if (kbface == 0)
    return (-1);

  ierr = 0;

  nbfpntd = 0;

  ibfin = (INT_1D *) ug_malloc (&ierr, (nbfpntd+1) * sizeof (INT_1D));

  ibfibf = (INT_3D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_3D));
  ibfichk = (INT_1D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_1D));
  mchkbf = (INT_1D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_1D));

  libfin = (INT_1D *) ug_malloc (&ierr, ((*nnode)+2) * sizeof (INT_1D));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100326 : unable to allocate required memory ***");
    ierr = 100326;
  }

  if (ierr == 0)
    ierr = ug3_ibfin (*nbface, &nbfpntd, *nnode, &nbfpnt,
                      *inibf, &ibfin, libfin);

  if (ierr == 0)
    ierr = ug3_ibfibf (merr, mmsg, mmultc, mreorder, *nbface,
                       *ibcibf, ibfin, *inibf, irfibf_, NULL,
                       libfin, ibfichk, mchkbf, ibfibf, ierribf_,
                       *x);

  ug_free (ibfin);
  ug_free (ibfichk);
  ug_free (mchkbf);
  ug_free (libfin);

  if (ierr)
  {
    ug_free (ibfibf);
    ug_error_message ("*** ERROR 100327 : unable to allocate required memory ***");
    return (100327);
  }

  nsplit = 0;

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC || (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
    {
      ibface1 = ibfibf[ibface][0];
      ibface2 = ibfibf[ibface][1];
      ibface3 = ibfibf[ibface][2];

      if (ibface1)
      {
        if ((*idibf)[ibface] != (*idibf)[ibface1] &&
            ((*ibcibf)[ibface1] == -TRANSP_UG3_GBC ||
             (*ibcibf)[ibface1] == -TRANSP_INTRNL_UG3_GBC))
        ibface1 = 0;
      }

      if (ibface2)
      {
        if ((*idibf)[ibface] != (*idibf)[ibface2] &&
            ((*ibcibf)[ibface2] == -TRANSP_UG3_GBC ||
             (*ibcibf)[ibface2] == -TRANSP_INTRNL_UG3_GBC))
        ibface2 = 0;
      }

      if (ibface3)
      {
        if ((*idibf)[ibface] != (*idibf)[ibface3] &&
            ((*ibcibf)[ibface3] == -TRANSP_UG3_GBC ||
             (*ibcibf)[ibface3] == -TRANSP_INTRNL_UG3_GBC))
        ibface3 = 0;
      }

      ibfibf[ibface][0] = ibface1;
      ibfibf[ibface][1] = ibface2;
      ibfibf[ibface][2] = ibface3;

      if ((ibface1 && ibface2 == 0 && ibface3 == 0) ||
          (ibface2 && ibface1 == 0 && ibface3 == 0) ||
          (ibface3 && ibface1 == 0 && ibface2 == 0))
        ++nsplit;
    }
  }

  if (nsplit == 0)
  {
    ug_free (ibfibf);

    return (0);
  }

  nbfacei = *nbface;
  nnodei = *nnode;

  *nbface = *nbface + 2 * nsplit;
  *nnode = *nnode + nsplit;

  ierr = 0;

  *ibcibf = (INT_1D *) ug_realloc (&ierr, *ibcibf, ((*nbface)+1) * sizeof (INT_1D));
  ibfibf = (INT_3D *) ug_realloc (&ierr, ibfibf, ((*nbface)+1) * sizeof (INT_3D));
  *idibf = (INT_1D *) ug_realloc (&ierr, *idibf, ((*nbface)+1) * sizeof (INT_1D));
  if (ierribf_) {
    *ierribf = (INT_1D *) ug_realloc (&ierr, *ierribf, ((*nbface)+1) * sizeof (INT_1D));
    ierribf_ = *ierribf;
  }
  *inibf = (INT_3D *) ug_realloc (&ierr, *inibf, ((*nbface)+1) * sizeof (INT_3D));
  if (iqibf_) {
    *iqibf = (INT_1D *) ug_realloc (&ierr, *iqibf, ((*nbface)+1) * sizeof (INT_1D));
    iqibf_ = *iqibf;
  }
  if (irfibf_) {
    *irfibf = (INT_1D *) ug_realloc (&ierr, *irfibf, ((*nbface)+1) * sizeof (INT_1D));
    irfibf_ = *irfibf;
  }

  if (del_) {
    *del = (DOUBLE_1D *) ug_realloc (&ierr, *del, ((*nnode)+1) * sizeof (DOUBLE_1D));
    del_ = *del;
  }
  if (ds_) {
    *ds = (DOUBLE_1D *) ug_realloc (&ierr, *ds, ((*nnode)+1) * sizeof (DOUBLE_1D));
    ds_ = *ds;
  }
  *x = (DOUBLE_3D *) ug_realloc (&ierr, *x, ((*nnode)+1) * sizeof (DOUBLE_3D));

  if (ierr)
  {
    ug_free (ibfibf);
    ug_error_message ("*** ERROR 100328 : unable to allocate required memory ***");
    return (100328);
  }

  kbface = nbfacei;
  knode = nnodei;

  for (ibface = 1; ibface <= nbfacei; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
    {
      ibface1 = ibfibf[ibface][0];
      ibface2 = ibfibf[ibface][1];
      ibface3 = ibfibf[ibface][2];

      jnode1 = 0;

      if (ibface1 && ibface2 == 0 && ibface3 == 0)
      {
        jbface = ibface1;

        jnode1 = (*inibf)[ibface][0];
        jnode2 = (*inibf)[ibface][1];
        jnode3 = (*inibf)[ibface][2];
      }
      else if (ibface2 && ibface1 == 0 && ibface3 == 0)
      {
        jbface = ibface2;

        jnode1 = (*inibf)[ibface][1];
        jnode2 = (*inibf)[ibface][2];
        jnode3 = (*inibf)[ibface][0];
      }
      else if (ibface3 && ibface1 == 0 && ibface2 == 0)
      {
        jbface = ibface3;

        jnode1 = (*inibf)[ibface][2];
        jnode2 = (*inibf)[ibface][0];
        jnode3 = (*inibf)[ibface][1];
      }

      if (jnode1)
      {
        ++knode;

	if (del_) del_[knode] = dc1d2 * (del_[jnode2] + del_[jnode3]);
	if (ds_) ds_[knode] = dc1d2 * (ds_[jnode2] + ds_[jnode3]);

        (*x)[knode][0] = dc1d2 * ((*x)[jnode2][0] + (*x)[jnode3][0]);
        (*x)[knode][1] = dc1d2 * ((*x)[jnode2][1] + (*x)[jnode3][1]);
        (*x)[knode][2] = dc1d2 * ((*x)[jnode2][2] + (*x)[jnode3][2]);

        ++kbface;

        (*ibcibf)[kbface] = (*ibcibf)[ibface];

        (*idibf)[kbface] = (*idibf)[ibface];

        (*inibf)[ibface][0] = jnode1;
        (*inibf)[ibface][1] = jnode2;
        (*inibf)[ibface][2] = knode;

        (*inibf)[kbface][0] = jnode1;
        (*inibf)[kbface][1] = knode;
        (*inibf)[kbface][2] = jnode3;

        ibfibf[ibface][0] = jbface;
        ibfibf[ibface][1] = kbface;
        ibfibf[ibface][2] = 0;

        ibfibf[kbface][0] = kbface+1;
        ibfibf[kbface][1] = 0;
        ibfibf[kbface][2] = ibface;

        if (iqibf_)
        {
          iqibf_[ibface] = 0;
          iqibf_[kbface] = 0;
        }

        if (irfibf_)
        {
          irfibf_[ibface] = 7;
          irfibf_[kbface] = 7;
        }

        if (ibfibf[jbface][0] == ibface)
        {
          jnode4 = (*inibf)[jbface][0];

          jbface1 = ibfibf[jbface][1];
          jbface2 = ibfibf[jbface][2];
        }
        else if (ibfibf[jbface][1] == ibface)
        {
          jnode4 = (*inibf)[jbface][1];

          jbface1 = ibfibf[jbface][2];
          jbface2 = ibfibf[jbface][0];
        }
        else //if (ibfibf[jbface][2] == ibface)
        {
          jnode4 = (*inibf)[jbface][2];

          jbface1 = ibfibf[jbface][0];
          jbface2 = ibfibf[jbface][1];
        }

        ++kbface;

        (*ibcibf)[kbface] = (*ibcibf)[jbface];

        (*idibf)[kbface] = (*idibf)[jbface];

        (*inibf)[jbface][0] = jnode4;
        (*inibf)[jbface][1] = knode;
        (*inibf)[jbface][2] = jnode2;

        (*inibf)[kbface][0] = jnode4;
        (*inibf)[kbface][1] = jnode3;
        (*inibf)[kbface][2] = knode;

        ibfibf[jbface][0] = ibface;
        ibfibf[jbface][1] = jbface1;
        ibfibf[jbface][2] = kbface;

        ibfibf[kbface][0] = kbface-1;
        ibfibf[kbface][1] = jbface;
        ibfibf[kbface][2] = jbface2;

        if (jbface2)
        {
          if (ibfibf[jbface2][0] == jbface)
            ibfibf[jbface2][0] = kbface;
          else if (ibfibf[jbface2][1] == jbface)
            ibfibf[jbface2][1] = kbface;
          else //if (ibfibf[jbface2][2] == jbface)
            ibfibf[jbface2][2] = kbface;
        }

        if (iqibf_)
        {
          iqibf_[jbface] = 0;
          iqibf_[kbface] = 0;
        }

        if (irfibf_)
        {
          irfibf_[jbface] = 7;
          irfibf_[kbface] = 7;
        }
      }
    }
  }

  if (ierribf_)
    ug_set_int (nbfacei+1, *nbface, 0, ierribf_);

  ug_free (ibfibf);

  return (0);

}

INT_ ug3_tbfd0
 (INT_ mmsg,
  INT_ *nbface,
  INT_ *nnode,
  INT_1D ** ibcibf,
  INT_1D ** idibf,
  INT_1D ** ierribf,
  INT_3D ** inibf,
  INT_1D ** iqibf,
  INT_1D ** irfibf,
  DOUBLE_1D ** del,
  DOUBLE_1D ** ds,
  DOUBLE_3D ** x)

{

/*
 * Perform primary tasks required to create duplicate faces and nodes for
 * transparent boundary surface faces.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_tbfd.c,v 1.25 2022/11/21 00:25:14 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  CHAR_133 Text;

  INT_1D *ierribf_ = NULL;
  INT_1D *iqibf_ = NULL;
  INT_1D *irfibf_ = NULL;

  DOUBLE_1D *del_ = NULL;
  DOUBLE_1D *ds_ = NULL;

  INT_3D *ibfibf = NULL;
  INT_1D *ibfichk = NULL;
  INT_1D *ibfin = NULL;
  INT_1D *jtbfitbf = NULL;
  INT_1D *jnin = NULL;
  INT_1D *jnjn = NULL;
  INT_1D *libfin = NULL;
  INT_1D *mchkbf = NULL;
  INT_1D *nbfpe = NULL;

  INT_ ibe, ibface, ibface1, ibface2, ibface3, ibfn1, ibfn2, ibfpe,
       id, idmax, idmin, ierr, inode, inode1, inode2, inode3,
       jbe, jbface, jbfn1, jbfn2, jbfpe, jd, jnode, jnode1, jnode2, jnode3,
       kbface, loc, loc1, loc2, merr,
       n, n1, n2, nbfacei, nbfpnt, nbfpntd, nid, 
       nnodeb, nnodebi, nnodei, nset, ntbface, ntbnode, pass, region;
  INT_ mclosed = 1;
  INT_ mmultc = 1;
  INT_ mreorder = 0;

  ierribf_ = (ierribf) ? *ierribf: NULL;
  iqibf_ = (iqibf) ? *iqibf: NULL;
  irfibf_ = (irfibf) ? *irfibf: NULL;

  del_ = (del) ? *del: NULL;
  ds_ = (ds) ? *ds: NULL;

  merr = (ierribf_) ? 1: 0;

  // check for transparent boundary surface faces

  ntbface = 0;

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
      ++ntbface;
  }

  if (ntbface == 0)
    return (-1);

  if (mmsg == 2)
  {
    snprintf (Text, sizeof(Text), "Duplicate Transp BF: BL-Transp Faces   =%10i", (int) ntbface);
    ug_message (Text);
  }

  // set number of boundary nodes

  ug3_nnodeb (*nbface, 0, &nnodeb, *inibf, NULL);

  // allocate space for temporary arrays

  ierr = 0;

  nbfpntd = 0;

  ibfin = (INT_1D *) ug_malloc (&ierr, (nbfpntd+1) * sizeof (INT_1D));

  ibfibf = (INT_3D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_3D));
  ibfichk = (INT_1D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_1D));
  mchkbf = (INT_1D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_1D));

  jnin = (INT_1D *) ug_malloc (&ierr, ((*nnode)+1) * sizeof (INT_1D));
  libfin = (INT_1D *) ug_malloc (&ierr, ((*nnode)+2) * sizeof (INT_1D));
  nbfpe = (INT_1D *) ug_malloc (&ierr, (nnodeb+1) * sizeof (INT_1D));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100324 : unable to allocate required memory ***");
    ierr = 100324;
  }

  // determine face node map

  if (ierr == 0)
    ierr = ug3_ibfin (*nbface, &nbfpntd, *nnode, &nbfpnt,
                      *inibf, &ibfin, libfin);

  // determine face neighbor map

  if (ierr == 0)
    ierr = ug3_ibfibf (merr, mmsg, mmultc, mreorder, *nbface,
                       *ibcibf, ibfin, *inibf, irfibf_, NULL,
                       libfin, ibfichk, mchkbf, ibfibf, ierribf_,
                       *x);

  if (ierr)
  {
    ug_free (ibfin);
    ug_free (ibfibf);
    ug_free (ibfichk);
    ug_free (mchkbf);
    ug_free (jnin);
    ug_free (libfin);
    ug_free (nbfpe);
    return (ierr);
  }

  // initialize number of boundary faces per edge flag

  for (inode = 1; inode <= nnodeb; ++inode)
  {
    nbfpe[inode] = 0;
  }

  // set number of boundary faces per edge flag at nodes on boundary edges of
  // transparent boundary surface faces

  for (jbface = 1; jbface <= *nbface; ++jbface)
  {
    if ((*ibcibf)[jbface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[jbface] == -TRANSP_INTRNL_UG3_GBC)
    {
      jd = (*idibf)[jbface];

      for (jbe = 0; jbe <= 2; ++jbe)
      {
        jbfn1 = (jbe < 2) ? jbe+1 : 0;
        jbfn2 = (jbfn1 < 2) ? jbfn1+1 : 0;

        jnode1 = (*inibf)[jbface][jbfn1];
        jnode2 = (*inibf)[jbface][jbfn2];

        loc1 = libfin[jnode1];
        loc2 = libfin[jnode1+1] - 1;

        ibface = ibfibf[jbface][jbe];

        id = (ibface) ? (*idibf)[ibface]: jd-1;

        if (id != jd)
        {
          ibfpe = 1;
          jbfpe = 0;

          for (loc = loc1; loc <= loc2; ++loc)
          {
            ibface = ibfin[loc];

            if (ibface != jbface)
            {
              inode1 = (*inibf)[ibface][0];
              inode2 = (*inibf)[ibface][1];
              inode3 = (*inibf)[ibface][2];

              if ((inode1 == jnode2 && inode2 == jnode1) ||
                  (inode2 == jnode2 && inode3 == jnode1) ||
                  (inode3 == jnode2 && inode1 == jnode1) ||
                  (inode1 == jnode1 && inode2 == jnode2) ||
                  (inode2 == jnode1 && inode3 == jnode2) ||
                  (inode3 == jnode1 && inode1 == jnode2))
              {
                if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
                    (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
                  ++ibfpe;
                else
                  ++jbfpe;
              }
            }
          }

          n = ibfpe + jbfpe/2;

          if (nbfpe[jnode1] < n) nbfpe[jnode1] = n;
          if (nbfpe[jnode2] < n) nbfpe[jnode2] = n;
        }
      }
    }
  }

  // break connection between transparent boundary surface faces with different
  // boundary surface IDs if they are non-manifold (multiply connected)

  for (jbface = 1; jbface <= *nbface; ++jbface)
  {
    if ((*ibcibf)[jbface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[jbface] == -TRANSP_INTRNL_UG3_GBC)
    {
      jd = (*idibf)[jbface];

      for (jbe = 0; jbe <= 2; ++jbe)
      {
        ibface = ibfibf[jbface][jbe];

        if (ibface)
        {
          if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
              (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
          {
            id = (*idibf)[ibface];

            if (jd != id)
            {
              jbfn1 = (jbe < 2) ? jbe+1 : 0;
              jbfn2 = (jbfn1 < 2) ? jbfn1+1 : 0;

              jnode1 = (*inibf)[jbface][jbfn1];
              jnode2 = (*inibf)[jbface][jbfn2];

              if (nbfpe[jnode1] > 2 && nbfpe[jnode2] > 2)
                ibfibf[jbface][jbe] = 0;
            }
          }
        }
      }
    }
  }

  // initialize transparent node flag at each node to zero

  // want to end up with jnin=n where n implies the following
  // n>0  : duplicate n times
  // n=0  : don't duplicate
  // n=-1 : don't duplicate (edge node of a transparent surface)
  // n=-2 : don't duplicate (corner edge node)

  for (inode = 1; inode <= nnodeb; ++inode)
  {
    jnin[inode] = 0;
  }

  // determine number of transparent boundary surface faces and
  // set node flag to 1 at transparent boundary surface faces

  ntbface = 0;

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
    {
      ++ntbface;

      inode1 = (*inibf)[ibface][0];
      inode2 = (*inibf)[ibface][1];
      inode3 = (*inibf)[ibface][2];

      jnin[inode1] = 1;
      jnin[inode2] = 1;
      jnin[inode3] = 1;
    }
  }

  // set node flag to -1 at boundary edges of transparent boundary surface faces

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
    {
      ibface1 = ibfibf[ibface][0];
      ibface2 = ibfibf[ibface][1];
      ibface3 = ibfibf[ibface][2];

      inode1 = (*inibf)[ibface][0];
      inode2 = (*inibf)[ibface][1];
      inode3 = (*inibf)[ibface][2];

      if (ibface1 == 0)
      {
        jnin[inode2] = -1;
        jnin[inode3] = -1;
      }

      if (ibface2 == 0)
      {
        jnin[inode1] = -1;
        jnin[inode3] = -1;
      }

      if (ibface3 == 0)
      {
        jnin[inode1] = -1;
        jnin[inode2] = -1;
      }
    }
  }

  // set node flag to number of other transparent boundary faces attached to
  // boundary edges of other boundary surface faces

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
    {
      for (ibe = 0; ibe <= 2; ++ibe)
      {
        if (ibfibf[ibface][ibe] == 0)
        {
          ibfn1 = (ibe < 2) ? ibe+1 : 0;
          ibfn2 = (ibfn1 < 2) ? ibfn1+1 : 0;

          inode1 = (*inibf)[ibface][ibfn1];
          inode2 = (*inibf)[ibface][ibfn2];

          if (jnin[inode1] == -1 && nbfpe[inode1] >= 2) jnin[inode1] = nbfpe[inode1]-1;
          if (jnin[inode2] == -1 && nbfpe[inode2] >= 2) jnin[inode2] = nbfpe[inode2]-1;
        }
      }
    }
  }

  // set node flag to -2 at all corner edge nodes on transparent surfaces

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
    {
      ibface1 = ibfibf[ibface][0];
      ibface2 = ibfibf[ibface][1];
      ibface3 = ibfibf[ibface][2];

      inode1 = (*inibf)[ibface][0];
      inode2 = (*inibf)[ibface][1];
      inode3 = (*inibf)[ibface][2];

      jnode1 = jnin[inode1];
      jnode2 = jnin[inode2];
      jnode3 = jnin[inode3];

      if (ibface1 == 0)
      {
        if (jnode2 >= 1 && jnode3 == -1)
          jnin[inode2] = -2;
        else if (jnode2 == -1 && jnode3 >= 1)
          jnin[inode3] = -2;
      }

      if (ibface2 == 0)
      {
        if (jnode1 >= 1 && jnode3 == -1)
          jnin[inode1] = -2;
        else if (jnode1 == -1 && jnode3 >= 1)
          jnin[inode3] = -2;
      }

      if (ibface3 == 0)
      {
        if (jnode1 >= 1 && jnode2 == -1)
          jnin[inode1] = -2;
        else if (jnode1 == -1 && jnode2 >= 1)
          jnin[inode2] = -2;
      }
    }
  }

  // determine number of transparent boundary surface face nodes
  // and set node map for nodes that are duplicated

  jnode = nnodeb;

  for (inode = 1; inode <= nnodeb; ++inode)
  {
    n = jnin[inode];

    if (n >= 1)
    {
      jnode = jnode + n;

      jnin[inode] = n;
    }
    else
      jnin[inode] = 0;
  }

  ntbnode = jnode - nnodeb;

  if (mmsg == 2)
  {
    snprintf (Text, sizeof(Text), "Duplicate Transp BF: BL-Transp Nodes   =%10i", (int) ntbnode);
    ug_message (Text);
  }

  // save old and set new number of nodes, boundary nodes, and boundary faces

  nnodei = *nnode;
  nnodebi = nnodeb;
  nbfacei = *nbface;

  *nnode = (*nnode) + ntbnode;
  nnodeb = nnodeb + ntbnode;
  *nbface = (*nbface) + ntbface;

  // reallocate space for boundary surface and work arrays

  ierr = 0;

  *ibcibf = (INT_1D *) ug_realloc (&ierr, *ibcibf, ((*nbface)+1) * sizeof (INT_1D));
  *idibf = (INT_1D *) ug_realloc (&ierr, *idibf, ((*nbface)+1) * sizeof (INT_1D));
  if (ierribf_) {
    *ierribf = (INT_1D *) ug_realloc (&ierr, *ierribf, ((*nbface)+1) * sizeof (INT_1D));
    ierribf_ = *ierribf;
  }
  *inibf = (INT_3D *) ug_realloc (&ierr, *inibf, ((*nbface)+1) * sizeof (INT_3D));
  if (iqibf_) {
    *iqibf = (INT_1D *) ug_realloc (&ierr, *iqibf, ((*nbface)+1) * sizeof (INT_1D));
    iqibf_ = *iqibf;
  }
  if (irfibf_) {
    *irfibf = (INT_1D *) ug_realloc (&ierr, *irfibf, ((*nbface)+1) * sizeof (INT_1D));
    irfibf_ = *irfibf;
  }

  if (del_) {
    *del = (DOUBLE_1D *) ug_realloc (&ierr, *del, ((*nnode)+1) * sizeof (DOUBLE_1D));
    del_ = *del;
  }
  if (ds_) {
    *ds = (DOUBLE_1D *) ug_realloc (&ierr, *ds, ((*nnode)+1) * sizeof (DOUBLE_1D));
    ds_ = *ds;
  }
  *x = (DOUBLE_3D *) ug_realloc (&ierr, *x, ((*nnode)+1) * sizeof (DOUBLE_3D));

  ibfibf = (INT_3D *) ug_realloc (&ierr, ibfibf, ((*nbface)+1) * sizeof (INT_3D));
  jtbfitbf = (INT_1D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_1D));
  ibfichk = (INT_1D *) ug_realloc (&ierr, ibfichk, (MAX (nnodebi,*nbface)+1) * sizeof (INT_1D));
  mchkbf = (INT_1D *) ug_realloc (&ierr, mchkbf, ((*nbface)+1) * sizeof (INT_1D));

  jnin = (INT_1D *) ug_realloc (&ierr, jnin, ((*nnode)+1) * sizeof (INT_1D));
  jnjn = (INT_1D *) ug_malloc (&ierr, (nnodebi+1) * sizeof (INT_1D));
  libfin = (INT_1D *) ug_realloc (&ierr, libfin, ((*nnode)+2) * sizeof (INT_1D));
  nbfpe = (INT_1D *) ug_realloc (&ierr, nbfpe, ((*nnode)+1) * sizeof (INT_1D));

  if (ierr)
  {
    ug_free (ibfin);
    ug_free (ibfibf);
    ug_free (jtbfitbf);
    ug_free (ibfichk);
    ug_free (mchkbf);
    ug_free (jnin);
    ug_free (jnjn);
    ug_free (libfin);
    ug_free (nbfpe);
    ug_error_message ("*** ERROR 100325 : unable to allocate required memory ***");
    return (100325);
  }

  // initialize secondary node flag

  for (inode = 1; inode <= nnodebi; ++inode)
  {
    jnjn[inode] = 0;
  }

  // shift location of node based data in the field so that duplicate boundary
  // data can be inserted in proper order

  for (inode = nnodei; inode > nnodebi; --inode)
  {
    jnode = inode + ntbnode;

    if (del_)  del_[jnode] = del_[inode];
    if (ds_)  ds_[jnode] = ds_[inode];

    (*x)[jnode][0] = (*x)[inode][0];
    (*x)[jnode][1] = (*x)[inode][1];
    (*x)[jnode][2] = (*x)[inode][2];
  }

  // duplicate node based data

  jnode = nnodebi;

  for (inode = 1; inode <= nnodebi; ++inode)
  {
    n = jnin[inode];

    if (n)
    {
      if (n == 1 && nbfpe[inode] < 2)
        jnin[inode] = jnode+1;
      else
      {
        jnin[inode] = inode;
        jnjn[inode] = jnode+1;
      }

      for (loc = 1; loc <= n; ++loc)
      {
        ++jnode;

        nbfpe[jnode] = 0;

        if (del_)  del_[jnode] = del_[inode];
        if (ds_)  ds_[jnode] = ds_[inode];

        (*x)[jnode][0] = (*x)[inode][0];
        (*x)[jnode][1] = (*x)[inode][1];
        (*x)[jnode][2] = (*x)[inode][2];
      }
    }
    else
      jnin[inode] = inode;
  }

  // determine range of boundary surface face ID flags

  idmax = (*idibf)[1];
  idmin = (*idibf)[1];

  for (ibface = 2; ibface <= nbfacei; ++ibface)
  {
    idmax = MAX (idmax, (*idibf)[ibface]);
    idmin = MIN (idmin, (*idibf)[ibface]);
  }

  // set boundary surface face ID offset for duplicated boundary surface faces

  nid = idmax - idmin + 1;

  // duplicate boundary surface face data and
  // set map for duplicate and original boundary surface faces

  jbface = nbfacei;

  for (ibface = 1; ibface <= nbfacei; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC ||
        (*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
    {
      ++jbface;

      if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC)
        (*ibcibf)[jbface] = -TMP_TRANSP_UG3_GBC;
      else
        (*ibcibf)[jbface] = -TMP_TRANSP_INTRNL_UG3_GBC;

      (*idibf)[jbface] = (*idibf)[ibface] + nid;

      if (iqibf_)
        iqibf_[jbface] = iqibf_[ibface];

      if (irfibf_)
      {
        irfibf_[ibface] = 7;
        irfibf_[jbface] = 7;
      }

      jtbfitbf[ibface] = jbface;
      jtbfitbf[jbface] = ibface;

      inode1 = (*inibf)[ibface][0];
      inode2 = (*inibf)[ibface][1];
      inode3 = (*inibf)[ibface][2];

      (*inibf)[jbface][0] = jnin[inode1];
      (*inibf)[jbface][1] = jnin[inode3];
      (*inibf)[jbface][2] = jnin[inode2];
    }
    else
      jtbfitbf[ibface] = 0;
  }

  // switch BC on transparent boundary faces from TRANSP BC to SOLID BC

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    if ((*ibcibf)[ibface] == -TRANSP_UG3_GBC)
      (*ibcibf)[ibface] = -SOLID_TRANSP_UG3_GBC;
    else if ((*ibcibf)[ibface] == -TRANSP_INTRNL_UG3_GBC)
      (*ibcibf)[ibface] = -SOLID_TRANSP_INTRNL_UG3_GBC;
    else if ((*ibcibf)[ibface] == -TMP_TRANSP_UG3_GBC)
      (*ibcibf)[ibface] = -SOLID_TMP_TRANSP_UG3_GBC;
    else if ((*ibcibf)[ibface] == -TMP_TRANSP_INTRNL_UG3_GBC)
      (*ibcibf)[ibface] = -SOLID_TMP_TRANSP_INTRNL_UG3_GBC;
  }

  // initialize error flag for new faces

  if (ierribf_)
    ug_set_int (nbfacei+1, *nbface, 0, ierribf_);

  // determine new face node map

  ierr = ug3_ibfin (*nbface, &nbfpntd, *nnode, &nbfpnt,
                    *inibf, &ibfin, libfin);

  // determine new face neighbor map

  if (ierr == 0)
    ierr = ug3_ibfibf (merr, mmsg, mmultc, mreorder, *nbface,
                       *ibcibf, ibfin, *inibf, irfibf_, NULL,
                       libfin, ibfichk, mchkbf, ibfibf, ierribf_,
                       *x);

  // error return

  if (ierr)
  {
    ug_free (ibfin);
    ug_free (ibfibf);
    ug_free (jtbfitbf);
    ug_free (ibfichk);
    ug_free (mchkbf);
    ug_free (jnin);
    ug_free (jnjn);
    ug_free (libfin);
    ug_free (nbfpe);
    return (ierr);
  }

  // mark boundary surface faces that are multiply connected with transparent

  for (ibface = 1; ibface <= *nbface; ++ibface)
  {
    mchkbf[ibface] = 0;
  }

  for (ibface = 1; ibface <= nbfacei; ++ibface)
  {
    inode1 = (*inibf)[ibface][0];
    inode2 = (*inibf)[ibface][1];
    inode3 = (*inibf)[ibface][2];

    if (nbfpe[inode1] >= 2 || nbfpe[inode2] >= 2 || nbfpe[inode3] >= 2)
    {
      mchkbf[ibface] = 1;

      jbface = jtbfitbf[ibface];

      if (jbface)
        mchkbf[jbface] = 1;
    }
  }

  // correct boundary surface face node connectivity for boundary surface faces
  // with non-manifold (multiply connected) transparent face connections

  // make two passes and only use solid faces to start on the first pass

  for (pass = 1; pass <= 2; ++pass)
  {
    // loop over original faces

    nset = 0;

    for (jbface = 1; jbface <= nbfacei; ++jbface)
    {
      // only check faces with non-manifold connections
      // only check solid BL generating faces on the 1st pass

      if (mchkbf[jbface] && (jtbfitbf[jbface] == 0 || pass == 2))
      {
        // loop over face edges

        for (jbe = 0; jbe <= 2; ++jbe)
        {
          // set neighbor face

          kbface = ibfibf[jbface][jbe];

          // only check transparent neighbors with non-manifold connections

          if (mchkbf[kbface] && jtbfitbf[kbface])
          {
            // set nodes of face

            ibface = jbface;

            ibfn1 = (jbe < 2) ? jbe+1: 0;
            ibfn2 = (ibfn1 < 2) ? ibfn1+1: 0;
            
            inode1 = (*inibf)[ibface][ibfn1];
            inode2 = (*inibf)[ibface][ibfn2];

            // set number of nodes available to reset
            // set available index for node 1

            if (inode1 <= nnodebi)
            {
              n1 = nbfpe[inode1];
              n2 = nbfpe[inode2];

              jnode1 = jnjn[inode1];
            }
            else
            {
              n1 = 0;
              n2 = 0;

              jnode1 = 0;
            }

            // check that this is a non-manifold edge

            if (jnode1 && n1 >= 2 && n2)
            {
              // set starting face in region

              ibface = jtbfitbf[kbface];

              // reset number of boundary faces per edge flag (for future checking)

              nbfpe[inode1] = 1;

              // loop over regions to reset
              // skip region 1 and leave node 1 for region 1 at original index

              region = 2;

              while (region <= n1 && ibface > 0)
              {
                // set starting conditions for loop over connected faces

                ibface1 = ibface;

                ibfn1 = ((*inibf)[ibface][0] == inode1) ? 0:
                        ((*inibf)[ibface][1] == inode1) ? 1: 2;
                ibfn2 = (ibfn1 < 2) ? ibfn1+1: 0;

                (*inibf)[ibface][ibfn1] = jnode1;

                kbface = ibface;
                ibface = ibfibf[ibface][ibfn2];

                // loop over connected faces attached to node 1 and reset node

                loc = 0;

                while (loc <= *nbface && ibface != ibface1)
                {
                  ibfn1 = (ibfibf[ibface][0] == kbface) ? 1:
                          (ibfibf[ibface][1] == kbface) ? 2: 0;
                  ibfn2 = (ibfn1 < 2) ? ibfn1+1: 0;

                  (*inibf)[ibface][ibfn1] = jnode1;

                  kbface = ibface;
                  ibface = ibfibf[ibface][ibfn2];

                  ++loc;
                }

                // set number of boundary faces per edge flag (for future checking)

                nbfpe[jnode1] = 1;

                // reset starting face and available node index

                if (region < n1)
                {
                  ibface = ibface1;

                  kbface = ((*inibf)[ibface][0] == jnode1) ? ibfibf[ibface][2]:
                           ((*inibf)[ibface][1] == jnode1) ? ibfibf[ibface][0]:
                                                             ibfibf[ibface][1];
                  ibface = jtbfitbf[kbface];

                  if (ibface == 0)
                  {
                    ibface = kbface;

                    while (loc <= *nbface && kbface != ibface1 && jtbfitbf[kbface] == 0)
                    {
                      kbface = ((*inibf)[kbface][0] == jnode1) ? ibfibf[kbface][2]:
                               ((*inibf)[kbface][1] == jnode1) ? ibfibf[kbface][0]:
                                                                 ibfibf[kbface][1];

                      ++loc;
                    }

                    ibface = jtbfitbf[kbface];
                  }

                  jnode1 = jnode1 + 1;
                }

                ++region;
              }

              // set number of edge nodes set

              ++nset;
            }
          }
        }
      }
    }

    if (mmsg == 2)
    {
      snprintf (Text, sizeof(Text), "Duplicate Transp BF: Pass, E-Nodes Set =%10i%10i", (int) pass, (int) nset);
      ug_message (Text);
    }
  }

  // determine new face node map

  ierr = ug3_ibfin (*nbface, &nbfpntd, *nnode, &nbfpnt,
                    *inibf, &ibfin, libfin);

  // determine new face neighbor map

  if (ierr == 0)
    ierr = ug3_ibfibf (merr, mmsg, mmultc, mreorder, *nbface,
                       *ibcibf, ibfin, *inibf, irfibf_, NULL,
                       libfin, ibfichk, mchkbf, ibfibf, ierribf_,
                       *x);

  // check connectivity

  if (ierr == 0)
    ierr = ug3_chkb (mclosed, merr, mmsg, *nbface,
                     *ibcibf, ibfibf, *inibf, ierribf_);

  // free space for temporary arrays

  ug_free (ibfin);
  ug_free (ibfibf);
  ug_free (jtbfitbf);
  ug_free (ibfichk);
  ug_free (mchkbf);
  ug_free (jnin);
  ug_free (jnjn);
  ug_free (libfin);
  ug_free (nbfpe);

  return (ierr);

}
