#include "UG3_LIB.h"

/*
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_set_ibcibf.c,v 1.13 2023/11/19 22:33:29 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

INT_ ug3_set_ibcibf (
  INT_ mmsg,
  INT_ code,
  INT_ nbface,
  INT_1D * ibcibf,
  INT_1D * idibf,
  UG_Param_Struct *Param_Struct_Ptr)
{
  // Set BC flag at all faces using parameter structure data.

  //    code=3 for AFLR3 with ANBL3
  //    code=4 for AFLR4
  //    code=5 for BLOOM3
  //    code=44 for AFLR4 part of AFLR43
  //    code=43 for AFLR3 part of AFLR43

  CHAR_UG_MAX Text;
  CHAR_31 Label;

  INT_1D *bl_ids = NULL;
  INT_1D *bl_int_ids = NULL;
  INT_1D *fixed_bl_int_ids = NULL;
  INT_1D *ff_ids = NULL;
  INT_1D *std_ids = NULL;
  INT_1D *trnsp_ids = NULL;
  INT_1D *trnsp_bl_ids = NULL;
  INT_1D *trnsp_bl_int_ids = NULL;
  INT_1D *trnsp_intl_ids = NULL;
  INT_1D *trnsp_intl_bl_ids = NULL;
  INT_1D *trnsp_src_ids = NULL;
  INT_1D *BC_ids = NULL;
  INT_1D *BC_list = NULL;

  INT_ found, i, ibface, id, id_max, id_min, iget;
  INT_ n_bl = 0;
  INT_ n_bl_int = 0;
  INT_ n_fixed_bl_int = 0;
  INT_ n_ff = 0;
  INT_ n_std = 0;
  INT_ n_trnsp = 0;
  INT_ n_trnsp_bl = 0;
  INT_ n_trnsp_bl_int = 0;
  INT_ n_trnsp_intl = 0;
  INT_ n_trnsp_intl_bl = 0;
  INT_ n_trnsp_src = 0;
  INT_ n_BC = 0;
  INT_ n_list = 0;

  code = ug_abs (code);

  // get BC flag lists specified in parameter structure

  if (Param_Struct_Ptr) {

    iget = ug_get_int_param_vector ("BL_IDs", &n_bl, &bl_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("BL_Int_IDs", &n_bl_int, &bl_int_ids, Param_Struct_Ptr);
    if (code == 3 || code == 43)
      iget = ug_get_int_param_vector ("Fixed_BL_Int_IDs", &n_fixed_bl_int, &fixed_bl_int_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("FF_IDs", &n_ff, &ff_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("Std_IDs", &n_std, &std_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("Trnsp_IDs", &n_trnsp, &trnsp_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("Trnsp_BL_IDs", &n_trnsp_bl, &trnsp_bl_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("Trnsp_BL_Int_IDs", &n_trnsp_bl_int, &trnsp_bl_int_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("Trnsp_Intl_IDs", &n_trnsp_intl, &trnsp_intl_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("Trnsp_Intl_BL_IDs", &n_trnsp_intl_bl, &trnsp_intl_bl_ids, Param_Struct_Ptr);
    if (code == 3 || code == 43 || code == 4 || code == 44)
      iget = ug_get_int_param_vector ("Trnsp_Src_IDs", &n_trnsp_src, &trnsp_src_ids, Param_Struct_Ptr);

    iget = ug_get_int_param_vector ("BC_IDs", &n_BC, &BC_ids, Param_Struct_Ptr);
    iget = ug_get_int_param_vector ("Grid_BC_Flag", &n_list, &BC_list, Param_Struct_Ptr);

    if (n_BC != n_list)
      n_BC = 0;
  }

  // set BC flag for faces on BL generating solid surfaces

  if (n_bl) {

    for (i = 0; i < n_bl; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == bl_ids[i])
          ibcibf[ibface] = -STD_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on BL intersecting surfaces

  if (n_bl_int) {

    for (i = 0; i < n_bl_int; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == bl_int_ids[i])
          ibcibf[ibface] = BL_INT_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on BL fixed intersecting surfaces

  if (n_fixed_bl_int) {

    for (i = 0; i < n_fixed_bl_int; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == fixed_bl_int_ids[i])
          ibcibf[ibface] = FIXED_BL_INT_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on farfield surfaces

  if (n_ff) {

    for (i = 0; i < n_ff; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == ff_ids[i])
          ibcibf[ibface] = FARFIELD_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on solid surfaces

  if (n_std) {

    for (i = 0; i < n_std; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == std_ids[i])
          ibcibf[ibface] = STD_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on transparent surfaces

  if (n_trnsp) {

    for (i = 0; i < n_trnsp; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == trnsp_ids[i])
          ibcibf[ibface] = TRANSP_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on transparent BL generating surfaces

  if (n_trnsp_bl) {

    for (i = 0; i < n_trnsp_bl; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == trnsp_bl_ids[i])
          ibcibf[ibface] = -TRANSP_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on transparent BL intersecting surfaces

  if (n_trnsp_bl_int) {

    for (i = 0; i < n_trnsp_bl_int; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == trnsp_bl_int_ids[i])
          ibcibf[ibface] = TRANSP_BL_INT_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on transparent internal surfaces

  if (n_trnsp_intl) {

    for (i = 0; i < n_trnsp_intl; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == trnsp_intl_ids[i])
          ibcibf[ibface] = TRANSP_INTRNL_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on transparent internal BL generating surfaces

  if (n_trnsp_intl_bl) {

    for (i = 0; i < n_trnsp_intl_bl; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == trnsp_intl_bl_ids[i])
          ibcibf[ibface] = -TRANSP_INTRNL_UG3_GBC;
      }
    }
  }

  // set BC flag for faces on transparent source surfaces

  if (n_trnsp_src) {

    for (i = 0; i < n_trnsp_src; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == trnsp_src_ids[i])
          ibcibf[ibface] = TRANSP_SRC_UG3_GBC;
      }
    }
  }

  // set BC flag for specified faces

  if (n_BC) {

    for (i = 0; i < n_BC; i++) {

      for (ibface = 1; ibface <= nbface; ibface++) {
        if (idibf[ibface] == BC_ids[i])
          ibcibf[ibface] = BC_list[i];
      }
    }
  }

  // check for invalid grid BC flag

  for (ibface = 1; ibface <= nbface; ibface++) {
    if (CHK_VALID_UG3_GBC (ibcibf[ibface]) == 0 &&
        ibcibf[ibface] != UNDEF_UG3_GBC) {
      ug_error_message ("*** ERROR 370 : invalid surface grid BC flag found ***");
      return 370;
    }
  }

  // output surface definition information

  if (mmsg) {

    ug_message ("");
    ug_message ("UG3      : Boundary Conditions");
    ug_message ("");
    ug_message ("UG3      :  B-Face ID            Boundary Condition");

    // loop over definitions

    id_min = ug_min_int (1, nbface, idibf);
    id_max = ug_max_int (1, nbface, idibf);

    for (id = id_min; id <= id_max; id++) {

      found = 0;

      ibface = 1;

      do {

        if (id == idibf[ibface]) {

          found = 1;

               if (ibcibf[ibface] == FARFIELD_UG3_GBC)       strcpy (Label, "                      FarField");
          else if (ibcibf[ibface] == STD_UG3_GBC)            strcpy (Label, "                         Solid");
          else if (ibcibf[ibface] == -STD_UG3_GBC)           strcpy (Label, "                    Solid w/BL");
          else if (ibcibf[ibface] == BL_INT_UG3_GBC)         strcpy (Label, "               BL Intersecting");
          else if (ibcibf[ibface] == TRANSP_UG3_GBC)         strcpy (Label, "                   Transparent");
          else if (ibcibf[ibface] == -TRANSP_UG3_GBC)        strcpy (Label, "              Transparent w/BL");
          else if (ibcibf[ibface] == TRANSP_SRC_UG3_GBC)     strcpy (Label, "            Transparent Source");
          else if (ibcibf[ibface] == TRANSP_INTRNL_UG3_GBC)  strcpy (Label, "          Internal Transparent");
          else if (ibcibf[ibface] == -TRANSP_INTRNL_UG3_GBC) strcpy (Label, "     Internal Transparent w/BL");
          else if (ibcibf[ibface] == TRANSP_BL_INT_UG3_GBC)  strcpy (Label, "   Transparent BL Intersecting");
          else if (ibcibf[ibface] == FIXED_BL_INT_UG3_GBC)   strcpy (Label, "         Fixed BL Intersecting");
          else                                               strcpy (Label, "                   Not Defined");

          snprintf (Text, sizeof(Text), "UG3      : %10i%30s", (int) id, Label);
          ug_message (Text);
        }

        ibface++;
      }
      while (ibface <= nbface && found == 0);
    }
  }

  return 0;
}
