#include "UG3_LIB.h"

/*
 * Read and write grid data from a MESH file using external routines.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_read_write_mesh.c,v 1.16 2024/11/02 06:00:34 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

INT_ (*ext_ug3_read_mesh) (char File_Name[],
                           INT_ Read_Task_Flag,
                           INT_ *Number_of_Bnd_Nodes,
                           INT_ *Number_of_Nodes,
                           INT_ *Number_of_Bnd_Edges,
                           INT_ *Number_of_Surf_Quads,
                           INT_ *Number_of_Surf_Trias,
                           INT_ *Number_of_Vol_Hexs,
                           INT_ *Number_of_Vol_Pents_5,
                           INT_ *Number_of_Vol_Pents_6,
                           INT_ *Number_of_Vol_Tets,
                           INT_1D * Bnd_Edge_ID_Flag,
                           INT_2D * Bnd_Edge_Connectivity,
                           INT_1D * Surf_ID_Flag,
                           INT_4D * Surf_Quad_Connectivity,
                           INT_3D * Surf_Tria_Connectivity,
                           INT_8D * Vol_Hex_Connectivity,
                           INT_1D * Vol_ID_Flag,
                           INT_5D * Vol_Pent_5_Connectivity,
                           INT_6D * Vol_Pent_6_Connectivity,
                           INT_4D * Vol_Tet_Connectivity,
                           DOUBLE_3D * Coordinates) = 0;

INT_ (*ext_ug3_p2_read_mesh) (char File_Name[],
                              INT_ Read_Task_Flag,
                              INT_ *Number_of_Bnd_Nodes,
                              INT_ *Number_of_Nodes,
                              INT_ *Number_of_Surf_Quads,
                              INT_ *Number_of_Surf_Trias,
                              INT_ *Number_of_Vol_Hexs,
                              INT_ *Number_of_Vol_Pents_5,
                              INT_ *Number_of_Vol_Pents_6,
                              INT_ *Number_of_Vol_Tets,
                              INT_ *Order,
                              INT_1D * Surf_ID_Flag,
                              INT_4D * Surf_Quad_Connectivity,
                              INT_3D * Surf_Tria_Connectivity,
                              INT_8D * Vol_Hex_Connectivity,
                              INT_1D * Vol_ID_Flag,
                              INT_5D * Vol_Pent_5_Connectivity,
                              INT_6D * Vol_Pent_6_Connectivity,
                              INT_4D * Vol_Tet_Connectivity,
                              INT_5D * P2_Surf_Quad_Connectivity,
                              INT_3D * P2_Surf_Tria_Connectivity,
                              INT_19D * P2_Vol_Hex_Connectivity,
                              INT_9D * P2_Vol_Pent_5_Connectivity,
                              INT_12D * P2_Vol_Pent_6_Connectivity,
                              INT_6D * P2_Vol_Tet_Connectivity,
                              DOUBLE_3D * Coordinates) = 0;

INT_ (*ext_ug3_read_sol) (char File_Name[],
                          INT_ Read_Task_Flag,
                          INT_ NDim,
                          INT_ *Number_of_Nodes,
                          INT_ *Number_of_U_Scalars,
                          INT_ *Number_of_U_Vectors,
                          INT_ *Number_of_U_Metrics,
                          INT_ *Number_of_U_SymMetrics,
                          CHAR_21 * U_Scalar_Labels,
                          CHAR_21 * U_Vector_Labels,
                          CHAR_21 * U_Metric_Labels,
                          CHAR_21 * U_SymMetric_Labels,
                          DOUBLE_1D * U_Scalars,
                          DOUBLE_1D * U_Vectors,
                          DOUBLE_1D * U_Metrics,
                          DOUBLE_1D * U_SymMetrics) = 0;

INT_ (*ext_ug3_write_mesh) (char File_Name[],
                            INT_ Write_Task_Flag,
                            INT_ Number_of_Nodes,
                            INT_ Number_of_Bnd_Edges,
                            INT_ Number_of_Surf_Quads,
                            INT_ Number_of_Surf_Trias,
                            INT_ Number_of_Vol_Hexs,
                            INT_ Number_of_Vol_Pents_5,
                            INT_ Number_of_Vol_Pents_6,
                            INT_ Number_of_Vol_Tets,
                            INT_1D * Bnd_Edge_ID_Flag,
                            INT_2D * Bnd_Edge_Connectivity,
                            INT_1D * Surf_ID_Flag,
                            INT_4D * Surf_Quad_Connectivity,
                            INT_3D * Surf_Tria_Connectivity,
                            INT_8D * Vol_Hex_Connectivity,
                            INT_1D * Vol_ID_Flag,
                            INT_5D * Vol_Pent_5_Connectivity,
                            INT_6D * Vol_Pent_6_Connectivity,
                            INT_4D * Vol_Tet_Connectivity,
                            DOUBLE_3D * Coordinates) = 0;

INT_ (*ext_ug3_p2_write_mesh) (char File_Name[],
                        INT_ Write_Task_Flag,
                        INT_ Number_of_Nodes,
                        INT_ Number_of_Surf_Quads,
                        INT_ Number_of_Surf_Trias,
                        INT_ Number_of_Vol_Hexs,
                        INT_ Number_of_Vol_Pents_5,
                        INT_ Number_of_Vol_Pents_6,
                        INT_ Number_of_Vol_Tets,
                        INT_ Order,
                        INT_1D * Surf_ID_Flag,
                        INT_4D * Surf_Quad_Connectivity,
                        INT_3D * Surf_Tria_Connectivity,
                        INT_8D * Vol_Hex_Connectivity,
                        INT_1D * Vol_ID_Flag,
                        INT_5D * Vol_Pent_5_Connectivity,
                        INT_6D * Vol_Pent_6_Connectivity,
                        INT_4D * Vol_Tet_Connectivity,
                        INT_5D * P2_Surf_Quad_Connectivity,
                        INT_3D * P2_Surf_Tria_Connectivity,
                        INT_19D * P2_Vol_Hex_Connectivity,
                        INT_9D * P2_Vol_Pent_5_Connectivity,
                        INT_12D * P2_Vol_Pent_6_Connectivity,
                        INT_6D * P2_Vol_Tet_Connectivity,
                        DOUBLE_3D * Coordinates) = 0;

INT_ (*ext_ug3_write_sol) (char File_Name[],
                    INT_ NDim,
                    INT_ Number_of_Nodes,
                    INT_ Number_of_U_Scalars,
                    INT_ Number_of_U_Vectors,
                    INT_ Number_of_U_Metrics,
                    INT_ Number_of_U_SymMetrics,
                    DOUBLE_1D * U_Scalars,
                    DOUBLE_1D * U_Vectors,
                    DOUBLE_1D * U_Metrics,
                    DOUBLE_1D * U_SymMetrics) = 0;

void ug3_check_read_write_mesh_sol (INT_ *check_rmesh,
                                    INT_ *check_rsol,
                                    INT_ *check_wmesh,
                                    INT_ *check_wsol)
{
  *check_rmesh = (ext_ug3_read_mesh) ? 1: 0;
  *check_rsol = (ext_ug3_read_sol) ? 1: 0;
  *check_wmesh = (ext_ug3_write_mesh) ? 1: 0;
  *check_wsol = (ext_ug3_write_sol) ? 1: 0;
  return;
}

void ug3_p2_check_read_write_mesh_sol (INT_ *p2_check_rmesh,
                                       INT_ *p2_check_wmesh)
{
  *p2_check_rmesh = (ext_ug3_p2_read_mesh) ? 1: 0;
  *p2_check_wmesh = (ext_ug3_p2_write_mesh) ? 1: 0;
  return;
}

INT_ ug3_read_mesh (char File_Name[],
                    INT_ Read_Task_Flag,
                    INT_ *Number_of_Bnd_Nodes,
                    INT_ *Number_of_Nodes,
                    INT_ *Number_of_Bnd_Edges,
                    INT_ *Number_of_Surf_Quads,
                    INT_ *Number_of_Surf_Trias,
                    INT_ *Number_of_Vol_Hexs,
                    INT_ *Number_of_Vol_Pents_5,
                    INT_ *Number_of_Vol_Pents_6,
                    INT_ *Number_of_Vol_Tets,
                    INT_1D * Bnd_Edge_ID_Flag,
                    INT_2D * Bnd_Edge_Connectivity,
                    INT_1D * Surf_ID_Flag,
                    INT_4D * Surf_Quad_Connectivity,
                    INT_3D * Surf_Tria_Connectivity,
                    INT_8D * Vol_Hex_Connectivity,
                    INT_1D * Vol_ID_Flag,
                    INT_5D * Vol_Pent_5_Connectivity,
                    INT_6D * Vol_Pent_6_Connectivity,
                    INT_4D * Vol_Tet_Connectivity,
                    DOUBLE_3D * Coordinates)
{
  INT_ Error_Flag;

  if (ext_ug3_read_mesh)
    Error_Flag = (*ext_ug3_read_mesh) (File_Name,
                                       Read_Task_Flag,
                                       Number_of_Bnd_Nodes,
                                       Number_of_Nodes,
                                       Number_of_Bnd_Edges,
                                       Number_of_Surf_Quads,
                                       Number_of_Surf_Trias,
                                       Number_of_Vol_Hexs,
                                       Number_of_Vol_Pents_5,
                                       Number_of_Vol_Pents_6,
                                       Number_of_Vol_Tets,
                                       Bnd_Edge_ID_Flag,
                                       Bnd_Edge_Connectivity,
                                       Surf_ID_Flag,
                                       Surf_Quad_Connectivity,
                                       Surf_Tria_Connectivity,
                                       Vol_Hex_Connectivity,
                                       Vol_ID_Flag,
                                       Vol_Pent_5_Connectivity,
                                       Vol_Pent_6_Connectivity,
                                       Vol_Tet_Connectivity,
                                       Coordinates);
  else
  {
    ug_error_message ("*** ERROR : MESH grid file type is not available ***");
    return (1);
  }

  return (Error_Flag);
}

INT_ ug3_p2_read_mesh (char File_Name[],
                       INT_ Read_Task_Flag,
                       INT_ *Number_of_Bnd_Nodes,
                       INT_ *Number_of_Nodes,
                       INT_ *Number_of_Surf_Quads,
                       INT_ *Number_of_Surf_Trias,
                       INT_ *Number_of_Vol_Hexs,
                       INT_ *Number_of_Vol_Pents_5,
                       INT_ *Number_of_Vol_Pents_6,
                       INT_ *Number_of_Vol_Tets,
                       INT_ *Order,
                       INT_1D * Surf_ID_Flag,
                       INT_4D * Surf_Quad_Connectivity,
                       INT_3D * Surf_Tria_Connectivity,
                       INT_8D * Vol_Hex_Connectivity,
                       INT_1D * Vol_ID_Flag,
                       INT_5D * Vol_Pent_5_Connectivity,
                       INT_6D * Vol_Pent_6_Connectivity,
                       INT_4D * Vol_Tet_Connectivity,
                       INT_5D * P2_Surf_Quad_Connectivity,
                       INT_3D * P2_Surf_Tria_Connectivity,
                       INT_19D * P2_Vol_Hex_Connectivity,
                       INT_9D * P2_Vol_Pent_5_Connectivity,
                       INT_12D * P2_Vol_Pent_6_Connectivity,
                       INT_6D * P2_Vol_Tet_Connectivity,
                       DOUBLE_3D * Coordinates)
{
  INT_ Error_Flag;

  if (ext_ug3_p2_read_mesh)
    Error_Flag = (*ext_ug3_p2_read_mesh) (File_Name,
                                          Read_Task_Flag,
                                          Number_of_Bnd_Nodes,
                                          Number_of_Nodes,
                                          Number_of_Surf_Quads,
                                          Number_of_Surf_Trias,
                                          Number_of_Vol_Hexs,
                                          Number_of_Vol_Pents_5,
                                          Number_of_Vol_Pents_6,
                                          Number_of_Vol_Tets,
                                          Order,
                                          Surf_ID_Flag,
                                          Surf_Quad_Connectivity,
                                          Surf_Tria_Connectivity,
                                          Vol_Hex_Connectivity,
                                          Vol_ID_Flag,
                                          Vol_Pent_5_Connectivity,
                                          Vol_Pent_6_Connectivity,
                                          Vol_Tet_Connectivity,
                                          P2_Surf_Quad_Connectivity,
                                          P2_Surf_Tria_Connectivity,
                                          P2_Vol_Hex_Connectivity,
                                          P2_Vol_Pent_5_Connectivity,
                                          P2_Vol_Pent_6_Connectivity,
                                          P2_Vol_Tet_Connectivity,
                                          Coordinates);
  else
  {
    ug_error_message ("*** ERROR : MESH grid file type is not available ***");
    return (1);
  }

  return (Error_Flag);
}

INT_ ug3_read_sol (char File_Name[],
                   INT_ Read_Task_Flag,
                   INT_ NDim,
                   INT_ *Number_of_Nodes,
                   INT_ *Number_of_U_Scalars,
                   INT_ *Number_of_U_Vectors,
                   INT_ *Number_of_U_Metrics,
                   INT_ *Number_of_U_SymMetrics,
                   CHAR_21 * U_Scalar_Labels,
                   CHAR_21 * U_Vector_Labels,
                   CHAR_21 * U_Metric_Labels,
                   CHAR_21 * U_SymMetric_Labels,
                   DOUBLE_1D * U_Scalars,
                   DOUBLE_1D * U_Vectors,
                   DOUBLE_1D * U_Metrics,
                   DOUBLE_1D * U_SymMetrics)
{
  INT_ Error_Flag;

  if (ext_ug3_read_sol)
    Error_Flag = (*ext_ug3_read_sol) (File_Name,
                                      Read_Task_Flag,
                                      NDim,
                                      Number_of_Nodes,
                                      Number_of_U_Scalars,
                                      Number_of_U_Vectors,
                                      Number_of_U_Metrics,
                                      Number_of_U_SymMetrics,
                                      U_Scalar_Labels,
                                      U_Vector_Labels,
                                      U_Metric_Labels,
                                      U_SymMetric_Labels,
                                      U_Scalars,
                                      U_Vectors,
                                      U_Metrics,
                                      U_SymMetrics);
  else
  {
    ug_error_message ("*** ERROR : MESH function file type is not available ***");
    return (1);
  }

  return (Error_Flag);
}

INT_ ug3_write_mesh (char File_Name[],
                     INT_ Write_Task_Flag,
                     INT_ Number_of_Nodes,
                     INT_ Number_of_Bnd_Edges,
                     INT_ Number_of_Surf_Quads,
                     INT_ Number_of_Surf_Trias,
                     INT_ Number_of_Vol_Hexs,
                     INT_ Number_of_Vol_Pents_5,
                     INT_ Number_of_Vol_Pents_6,
                     INT_ Number_of_Vol_Tets,
                     INT_1D * Bnd_Edge_ID_Flag,
                     INT_2D * Bnd_Edge_Connectivity,
                     INT_1D * Surf_ID_Flag,
                     INT_4D * Surf_Quad_Connectivity,
                     INT_3D * Surf_Tria_Connectivity,
                     INT_8D * Vol_Hex_Connectivity,
                     INT_1D * Vol_ID_Flag,
                     INT_5D * Vol_Pent_5_Connectivity,
                     INT_6D * Vol_Pent_6_Connectivity,
                     INT_4D * Vol_Tet_Connectivity,
                     DOUBLE_3D * Coordinates)
{
  INT_ Error_Flag;

  if (ext_ug3_write_mesh)
    Error_Flag = (*ext_ug3_write_mesh) (File_Name,
                                        Write_Task_Flag,
                                        Number_of_Nodes,
                                        Number_of_Bnd_Edges,
                                        Number_of_Surf_Quads,
                                        Number_of_Surf_Trias,
                                        Number_of_Vol_Hexs,
                                        Number_of_Vol_Pents_5,
                                        Number_of_Vol_Pents_6,
                                        Number_of_Vol_Tets,
                                        Bnd_Edge_ID_Flag,
                                        Bnd_Edge_Connectivity,
                                        Surf_ID_Flag,
                                        Surf_Quad_Connectivity,
                                        Surf_Tria_Connectivity,
                                        Vol_Hex_Connectivity,
                                        Vol_ID_Flag,
                                        Vol_Pent_5_Connectivity,
                                        Vol_Pent_6_Connectivity,
                                        Vol_Tet_Connectivity,
                                        Coordinates);
  else
  {
    ug_error_message ("*** ERROR : MESH grid file type is not available ***");
    return (1);
  }

  return (Error_Flag);
}

INT_ ug3_p2_write_mesh (char File_Name[],
                        INT_ Write_Task_Flag,
                        INT_ Number_of_Nodes,
                        INT_ Number_of_Surf_Quads,
                        INT_ Number_of_Surf_Trias,
                        INT_ Number_of_Vol_Hexs,
                        INT_ Number_of_Vol_Pents_5,
                        INT_ Number_of_Vol_Pents_6,
                        INT_ Number_of_Vol_Tets,
                        INT_ Order,
                        INT_1D * Surf_ID_Flag,
                        INT_4D * Surf_Quad_Connectivity,
                        INT_3D * Surf_Tria_Connectivity,
                        INT_8D * Vol_Hex_Connectivity,
                        INT_1D * Vol_ID_Flag,
                        INT_5D * Vol_Pent_5_Connectivity,
                        INT_6D * Vol_Pent_6_Connectivity,
                        INT_4D * Vol_Tet_Connectivity,
                        INT_5D * P2_Surf_Quad_Connectivity,
                        INT_3D * P2_Surf_Tria_Connectivity,
                        INT_19D * P2_Vol_Hex_Connectivity,
                        INT_9D * P2_Vol_Pent_5_Connectivity,
                        INT_12D * P2_Vol_Pent_6_Connectivity,
                        INT_6D * P2_Vol_Tet_Connectivity,
                        DOUBLE_3D * Coordinates)
{
  INT_ Error_Flag;

  if (ext_ug3_write_mesh)
    Error_Flag = (*ext_ug3_p2_write_mesh) (File_Name,
                                           Write_Task_Flag,
                                           Number_of_Nodes,
                                           Number_of_Surf_Quads,
                                           Number_of_Surf_Trias,
                                           Number_of_Vol_Hexs,
                                           Number_of_Vol_Pents_5,
                                           Number_of_Vol_Pents_6,
                                           Number_of_Vol_Tets,
                                           Order,
                                           Surf_ID_Flag,
                                           Surf_Quad_Connectivity,
                                           Surf_Tria_Connectivity,
                                           Vol_Hex_Connectivity,
                                           Vol_ID_Flag,
                                           Vol_Pent_5_Connectivity,
                                           Vol_Pent_6_Connectivity,
                                           Vol_Tet_Connectivity,
                                           P2_Surf_Quad_Connectivity,
                                           P2_Surf_Tria_Connectivity,
                                           P2_Vol_Hex_Connectivity,
                                           P2_Vol_Pent_5_Connectivity,
                                           P2_Vol_Pent_6_Connectivity,
                                           P2_Vol_Tet_Connectivity,
                                           Coordinates);
  else
  {
    ug_error_message ("*** ERROR : MESH grid file type is not available ***");
    return (1);
  }

  return (Error_Flag);
}

INT_ ug3_write_sol (char File_Name[],
                    INT_ NDim,
                    INT_ Number_of_Nodes,
                    INT_ Number_of_U_Scalars,
                    INT_ Number_of_U_Vectors,
                    INT_ Number_of_U_Metrics,
                    INT_ Number_of_U_SymMetrics,
                    DOUBLE_1D * U_Scalars,
                    DOUBLE_1D * U_Vectors,
                    DOUBLE_1D * U_Metrics,
                    DOUBLE_1D * U_SymMetrics)
{
  INT_ Error_Flag;

  if (ext_ug3_write_sol)
    Error_Flag = (*ext_ug3_write_sol) (File_Name,
                                       NDim,
                                       Number_of_Nodes,
                                       Number_of_U_Scalars,
                                       Number_of_U_Vectors,
                                       Number_of_U_Metrics,
                                       Number_of_U_SymMetrics,
                                       U_Scalars,
                                       U_Vectors,
                                       U_Metrics,
                                       U_SymMetrics);
  else
  {
    ug_error_message ("*** ERROR : MESH function file type is not available ***");
    return (1);
  }

  return (Error_Flag);
}

// register names of external mesh library routines

void ug3_mesh_register_read_grid (INT_ (*ext_routine) (char File_Name[],
                                                       INT_ Read_Task_Flag,
                                                       INT_ *Number_of_Bnd_Nodes,
                                                       INT_ *Number_of_Nodes,
                                                       INT_ *Number_of_Bnd_Edges,
                                                       INT_ *Number_of_Surf_Quads,
                                                       INT_ *Number_of_Surf_Trias,
                                                       INT_ *Number_of_Vol_Hexs,
                                                       INT_ *Number_of_Vol_Pents_5,
                                                       INT_ *Number_of_Vol_Pents_6,
                                                       INT_ *Number_of_Vol_Tets,
                                                       INT_1D * Bnd_Edge_ID_Flag,
                                                       INT_2D * Bnd_Edge_Connectivity,
                                                       INT_1D * Surf_ID_Flag,
                                                       INT_4D * Surf_Quad_Connectivity,
                                                       INT_3D * Surf_Tria_Connectivity,
                                                       INT_8D * Vol_Hex_Connectivity,
                                                       INT_1D * Vol_ID_Flag,
                                                       INT_5D * Vol_Pent_5_Connectivity,
                                                       INT_6D * Vol_Pent_6_Connectivity,
                                                       INT_4D * Vol_Tet_Connectivity,
                                                       DOUBLE_3D * Coordinates))
{
  ext_ug3_read_mesh = ext_routine;
  return;
}

void ug3_p2_mesh_register_read_grid (INT_ (*ext_routine) (char File_Name[],
                                                          INT_ Read_Task_Flag,
                                                          INT_ *Number_of_Bnd_Nodes,
                                                          INT_ *Number_of_Nodes,
                                                          INT_ *Number_of_Surf_Quads,
                                                          INT_ *Number_of_Surf_Trias,
                                                          INT_ *Number_of_Vol_Hexs,
                                                          INT_ *Number_of_Vol_Pents_5,
                                                          INT_ *Number_of_Vol_Pents_6,
                                                          INT_ *Number_of_Vol_Tets,
                                                          INT_ *Order,
                                                          INT_1D * Surf_ID_Flag,
                                                          INT_4D * Surf_Quad_Connectivity,
                                                          INT_3D * Surf_Tria_Connectivity,
                                                          INT_8D * Vol_Hex_Connectivity,
                                                          INT_1D * Vol_ID_Flag,
                                                          INT_5D * Vol_Pent_5_Connectivity,
                                                          INT_6D * Vol_Pent_6_Connectivity,
                                                          INT_4D * Vol_Tet_Connectivity,
                                                          INT_5D * P2_Surf_Quad_Connectivity,
                                                          INT_3D * P2_Surf_Tria_Connectivity,
                                                          INT_19D * P2_Vol_Hex_Connectivity,
                                                          INT_9D * P2_Vol_Pent_5_Connectivity,
                                                          INT_12D * P2_Vol_Pent_6_Connectivity,
                                                          INT_6D * P2_Vol_Tet_Connectivity,
                                                          DOUBLE_3D * Coordinates))
{
  ext_ug3_p2_read_mesh = ext_routine;
  return;
}

void ug3_mesh_register_read_func (INT_ (*ext_routine) (char File_Name[],
                                                       INT_ Read_Task_Flag,
                                                       INT_ NDim,
                                                       INT_ *Number_of_Nodes,
                                                       INT_ *Number_of_U_Scalars,
                                                       INT_ *Number_of_U_Vectors,
                                                       INT_ *Number_of_U_Metrics,
                                                       INT_ *Number_of_U_SymMetrics,
                                                       CHAR_21 * U_Scalar_Labels,
                                                       CHAR_21 * U_Vector_Labels,
                                                       CHAR_21 * U_Metric_Labels,
                                                       CHAR_21 * U_SymMetric_Labels,
                                                       DOUBLE_1D * U_Scalars,
                                                       DOUBLE_1D * U_Vectors,
                                                       DOUBLE_1D * U_Metrics,
                                                       DOUBLE_1D * U_SymMetrics))
{
  ext_ug3_read_sol = ext_routine;
  return;
}

void ug3_mesh_register_write_grid (INT_ (*ext_routine) (char File_Name[],
                                                        INT_ Write_Task_Flag,
                                                        INT_ Number_of_Nodes,
                                                        INT_ Number_of_Bnd_Edges,
                                                        INT_ Number_of_Surf_Quads,
                                                        INT_ Number_of_Surf_Trias,
                                                        INT_ Number_of_Vol_Hexs,
                                                        INT_ Number_of_Vol_Pents_5,
                                                        INT_ Number_of_Vol_Pents_6,
                                                        INT_ Number_of_Vol_Tets,
                                                        INT_1D * Bnd_Edge_ID_Flag,
                                                        INT_2D * Bnd_Edge_Connectivity,
                                                        INT_1D * Surf_ID_Flag,
                                                        INT_4D * Surf_Quad_Connectivity,
                                                        INT_3D * Surf_Tria_Connectivity,
                                                        INT_8D * Vol_Hex_Connectivity,
                                                        INT_1D * Vol_ID_Flag,
                                                        INT_5D * Vol_Pent_5_Connectivity,
                                                        INT_6D * Vol_Pent_6_Connectivity,
                                                        INT_4D * Vol_Tet_Connectivity,
                                                        DOUBLE_3D * Coordinates))
{
  ext_ug3_write_mesh = ext_routine;
  return;
}

void ug3_p2_mesh_register_write_grid (INT_ (*ext_routine) (char File_Name[],
                                                           INT_ Write_Task_Flag,
                                                           INT_ Number_of_Nodes,
                                                           INT_ Number_of_Surf_Quads,
                                                           INT_ Number_of_Surf_Trias,
                                                           INT_ Number_of_Vol_Hexs,
                                                           INT_ Number_of_Vol_Pents_5,
                                                           INT_ Number_of_Vol_Pents_6,
                                                           INT_ Number_of_Vol_Tets,
                                                           INT_ Order,
                                                           INT_1D * Surf_ID_Flag,
                                                           INT_4D * Surf_Quad_Connectivity,
                                                           INT_3D * Surf_Tria_Connectivity,
                                                           INT_8D * Vol_Hex_Connectivity,
                                                           INT_1D * Vol_ID_Flag,
                                                           INT_5D * Vol_Pent_5_Connectivity,
                                                           INT_6D * Vol_Pent_6_Connectivity,
                                                           INT_4D * Vol_Tet_Connectivity,
                                                           INT_5D * P2_Surf_Quad_Connectivity,
                                                           INT_3D * P2_Surf_Tria_Connectivity,
                                                           INT_19D * P2_Vol_Hex_Connectivity,
                                                           INT_9D * P2_Vol_Pent_5_Connectivity,
                                                           INT_12D * P2_Vol_Pent_6_Connectivity,
                                                           INT_6D * P2_Vol_Tet_Connectivity,
                                                           DOUBLE_3D * Coordinates))
{
  ext_ug3_p2_write_mesh = ext_routine;
  return;
}

void ug3_mesh_register_write_func (INT_ (*ext_routine) (char File_Name[],
                                                        INT_ NDim,
                                                        INT_ Number_of_Nodes,
                                                        INT_ Number_of_U_Scalars,
                                                        INT_ Number_of_U_Vectors,
                                                        INT_ Number_of_U_Metrics,
                                                        INT_ Number_of_U_SymMetrics,
                                                        DOUBLE_1D * U_Scalars,
                                                        DOUBLE_1D * U_Vectors,
                                                        DOUBLE_1D * U_Metrics,
                                                        DOUBLE_1D * U_SymMetrics))
{
  ext_ug3_write_sol = ext_routine;
  return;
}
