#include "UG3_LIB.h"

static INT_ ug3_read_sfunc_func
 (FILE * Func_File,
  INT_ File_Format,
  INT_ n,
  INT_ Number_of_Nodes,
  INT_ Number_of_U_Funcs,
  DOUBLE_1D * U_Funcs);

static INT_ ug3_read_sfunc_label
 (FILE * Func_File,
  INT_ File_Format,
  INT_ Read_Task_Flag,
  INT_ Number_of_U_Funcs,
  CHAR_21 * U_Func_Labels);

INT_ ug3_read_sfunc
 (FILE * Func_File,
  INT_ File_Format,
  INT_ Read_Task_Flag,
  INT_ NDim,
  INT_ *Number_of_Nodes,
  INT_ *Number_of_U_Scalars,
  INT_ *Number_of_U_Vectors,
  INT_ *Number_of_U_Matrixes,
  INT_ *Number_of_U_Metrics,
  INT_ SFUNC_Flag,
  CHAR_21 * U_Scalar_Labels,
  CHAR_21 * U_Vector_Labels,
  CHAR_21 * U_Matrix_Labels,
  CHAR_21 * U_Metric_Labels,
  DOUBLE_1D * U_Scalars,
  DOUBLE_1D * U_Vectors,
  DOUBLE_1D * U_Matrixes,
  DOUBLE_1D * U_Metrics)
{

/*
 * Read function data from a SFUNC/UFUNC function file.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_read_sfunc.c,v 1.4 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_ ierr, n, Mode_Flag, 
       Number_of_Bytes, Number_of_Read_Items, Read_Flag, Special_Read_Items;

  int i1 = 0;
  int i2 = 0;
  int i3 = 0;

  Mode_Flag = ug_get_file_stream_mode_flag (Func_File);

  if (File_Format == UG_FIO_FORMATTED)
  {
    *Number_of_Nodes = 0;
    *Number_of_U_Scalars = 0;
    *Number_of_U_Vectors = 0;
    *Number_of_U_Matrixes = 0;
    *Number_of_U_Metrics = 0;

    Read_Flag = fscanf (Func_File, "%i %i %i", &i1, &i2, &i3);

    *Number_of_Nodes = (INT_) i1;
    *Number_of_U_Scalars = (INT_) i2;
    *Number_of_U_Vectors = (INT_) i3;

    if (SFUNC_Flag)
    {
      Read_Flag = fscanf (Func_File, " %i %i", &i1, &i2);

      *Number_of_U_Matrixes = (INT_) i1;
      *Number_of_U_Metrics = (INT_) i2;
    }

    if (Read_Flag == EOF)
    {
      ug_error_message ("*** ERROR : error reading SFUNC function file ***");
      return (1);
    }
  }
  else if (Read_Task_Flag == 1 || Mode_Flag == UG_FIO_STANDARD_FILE_MODE)
  {
    *Number_of_Nodes = 0;
    *Number_of_U_Scalars = 0;
    *Number_of_U_Vectors = 0;
    *Number_of_U_Matrixes = 0;
    *Number_of_U_Metrics = 0;

    Special_Read_Items = ((File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE) ? 1 : 0);

    Read_Flag = 0;
    
    if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
      Read_Flag = Read_Flag + ug_fread (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

    Read_Flag = Read_Flag + ug_fread (Number_of_Nodes, sizeof (INT_), 1, Func_File);
    Read_Flag = Read_Flag + ug_fread (Number_of_U_Scalars, sizeof (INT_), 1, Func_File);
    Read_Flag = Read_Flag + ug_fread (Number_of_U_Vectors, sizeof (INT_), 1, Func_File);

    if (SFUNC_Flag)
    {
      Read_Flag = Read_Flag + ug_fread (Number_of_U_Matrixes, sizeof (INT_), 1, Func_File);
      Read_Flag = Read_Flag + ug_fread (Number_of_U_Metrics, sizeof (INT_), 1, Func_File);
    }

    if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
      Read_Flag = Read_Flag + ug_fread (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

    Number_of_Read_Items = 3 + 2*SFUNC_Flag + Special_Read_Items + Special_Read_Items;

    if (Read_Flag != Number_of_Read_Items)
    {
      ug_error_message ("*** ERROR : error reading SFUNC function file ***");
      return (1);
    }
  }

  if (Read_Task_Flag == 2 ||
      (Read_Task_Flag >= 2 && Mode_Flag == UG_FIO_STANDARD_FILE_MODE))
  {
    ierr = ug3_read_sfunc_label (Func_File, File_Format, Read_Task_Flag,
                                   *Number_of_U_Scalars,
                                   U_Scalar_Labels);
  
    if (ierr == 0)
      ierr = ug3_read_sfunc_label (Func_File, File_Format, Read_Task_Flag,
                                     *Number_of_U_Vectors,
                                     U_Vector_Labels);

    if (SFUNC_Flag)
    {
      if (ierr == 0)
        ierr = ug3_read_sfunc_label (Func_File, File_Format, Read_Task_Flag,
                                       *Number_of_U_Matrixes,
                                       U_Matrix_Labels);

      if (ierr == 0)
        ierr = ug3_read_sfunc_label (Func_File, File_Format, Read_Task_Flag,
                                       *Number_of_U_Metrics,
                                       U_Metric_Labels);
    }
  }

  if (Read_Task_Flag == 3)
  {
    n = 1;

    ierr = ug3_read_sfunc_func (Func_File, File_Format, n,
                                  *Number_of_Nodes, *Number_of_U_Scalars,
                                  U_Scalars);

    n = NDim;

    if (ierr == 0)
      ierr = ug3_read_sfunc_func (Func_File, File_Format, n,
                                    *Number_of_Nodes, *Number_of_U_Vectors,
                                    U_Vectors);

    if (SFUNC_Flag)
    {
      n = NDim*NDim;

      if (ierr == 0)
        ierr = ug3_read_sfunc_func (Func_File, File_Format, n,
                                      *Number_of_Nodes, *Number_of_U_Matrixes,
                                      U_Matrixes);

      n = 3*(NDim-1);

      if (ierr == 0)
        ierr = ug3_read_sfunc_func (Func_File, File_Format, n,
                                      *Number_of_Nodes, *Number_of_U_Metrics,
                                      U_Metrics);
    }
  }

  return (0);

}

INT_ ug3_read_sfunc_func
 (FILE * Func_File,
  INT_ File_Format,
  INT_ n,
  INT_ Number_of_Nodes,
  INT_ Number_of_U_Funcs,
  DOUBLE_1D * U_Funcs)

{

/*
 * Read a single function.
 */

  DOUBLE_1D * U_Func;

  INT_ i, Index, Number_of_Bytes, Number_of_Read_Items, Read_Flag,
       Special_Read_Items, U_Func_Index, U_Func_Location;

  float TMP_float;

  if (File_Format == UG_FIO_FORMATTED)
  {
    U_Func_Location = 0;

    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      U_Func = &U_Funcs[U_Func_Location];

      for (Index = 1; Index <= Number_of_Nodes; ++Index)
      {
        for (i = 0; i < n; ++i)
        {
          Read_Flag = fscanf (Func_File, "%lg", &U_Func[i+Index*n]);
        }
      }

      U_Func_Location = U_Func_Location + n*(Number_of_Nodes+1);

      if (Read_Flag == EOF)
      {
        ug_error_message ("*** ERROR : error reading SFUNC function file ***");
        return (1);
      }
    }
  }

  else
  {
    Special_Read_Items = ((File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE) ? 1 : 0);

    U_Func_Location = 0;

    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      Read_Flag = 0;

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
        Read_Flag = Read_Flag + ug_fread (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

      U_Func = &U_Funcs[U_Func_Location];

      if (File_Format == UG_FIO_UNFORMATTED_SINGLE || File_Format == UG_FIO_BINARY_SINGLE)
      {
        for (Index = 1; Index <= Number_of_Nodes; ++Index)
        {
          for (i = 0; i < n; ++i)
          {
            Read_Flag = Read_Flag + ug_fread (&TMP_float, sizeof (float), 1, Func_File);

            U_Func[i+Index*n] = (double) TMP_float;
          }
        }
      }
      else
        Read_Flag = Read_Flag + ug_fread (&U_Func[n], sizeof (double), Number_of_Nodes*n, Func_File);

      U_Func_Location = U_Func_Location + n*(Number_of_Nodes+1);

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
        Read_Flag = Read_Flag + ug_fread (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

      Number_of_Read_Items = n * Number_of_Nodes + Special_Read_Items + Special_Read_Items;

      if (Read_Flag != Number_of_Read_Items)
      {
        ug_error_message ("*** ERROR : error reading SFUNC function file ***");
        return (1);
      }
    }
  }

  return (0);

}

INT_ ug3_read_sfunc_label
 (FILE * Func_File,
  INT_ File_Format,
  INT_ Read_Task_Flag,
  INT_ Number_of_U_Funcs,
  CHAR_21 * U_Func_Labels)

{

/*
 * Read a single function.
 */

  CHAR_21 Label;
  CHAR_133 Long_Label;

  INT_ Label_Length, Max_Label_Length, Number_of_Bytes, Number_of_Read_Items,
       Read_Flag, Special_Read_Items, U_Func_Index; 

  Max_Label_Length = (INT_) sizeof (CHAR_21) - 1;

  if (File_Format == UG_FIO_FORMATTED)
  {
    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      Read_Flag = fscanf (Func_File, "%s\n", Long_Label);

      if (Read_Flag == EOF)
      {
        ug_error_message ("*** ERROR : error reading SFUNC function file ***");
        return (1);
      }

      if (U_Func_Labels && Read_Task_Flag == 2)
      {
        Label_Length = (INT_) strcspn (Long_Label, " ");
        Label_Length = MIN (Label_Length, Max_Label_Length);

        strcpy (U_Func_Labels[U_Func_Index], "");

        strncat (U_Func_Labels[U_Func_Index], Long_Label, Label_Length);
      }
    }
  }
  else
  {
    Special_Read_Items = ((File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE) ? 1 : 0);

    for (U_Func_Index = 0; U_Func_Index < Number_of_U_Funcs; ++U_Func_Index)
    {
      Read_Flag = 0;

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE) 
        Read_Flag = Read_Flag + ug_fread (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

      Read_Flag = Read_Flag + ug_fread (&Label, sizeof (char), 21, Func_File);

      if (File_Format == UG_FIO_UNFORMATTED_DOUBLE || File_Format == UG_FIO_UNFORMATTED_SINGLE)
        Read_Flag = Read_Flag + ug_fread (&Number_of_Bytes, sizeof (INT_), 1, Func_File);

      Number_of_Read_Items = 21 + Special_Read_Items + Special_Read_Items;

      if (Read_Flag != Number_of_Read_Items)
      {
        ug_error_message ("*** ERROR : error reading SFUNC function file ***");
        return (1);
      }

      if (U_Func_Labels && Read_Task_Flag == 2)
      {
        Label_Length = (INT_) strcspn (Label, " ");
        Label_Length = MIN (Label_Length, Max_Label_Length);

        strcpy (U_Func_Labels[U_Func_Index], "");

        strncat (U_Func_Labels[U_Func_Index], Label, Label_Length);
      }
    }

  }

  return (0);

}
