#include "UG3_LIB.h"

INT_ ug3_read_func_file
 (char File_Name[],
  INT_ Message_Flag,
  INT_ NDim,
  INT_ *Number_of_Nodes,
  INT_ *Number_of_U_Scalars,
  INT_ *Number_of_U_Vectors,
  INT_ *Number_of_U_Matrixes,
  INT_ *Number_of_U_Metrics,
  CHAR_21 ** U_Scalar_Labels,
  CHAR_21 ** U_Vector_Labels,
  CHAR_21 ** U_Matrix_Labels,
  CHAR_21 ** U_Metric_Labels,
  DOUBLE_1D ** U_Scalars,
  DOUBLE_1D ** U_Vectors,
  DOUBLE_1D ** U_Matrixes,
  DOUBLE_1D ** U_Metrics)
{

/*
 * Read function data from a function file.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_read_func_file.c,v 1.6 2022/11/21 00:25:14 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  FILE *Func_File = NULL;

  CHAR_UG_MAX drive, dname, ext, fname, File_Type, Text;
  CHAR_51 Name_Text;

  INT_ Close_Flag, Index, File_Mode, File_Format, Read_Task_Flag;
  INT_ SFUNC_Flag = 1;
  INT_ ierr = 0;

  // check file format

  ierr = ug_file_format_check (File_Name);

  if (ierr)
    return (ierr);

  // get file info

  ug_file_info (File_Name, File_Type, &File_Format, &File_Mode);

  // output header

  if (Message_Flag >= 1)
  {
    ug_message (" ");
    ug_message ("UG3      : INPUT FUNCTION FIELD");
    ug_message (" ");
    ug_message ("UG3      : Reading data");
    ug_message (" ");
  }

  // initialize number of function types

  *Number_of_U_Scalars = 0;
  *Number_of_U_Vectors = 0;
  *Number_of_U_Matrixes = 0;
  *Number_of_U_Metrics = 0;

  // make three passes

  // the first pass reads the dimensions
  // the second reads the labels and the third reads the data

  Read_Task_Flag = 1;

  do
  {
    // open file and set close flag

    if (File_Mode == UG_FIO_TMP_FILE_MODE)
    {
      Func_File = ug_fopen (File_Name, "r_tmp");

      Close_Flag = 1;
    }
    else
    {
      Func_File = ug_fopen (File_Name, "r");

      Close_Flag = 0;
    }

    if (Func_File == NULL)
    {
      ug_error_message ("*** ERROR 344 : unable to open function file ***");
      ierr = 344;
    }

    // allocate function flag and label arrays

    if (ierr == 0 && Read_Task_Flag == 2)
    {
      ierr = 0;

      *U_Scalar_Labels = (CHAR_21 *) ug_malloc (&ierr,
                                                (*Number_of_U_Scalars) * sizeof (CHAR_21));
      *U_Vector_Labels = (CHAR_21 *) ug_malloc (&ierr,
                                                (*Number_of_U_Vectors) * sizeof (CHAR_21));
      *U_Matrix_Labels = (CHAR_21 *) ug_malloc (&ierr,
                                                (*Number_of_U_Matrixes) * sizeof (CHAR_21));
      *U_Metric_Labels = (CHAR_21 *) ug_malloc (&ierr,
                                                (*Number_of_U_Metrics) * sizeof (CHAR_21));

      if (ierr > 0)
      {
        ug_error_message ("*** ERROR 100335 : unable to malloc function label arrays ***");
        return (100335);
      }
    }

    // initialize function labels

    if (ierr == 0 && Read_Task_Flag == 2)
    {
      if (*U_Scalar_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Scalars; ++Index)
        {
          strcpy ((*U_Scalar_Labels)[Index], "                    ");
        }
      }

      if (*U_Vector_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Vectors; ++Index)
        {
          strcpy ((*U_Vector_Labels)[Index], "                    ");
        }
      }

      if (*U_Matrix_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Matrixes; ++Index)
        {
          strcpy ((*U_Matrix_Labels)[Index], "                    ");
        }
      }

      if (*U_Metric_Labels)
      {
        for (Index = 0; Index < *Number_of_U_Metrics; ++Index)
        {
          strcpy ((*U_Metric_Labels)[Index], "                    ");
        }
      }
    }

    // malloc function data arrays

    if (ierr == 0 && Read_Task_Flag == 3)
    {
      ierr = 0;

      *U_Scalars = (DOUBLE_1D *) ug_malloc (&ierr,
                                            (*Number_of_U_Scalars)
                                          * ((*Number_of_Nodes)+1)
                                          * sizeof (DOUBLE_1D));
      *U_Vectors = (DOUBLE_1D *) ug_malloc (&ierr,
                                            (*Number_of_U_Vectors)
                                          * NDim * ((*Number_of_Nodes)+1)
                                          * sizeof (DOUBLE_1D));
      *U_Matrixes = (DOUBLE_1D *) ug_malloc (&ierr,
                                            (*Number_of_U_Matrixes)
                                          * NDim*NDim * ((*Number_of_Nodes)+1)
                                          * sizeof (DOUBLE_1D));
      *U_Metrics = (DOUBLE_1D *) ug_malloc (&ierr,
                                            (*Number_of_U_Metrics)
                                          * 3*(NDim-1) * ((*Number_of_Nodes)+1)
                                          * sizeof (DOUBLE_1D));

      if (ierr > 0)
      {
        ug_error_message ("*** ERROR 100336 : unable to malloc function data arrays ***");
        return (100336);
      }
    }

    // read function file

    if (ierr == 0)
    {
       if (strcmp (File_Type, ".ufunc") == 0)
         SFUNC_Flag = 0;

      if (strcmp (File_Type, ".ufunc") == 0 ||
          strcmp (File_Type, ".sfunc") == 0)
      {
        ierr = ug3_read_sfunc (Func_File, File_Format,
                               Read_Task_Flag, NDim, Number_of_Nodes,
                               Number_of_U_Scalars, Number_of_U_Vectors,
                               Number_of_U_Matrixes, Number_of_U_Metrics,
                               SFUNC_Flag,
                               *U_Scalar_Labels, *U_Vector_Labels,
                               *U_Matrix_Labels, *U_Metric_Labels,
                               *U_Scalars, *U_Vectors,
                               *U_Matrixes, *U_Metrics);

        if (ierr)
        {
          ug_error_message ("*** ERROR 344 : error reading function file ***");
          ierr = 344;
        }
      }

      else if (strcmp (File_Type, ".sol") == 0 ||
               strcmp (File_Type, ".solb") == 0)
      {
        Close_Flag = ug_fclose (Func_File);

        ierr = ug3_read_sol (File_Name, 
                             Read_Task_Flag, NDim, Number_of_Nodes,
                             Number_of_U_Scalars, Number_of_U_Vectors,
                             Number_of_U_Matrixes, Number_of_U_Metrics,
                             *U_Scalar_Labels, *U_Vector_Labels,
                             *U_Matrix_Labels, *U_Metric_Labels,
                             *U_Scalars, *U_Vectors, *U_Matrixes, *U_Metrics);

        Close_Flag = -1;

        if (ierr)
        {
          ug_error_message ("*** ERROR 344 : error reading function file ***");
          ierr = 344;
        }
      }

      else
      { 
        ug_error_message ("*** ERROR 344 : unknown input function file type ***");
        ierr = 344;
      }
    }

    // close file

    if (Close_Flag == 1)
      Close_Flag = ug_fclose (Func_File);

    // output function data information

    if (ierr == 0 && Read_Task_Flag == 1 && Message_Flag >= 1)
    {
      ug_splitpath (File_Name, drive, dname, fname, ext);

      strcat (fname, ext);
      strcpy (Name_Text, "");
      strncat (Name_Text, fname, 50);

      snprintf (Text, sizeof(Text), "UG3      : Function File Name= %s", Name_Text);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG3      : Nodes             =%10i", (int) *Number_of_Nodes);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG3      : Scalar Variables  =%10i", (int) *Number_of_U_Scalars);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG3      : Vector Variables  =%10i", (int) *Number_of_U_Vectors);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG3      : Matrix Variables  =%10i", (int) *Number_of_U_Matrixes);
      ug_message (Text);

      snprintf (Text, sizeof(Text), "UG3      : Metric Variables  =%10i", (int) *Number_of_U_Metrics);
      ug_message (Text);
    }

    ++Read_Task_Flag;
  }
  while (Read_Task_Flag <= 3 && ierr == 0);

  // close file

  if (Close_Flag == 0)
    Close_Flag = ug_fclose (Func_File);

  // output function labels

  if (ierr == 0 && Message_Flag >= 1)
  {
    if (*U_Scalar_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Scalars; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG3      : Scalar Label %-4i = %s",
                 (int) Index, (*U_Scalar_Labels)[Index]);

        ug_message (Text);
      }
    }

    if (*U_Vector_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Vectors; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG3      : Vector Label %-4i = %s",
                 (int) Index, (*U_Vector_Labels)[Index]);

        ug_message (Text);
      }
    }

    if (*U_Matrix_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Matrixes; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG3      : Matrix Label %-4i = %s",
                 (int) Index, (*U_Matrix_Labels)[Index]);

        ug_message (Text);
      }
    }

    if (*U_Metric_Labels)
    {
      for (Index = 0; Index < *Number_of_U_Metrics; ++Index)
      {
        snprintf (Text, sizeof(Text), "UG3      : Metric Label %-4i = %s",
                 (int) Index, (*U_Metric_Labels)[Index]);

        ug_message (Text);
      }
    }

    ug_message (" ");
  }

  return (ierr);

}
