#include "UG3_LIB.h"

/*
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_qvrr.c,v 1.4 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

double ug3_qvrr (
  double x11,
  double x12,
  double x13,
  double x21,
  double x22,
  double x23,
  double x31,
  double x32,
  double x33,
  double x41,
  double x42,
  double x43)
{

  // Determine tet volume to circum-radius cubed function, q.

  double a1231, a1232, a1233, a1421, a1422, a1423, a1341, a1342, a1343,
         cq, dx1, dx2, dx3,
         dx211, dx212, dx213, dx21s,
         dx311, dx312, dx313, dx31s,
         dx411, dx412, dx413, dx41s,
         q, rcs, vol, w;

  cq = 3.0 * sqrt (3.0) / 16.0;

  dx211 = x21 - x11;
  dx212 = x22 - x12;
  dx213 = x23 - x13;
  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx411 = x41 - x11;
  dx412 = x42 - x12;
  dx413 = x43 - x13;

  // area vectors

  a1231 = dx212 * dx313 - dx213 * dx312;
  a1232 = dx213 * dx311 - dx211 * dx313;
  a1233 = dx211 * dx312 - dx212 * dx311;
  a1341 = dx312 * dx413 - dx313 * dx412;
  a1342 = dx313 * dx411 - dx311 * dx413;
  a1343 = dx311 * dx412 - dx312 * dx411;
  a1421 = dx412 * dx213 - dx413 * dx212;
  a1422 = dx413 * dx211 - dx411 * dx213;
  a1423 = dx411 * dx212 - dx412 * dx211;

  // 6 * volume

  vol = dx211 * a1341 + dx212 * a1342 + dx213 * a1343;

  // delta vector dot products

  dx21s = dx211 * dx211 + dx212 * dx212 + dx213 * dx213;
  dx31s = dx311 * dx311 + dx312 * dx312 + dx313 * dx313;
  dx41s = dx411 * dx411 + dx412 * dx412 + dx413 * dx413;

  // determine the circumcenter delta vector

  w = 0.5 / vol;

  dx1 = w * (dx21s * a1341 + dx31s * a1421 + dx41s * a1231);
  dx2 = w * (dx21s * a1342 + dx31s * a1422 + dx41s * a1232);
  dx3 = w * (dx21s * a1343 + dx31s * a1423 + dx41s * a1233);

  // determine circumsphere radius squared

  rcs = dx1 * dx1 + dx2 * dx2 + dx3 * dx3;

  q = cq * vol / (rcs * sqrt (rcs));

  return (q);
}

double ug3_tet_qvrr (INT_4D in_tet, DOUBLE_3D * x)
{
  // Determine tet volume to circum-radius cubed function, q.

  return (ug3_qvrr (x[in_tet[0]][0], x[in_tet[0]][1], x[in_tet[0]][2],
                    x[in_tet[1]][0], x[in_tet[1]][1], x[in_tet[1]][2],
                    x[in_tet[2]][0], x[in_tet[2]][1], x[in_tet[2]][2],
                    x[in_tet[3]][0], x[in_tet[3]][1], x[in_tet[3]][2]));
}

double ug3_pyramid_qvrr (INT_5D in_pyramid, DOUBLE_3D * x)
{
  // Determine pyramid volume to circum-radius cubed function, q.

  INT_ in_tet[4];

  INT_ map_tet[2][2][4] = { { {0,1,2,3}, {3,1,2,4} },
                            { {0,1,2,4}, {0,4,2,3} } };

  INT_ i, j, k;

  double qj, qmin;
  double q = 0.0;

  for (i = 0; i < 2; i++) {

    qmin = 1.0;

    for (j = 0; j < 2; j++) {

      for (k = 0; k < 4; k++) {
        in_tet[k] = in_pyramid[map_tet[i][j][k]];
      }

      qj = ug3_tet_qvrr (in_tet, x);

      qmin = MIN (qmin, qj);
    }

    q = MAX (q, qmin);
  }

  return q;
}

double ug3_prism_qvrr (INT_6D in_prism, DOUBLE_3D * x)
{
  // Determine prism volume to circum-radius cubed function, q.

  INT_ in_tet[4];

  INT_ map_tet[6][3][4] = { { {0,1,2,3}, {3,1,2,4}, {3,4,2,5} },
                            { {0,1,2,3}, {3,1,2,5}, {3,1,5,4} },
                            { {0,1,2,4}, {0,4,2,3}, {3,4,2,5} },
                            { {0,1,2,4}, {0,4,2,5}, {0,4,5,3} },
                            { {0,1,2,5}, {0,1,5,3}, {3,1,5,4} },
		            { {0,1,2,5}, {0,1,5,4}, {0,4,5,3} } };

  INT_ i, j, k;

  double qj, qmin;
  double q = 0.0;

  for (i = 0; i < 6; i++) {

    qmin = 1.0;

    for (j = 0; j < 3; j++) {

      for (k = 0; k < 4; k++) {
        in_tet[k] = in_prism[map_tet[i][j][k]];
      }

      qj = ug3_tet_qvrr (in_tet, x);

      qmin = MIN (qmin, qj);
    }

    q = MAX (q, qmin);
  }

  return q;
}

double ug3_hex_qvrr (INT_8D in, DOUBLE_3D * x)
{
  // Determine hex volume to circum-radius cubed function, q.

  INT_ in_prism[6];
  INT_ in_tet[4];

  INT_ map_prism[2][2][6] = { { {0,1,2,4,5,6}, {0,2,3,4,6,7} },
                              { {0,1,2,4,5,6}, {0,2,3,4,6,7} } };

  INT_ map_tet[2][5][4] = { { {0,1,2,5}, {0,2,3,7}, {0,5,2,7}, {0,5,7,4}, {2,7,5,6} },
                            { {0,1,3,4}, {1,2,3,6}, {1,6,4,5}, {1,3,4,7}, {7,4,1,6} } };

  INT_ i, j, k;

  double qj, qmin;
  double q = 0.0;

  for (i = 0; i < 2; i++) {

    qmin = 1.0;

    for (j = 0; j < 2; j++) {

      for (k = 0; k < 6; k++) {
        in_prism[k] = in[map_prism[i][j][k]];
      }

      qj = ug3_prism_qvrr (in_prism, x);

      qmin = MIN (qmin, qj);
    }

    q = MAX (q, qmin);
  }

  for (i = 0; i < 2; i++) {

    qmin = 1.0;

    for (j = 0; j < 5; j++) {

      for (k = 0; k < 4; k++) {
        in_tet[k] = in[map_tet[i][j][k]];
      }

      qj = ug3_tet_qvrr (in_tet, x);

      qmin = MIN (qmin, qj);
    }

    q = MAX (q, qmin);
  }

  return q;
}
