#include "UG3_LIB.h"

void ug3_proj_vector (double b1,
                      double b2,
                      double b3,
                      double *tu11,
                      double *tu12,
                      double *tu13,
                      double *tu21,
                      double *tu22,
                      double *tu23)
{

/*
 * Determine the transformation vectors for a plane defined by a normal vector.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_proj.c,v 1.6 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  double tu31m, tu32m, tu33m, tu3_minm, dc0, dc1, tu31, tu32, tu33, w;

  dc0 = 0.0;
  dc1 = 1.0;

  // transformation from xyz to uvw space (where w is normal to the plane)

  // set normalized w vector

  tu31 = b1;
  tu32 = b2;
  tu33 = b3;

  w = dc1 / sqrt (tu31 * tu31 + tu32 * tu32 + tu33 * tu33);

  tu31 = w * tu31;
  tu32 = w * tu32;
  tu33 = w * tu33;

  // get magnitude of w components

  tu31m = fabs (tu31);
  tu32m = fabs (tu32);
  tu33m = fabs (tu33);

  tu3_minm = MIN (tu31m, tu32m);
  tu3_minm = MIN (tu3_minm, tu33m);

  // set u vector direction based on minimum w vector component

  if (tu31m == tu3_minm)
  {
    *tu11 = dc1;
    *tu12 = dc0;
    *tu13 = dc0;
  }
  else if (tu32m == tu3_minm)
  {
    *tu11 = dc0;
    *tu12 = dc1;
    *tu13 = dc0;
  }
  else 
  {
    *tu11 = dc0;
    *tu12 = dc0;
    *tu13 = dc1;
  }

  // determine orthogonal v vector from w vector and chosen u vector direction

  *tu21 = tu32 * (*tu13) - tu33 * (*tu12);
  *tu22 = tu33 * (*tu11) - tu31 * (*tu13);
  *tu23 = tu31 * (*tu12) - tu32 * (*tu11);

  w = dc1 / sqrt ((*tu21) * (*tu21) + (*tu22) * (*tu22) + (*tu23) * (*tu23));

  *tu21 = w * (*tu21);
  *tu22 = w * (*tu22);
  *tu23 = w * (*tu23);

  // determine orthogonal u vector from w vector and v vector

  *tu11 = (*tu22) * tu33 - (*tu23) * tu32;
  *tu12 = (*tu23) * tu31 - (*tu21) * tu33;
  *tu13 = (*tu21) * tu32 - (*tu22) * tu31;

  return;

}

void ug3_proj_uv_from_xyz (double *du1,
                           double *du2,
                           double dx1,
                           double dx2,
                           double dx3,
                           double tu11,
                           double tu12,
                           double tu13,
                           double tu21,
                           double tu22,
                           double tu23)
{
  // Given dx1,dx2,dx3 in xyz physical space find
  // du1,du2 in uv transformed space.

  *du1 = tu11 * dx1 + tu12 * dx2 + tu13 * dx3;
  *du2 = tu21 * dx1 + tu22 * dx2 + tu23 * dx3;

  return;
}

void ug3_proj_xyz_from_uv (double du1,
                           double du2,
                           double *dx1,
                           double *dx2,
                           double *dx3,
                           double tu11,
                           double tu12,
                           double tu13,
                           double tu21,
                           double tu22,
                           double tu23)
{
  // Given du1,du2 in uv transformed space find
  // dx1,dx2,dx3 in xyz physical space.

  *dx1 = tu11 * du1 + tu21 * du2;
  *dx2 = tu12 * du1 + tu22 * du2;
  *dx3 = tu13 * du1 + tu23 * du2;

  return;
}
