#include "UG3_LIB.h"

/*
 * Routines used to evaluate the P2-tet element quality measure function
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_p2_tet_q.c,v 1.8 2020/02/15 05:26:44 marcum Exp $
 * Copyright 2018-2020, Remi Feuillet and Frederic Alauzet, INRIA, and David Marcum
 */

//--- tables of edges numbering
static INT_ t2e[6][2] = {{0,1} , {0,2} , {0,3} , {1,2} , {1,3} , {2,3} };

//--- Jacobian control coefficients at vertices for a P2Tet
static INT_ CCV_Tet[4][3][2] =  { { {0,4},{0,5},{0,6} }, 
                                  { {4,1},{4,7},{4,8} }, 
                                  { {5,7},{5,2},{5,9} }, 
                                  { {6,8},{6,9},{6,3} } };

//--- Jacobian control coefficients at edges for a P2Tet
static INT_ CCE_Tet[12][3][3][2] = { { { {0,4},{0,5},{4,8} }, { {0,4},{4,7},{0,6} }, { {4,1},{0,5},{0,6} } }, 
                                     { { {4,1},{4,7},{4,6} }, { {4,1},{0,5},{4,8} }, { {0,4},{4,7},{4,8} } }, 
                                     { { {0,4},{0,5},{6,3} }, { {0,4},{6,9},{0,6} }, { {6,8},{0,5},{0,6} } }, 
                                     { { {6,8},{6,9},{0,6} }, { {6,8},{0,5},{6,3} }, { {0,4},{6,9},{6,3} } }, 
                                     { { {4,1},{4,7},{5,9} }, { {4,1},{5,2},{4,8} }, { {5,7},{4,7},{4,8} } }, 
                                     { { {4,1},{5,2},{5,9} }, { {5,7},{4,7},{5,9} }, { {5,7},{5,2},{4,8} } }, 
                                     { { {5,7},{5,2},{0,6} }, { {5,7},{0,5},{5,9} }, { {0,4},{5,2},{5,9} } }, 
                                     { { {0,4},{0,5},{5,9} }, { {0,4},{5,2},{0,6} }, { {5,7},{0,5},{0,6} } }, 
                                     { { {4,1},{4,7},{6,3} }, { {4,1},{6,9},{4,8} }, { {6,8},{4,7},{4,8} } }, 
                                     { { {4,1},{6,9},{6,3} }, { {6,8},{4,7},{6,3} }, { {6,8},{6,9},{4,8} } }, 
                                     { { {5,7},{5,2},{6,3} }, { {5,7},{6,9},{5,9} }, { {6,8},{5,2},{5,9} } }, 
                                     { { {6,8},{6,9},{5,9} }, { {5,7},{6,9},{6,3} }, { {6,8},{5,2},{6,3} } }};

//--- Jacobian control coefficients at faces for a P2Tet
static INT_ CCF_Tet[4][6][3][2] = { { { {4,1},{5,2},{6,3} }, { {4,1},{6,9},{5,9} }, { {5,7},{4,7},{6,3} }, { {5,7},{6,9},{4,8} }, { {6,8},{4,7},{5,9} }, { {6,8},{5,2},{4,8} } }, 
                                    { { {0,4},{5,2},{6,3} }, { {0,4},{6,9},{5,9} }, { {5,7},{0,5},{6,3} }, { {5,7},{6,9},{0,6} }, { {6,8},{0,5},{5,9} }, { {6,8},{5,2},{0,6} } }, 
                                    { { {0,4},{4,7},{6,3} }, { {0,4},{6,9},{4,8} }, { {4,1},{0,5},{6,3} }, { {4,1},{6,9},{0,6} }, { {6,8},{0,5},{4,8} }, { {6,8},{4,7},{0,6} } }, 
                                    { { {0,4},{4,7},{5,9} }, { {0,4},{5,2},{4,8} }, { {4,1},{0,5},{5,9} }, { {4,1},{5,2},{0,6} }, { {5,7},{0,5},{4,8} }, { {5,7},{4,7},{0,6} } } };

//--- Subvolumes for the computation of the volume of the P2 tetrahedron
static INT_ VT_Tet[8][4] = { {0,4,5,6}, {1,7,4,8}, {2,5,7,9}, {6,8,9,3}, {4,7,5,8}, {5,7,9,8}, {6,5,9,7}, {6,4,5,7}};

//--- Subsurfaces for the computation of the surface of the P2 tetrahedron
static INT_ ST_Tet[4][4][3] = { { {1,4,5}, {0,5,4}, {2,7,5}, {4,5,7} },
                                { {0,4,6}, {1,8,4}, {3,6,8}, {4,8,6} },
                                { {1,7,8}, {2,9,7}, {3,8,9}, {7,9,8} },
                                { {2,5,9}, {0,6,5}, {3,9,6}, {5,6,9} }};

//--- Sub-lengths for the computation of the highest edge length of the P2 tetrahedron
static INT_ LT_Tet[6][2][2] = { { {0,4}, {4,1} },
                                { {0,5}, {5,2} },
                                { {0,6}, {6,3} },
                                { {1,7}, {7,2} },
                                { {1,8}, {8,3} },
                                { {2,9}, {9,3} }};

//--- 3D geometric routines

static inline void Geo_VecCrossProduct3d (DOUBLE_3D u, DOUBLE_3D v, DOUBLE_3D w)
{
  w[0] = u[1]*v[2] - u[2]*v[1];
  w[1] = u[2]*v[0] - u[0]*v[2];
  w[2] = u[0]*v[1] - u[1]*v[0];
}

static inline double Geo_VecDotProduct3d (DOUBLE_3D u, DOUBLE_3D v)
{
  double dot;
  
  dot = u[0]*v[0] + u[1]*v[1] + u[2]*v[2];
  
  return (dot);
}

static inline double Geo_VecNorm3d (DOUBLE_3D u)
{
  double nrm;
  
  nrm = Geo_VecDotProduct3d (u, u);
  nrm = sqrt (nrm);
  
  return (nrm);
}

static inline double Geo_VecDeterminant3d (DOUBLE_3D u, DOUBLE_3D v, DOUBLE_3D w)
{
  double    det;
  DOUBLE_3D  crs;

  Geo_VecCrossProduct3d (u, v, crs);
  det = Geo_VecDotProduct3d (crs, w);
  
  return (det);
}

static inline double Geo_VecCrossProductNorm3d (DOUBLE_3D u, DOUBLE_3D v, DOUBLE_3D w)
{
  double nrm;
  
  w[0] = u[1]*v[2] - u[2]*v[1];
  w[1] = u[2]*v[0] - u[0]*v[2];
  w[2] = u[0]*v[1] - u[1]*v[0];

  nrm  = fabs (w[0]*w[0]+w[1]*w[1]+w[2]*w[2]);
  nrm  = sqrt (nrm);
  
  return (nrm);
}

static inline double Geo_TetraVolume (DOUBLE_3D crd0, DOUBLE_3D crd1, DOUBLE_3D crd2, DOUBLE_3D crd3)
{
  INT_      i;
  double    dc6, vol;
  DOUBLE_3D vec1, vec2, vec3, vno; 

  dc6 = 6.0;

  for (i=0; i<3; ++i) {
    vec1[i] = crd1[i] - crd0[i];
    vec2[i] = crd2[i] - crd0[i];
    vec3[i] = crd3[i] - crd0[i];    
  }

  Geo_VecCrossProduct3d (vec2, vec3, vno);

  vol  = Geo_VecDotProduct3d (vec1, vno);
  vol  = vol / dc6;

  return (vol);
}

//--- 3D geometric routines

static inline double Geo_TriangleFaceArea (DOUBLE_3D crd0, DOUBLE_3D crd1, DOUBLE_3D crd2)
{
  INT_      i;
  double    air, dc1d2, nrm;
  DOUBLE_3D vec[2], crs;

  dc1d2 = 0.5;

  //--- Face edges : edg[0]=P1P2, edg[1]=P2P0, edg[2]=P0P1 
  for (i=0; i<3; ++i) {
    vec[0][i] = crd2[i] - crd0[i];
    vec[1][i] = crd0[i] - crd1[i];
  }
  
  //--- Compute triangle face area : 0.5*||edg0/\edg1||
  nrm = Geo_VecCrossProductNorm3d (vec[0], vec[1], crs);
  
  air = dc1d2*nrm;

  return (air);
}

static inline double Geo_EdgeLength3d (DOUBLE_3D crd0, DOUBLE_3D crd1)
{
  INT_    i;
  double   len;
  DOUBLE_3D vec;

  for (i=0; i<3; ++i) 
    vec[i] = crd1[i] - crd0[i];

  len = Geo_VecNorm3d (vec);

  return (len);
}

static INT_ Geo_SubdivideEdge(double *N, double **NSub, INT_ NbrPnt, double u, double v)
{
  //--- Recursive algorithm of edge subdivision
  INT_     i;
  INT_     ierr = 0;
  double  *NBry, *NSubBry;

  NBry    = NULL;
  NSubBry = NULL;

  if ( NbrPnt <= 1 )
    (*NSub) = NULL;

  //--- create the barycenter and put it twice in order to have an even number of points 
  else if ( NbrPnt == 2 ) { 
    (*NSub) = ug_malloc(&ierr, 4*sizeof(double));

    if (ierr) {
      free(NBry);
      free(NSubBry);
      ug_error_message ("*** ERROR 100349 : unable to allocate required memory ***");
      return (100349);
    }

    (*NSub)[0] = N[0];
    (*NSub)[1] = u*N[0] + v*N[1];
    (*NSub)[2] = u*N[0] + v*N[1];
    (*NSub)[3] = N[1];
  }

  else {
    //--- apply the algorithm to the curve defined by the barycenters
    NBry = ug_malloc(&ierr, (NbrPnt-1)*sizeof(double));
    for (i=0; i<NbrPnt-1; i++)
      NBry[i] = u*N[i] + v*N[i+1];

    ierr = Geo_SubdivideEdge (NBry, &NSubBry, NbrPnt-1, u, v);

    if (ierr) {
      free(NBry);
      free(NSubBry);
      return (ierr);
    }

    (*NSub) =  ug_malloc(&ierr, 2*NbrPnt*sizeof(double));

    if (ierr) {
      free(NBry);
      free(NSubBry);
      ug_error_message ("*** ERROR 100350 : unable to allocate required memory ***");
      return (100350);
    }

    (*NSub)[0] = N[0];
    (*NSub)[2*NbrPnt-1] = N[NbrPnt - 1];

    for (i=0; i<2*(NbrPnt-1); i++)
      (*NSub)[i+1] = NSubBry[i];

    free(NBry);
    free(NSubBry);
  }

  return (0);
}

static INT_ Geo_CheckP2JacobianOnEdge (double* N, INT_ NbrPnt, double u, double v, double *res)
{
  //--- Recursive algorithm to check the jacobian on an edge in the case of a negative control point
  INT_   i;
  INT_   ierr = 0;
  double *NSub, *NBis, Nmin, res1, res2;

  NSub = NULL;
  NBis = NULL;
  res1 = -10000.0;
  res2 = -10000.0;

  //--- Subdivision of the edge based on a de Casteljau's algorithm
  ierr = Geo_SubdivideEdge(N, &NSub, NbrPnt, u, v);

  if (ierr) {
    free(NSub);
    NSub = NULL;
    return (ierr);
  }

  //--- Look for the sign of the jacobian in the barycentric zone
  if ( NSub[NbrPnt] < 0 ) {
    //--- the jacobian is negative in (u,v)
    free(NSub);
    *res = res1;
    return 0;
  }

  NBis = malloc(NbrPnt*sizeof(double));

  //--- check in the first part of the edge
  for (i=0; i<NbrPnt; i++) {
    NBis[i] = NSub[i];
    if ( i == 0 ) {
      //Nmax = NBis[0];
      Nmin = NBis[0];
    }
    else {
      Nmin = MIN(Nmin, NBis[i]);
    }
  }

  if ( Nmin < 0 ) {
    ierr = Geo_CheckP2JacobianOnEdge (NBis, NbrPnt, u, v, &res1);
    free(NBis);
    NBis = NULL;
    if (ierr) {
      free(NSub);
      NSub = NULL;
      return (ierr);
    }
    if ( res1 < 0 ) { //--- invalid
      free(NSub);
      NSub = NULL;
      *res = res1;
      return 0;
    }
  }
  else
    res1 = Nmin;

  if ( NBis ) {
    free(NBis);
    NBis = NULL;
  }

  NBis = ug_malloc (&ierr, NbrPnt*sizeof(double));
  if (ierr) {
    free(NSub);
    NSub = NULL;
    return (ierr);
  }

  //--- check in the second part of the edge
  for (i=0; i<NbrPnt; i++) {
    NBis[i] = NSub[i + NbrPnt];
    if ( i == 0 ) {
      //Nmax = NBis[0];
      Nmin = NBis[0];
    }
    else {
      Nmin = MIN(Nmin, NBis[i]);
    }
  }
  if ( Nmin < 0 ) {
    ierr = Geo_CheckP2JacobianOnEdge(NBis, NbrPnt, u, v, &res2);
    free(NBis);
    NBis = NULL;
    if (ierr) {
      free(NSub);
      NSub = NULL;
      return (ierr);
    }

    if ( res2 < 0 ) { //--- invalid
      free(NSub);
      NSub = NULL;
      *res = res2;
      return 0;
    }
  }
  else
    res2 = Nmin;

  if ( NBis ) {
    free(NBis);
    NBis = NULL;
  }

  free(NSub);
  NSub = NULL;
  *res = MIN(res1,res2);
  return 0;
}

//--- P2-Tet Element Quality Measure Function

INT_ ug3_p2_tet_q (INT_ ielem,
                   INT_4D * iniel,
                   INT_6D * p2_iniel,
                   double *qua,
                   double *vol,
                   DOUBLE_3D * x)
{
  DOUBLE_4X3 x_p1;
  DOUBLE_6X3 x_p2;
  INT_ i;
  for (i=0; i<4; i++) {
    memcpy(x_p1[i], x[iniel[ielem][i]], sizeof(DOUBLE_3D));
  }
  for (i=0; i<6; i++) {
    memcpy(x_p2[i], x[p2_iniel[ielem][i]], sizeof(DOUBLE_3D));
  }
  return (ug3_p2_tet_q_i (qua, vol, x_p1, x_p2));
}

INT_ ug3_p2_tet_q_i (double *qua,
                     double *vol,
                     DOUBLE_4X3 x_p1,
                     DOUBLE_6X3 x_p2)
{
  INT_       i, j, k, l;
  INT_       ierr = 0;
  double     dc0, dc1, dc3, dc6, dc8, dc1d2, dc1d3;
  double     N[20], Nedg[4], Nmax, Nmin, NminIni, Air, AirFac, AirMax, lgt, h, res, p1_vol;
  DOUBLE_3D  M[10], P[10], u[3];

  dc0 = 0.0;
  dc1 = 1.0;
  dc3 = 3.0;
  dc6 = 6.0;
  dc8 = 8.0;
  dc1d2 = 0.5;
  dc1d3 = dc1/dc3;

  *qua = dc0;
  *vol = dc0;

  //--- get 10 Lagrange points of the P2 tetrahedron
  memcpy(M, x_p1, sizeof(DOUBLE_4X3));
  memcpy(&(M[4][0]), x_p2, sizeof(DOUBLE_6X3));

  //--- Computation of control points ( only edges control points in P2)
  memcpy(P, x_p1, sizeof(DOUBLE_4X3));
  for (i=0; i<6; i++) {
    for (j=0; j<3; j++) {
      P[i+4][j] = M[i+4][j] + M[i+4][j] - dc1d2* (M[t2e[i][0]][j] - M[t2e[i][1]][j]);
    }
  }

  //--- Control coefficients of the jacobian at vertices
  for (i=0; i<4;i++) {
    for (j=0; j<3; j++) {
      for (k=0; k<3; k++) {
        u[j][k] = P[CCV_Tet[i][j][1]][k] - P[CCV_Tet[i][j][0]][k];
      }
    }

    N[i] = dc8 * Geo_VecDeterminant3d(u[0], u[1], u[2]);

    if ( i == 0 ) {
      Nmin = N[i];
      Nmax = N[i]; 
    }
    else {
      Nmin = MIN(Nmin, N[i]);
      Nmax = MAX(Nmax, N[i]);
    }
    if ( Nmin <= 0 ) //--- the high-order element is invalid
      return -1;
  }

  //--- Control coefficients of the jacobian at edges
  for (i=0; i<6; i++) {
    NminIni = Nmin;
    N[4 + 2*i] = 0;
    for (j=0; j<3; j++) {
      for (k=0; k<3; k++) {
        for (l=0; l<3; l++) {
          u[k][l] = P[CCE_Tet[2*i][j][k][1]][l] - P[CCE_Tet[2*i][j][k][0]][l];
        }
      }
      N[4 + 2*i] += dc8 * Geo_VecDeterminant3d(u[0], u[1], u[2]);
    }
    N[4 + 2*i] /= dc3;
    Nmin = MIN(Nmin, N[4 + 2*i]);
    Nmax = MAX(Nmax, N[4 + 2*i]);

    N[4 + 2*i + 1] = 0;
    for (j=0; j<3; j++) {
      for (k=0; k<3; k++) {
        for (l=0; l<3; l++) {
          u[k][l] = P[CCE_Tet[2*i + 1][j][k][1]][l] - P[CCE_Tet[2*i + 1][j][k][0]][l];
        }
      }
      N[4 + 2*i + 1] += dc8 * Geo_VecDeterminant3d(u[0], u[1], u[2]);
    }
    N[4 + 2*i + 1] /= dc3;
    Nmin = MIN(Nmin, N[4 + 2*i + 1]);
    Nmax = MAX(Nmax, N[4 + 2*i + 1]);


    if ( Nmin <= 0 ) {
      Nedg[0] = N[t2e[i][0]];
      Nedg[1] = N[4 + 2*i];
      Nedg[2] = N[4 + 2*i + 1];
      Nedg[3] = N[t2e[i][1]];
      ierr = Geo_CheckP2JacobianOnEdge(Nedg, 4, dc1d2, dc1d2, &res);
      if (ierr)
        return (ierr);
      Nmin = MIN(NminIni, res);
      if ( Nmin < 0)
        return -1;
    }
  }

  //---- Control coefficients of the jacobian at faces
  for (i=0; i<4; i++) {
    N[16 + i] = 0;
    for (j=0;j<6;j++) {
      for (k=0; k<3; k++) {
        for (l=0; l<3; l++) {
          u[k][l] = P[CCF_Tet[i][j][k][1]][l] - P[CCF_Tet[i][j][k][0]][l];
        }
      }
      N[16 + i] += dc8*Geo_VecDeterminant3d(u[0], u[1], u[2]);
    }
    N[16 + i] /= dc6;
    Nmin = MIN(Nmin, N[16 + i]);
    Nmax = MAX(Nmax, N[16 + i]);
    
    if ( Nmin <= 0 )
      return -1;
  }

  //---  Compute the approximation of the P2 Volume
  *vol = dc0;
  for (i=0; i<8; i++) {
    (*vol) += Geo_TetraVolume(M[VT_Tet[i][0]], M[VT_Tet[i][1]], M[VT_Tet[i][2]], M[VT_Tet[i][3]]);
  }
  
  //--- Computation of the approximation of the surface of each face
  Air    = 0;
  AirMax = 0;
  for (i=0; i<4; i++) {
    AirFac = 0;
    for (j=0; j<4; j++) {
      AirFac += Geo_TriangleFaceArea(M[ST_Tet[i][j][0]], M[ST_Tet[i][j][1]], M[ST_Tet[i][j][2]]);
    }
    Air += AirFac;
    if ( i == 0 )
      AirMax = AirFac;
    else 
      AirMax = MAX(AirMax, AirFac);
  }

  //--- Get the maximal height/length
  h = 0;
  for (i=0; i<6; i++) {
    lgt = Geo_EdgeLength3d(M[LT_Tet[i][0][0]], M[LT_Tet[i][0][1]]) + Geo_EdgeLength3d(M[LT_Tet[i][1][0]], M[LT_Tet[i][1][1]]);
    h = MAX(lgt, h);
  }
  
  //--- Computation of the P1 area
  p1_vol = Geo_TetraVolume(M[0], M[1], M[2], M[3]);
  
  if ( p1_vol <= 0 )
    return -1;

  *qua = pow(Nmax/Nmin,dc1d3)*(h * Air * MAX((*vol),p1_vol) )/ (dc6 * sqrt(dc6)* (*vol) * MIN((*vol),p1_vol)); 

  return 0;
}

