#include "UG3_LIB.h"

INT_ ug3_msd_elem (INT_ ielem0,
                   INT_ isdf_min,
                   INT_ nbface,
                   INT_ nelem,
                   INT_ nlayer_sd,
                   INT_ nnode,
                   INT_ *nelpntd,
                   INT_3D * inibf,
                   INT_4D * iniel,
                   INT_1D * isdfibf,
                   INT_1D ** ielin,
                   INT_1D ** lielin,
                   INT_1D ** msd_elem)
{

/*
 * Set element flag for sub-domain interface boundary regions.
 *
 * Note that isdfibf is a boundary face flag that identifies sub-domain
 * interface boundary regions as those with isdfibf[ibface] >= isdf_min.
 * e.g. if isdfibf = ibcibf then isdf_min = PART_UG3_GBC
 * e.g. if isdfibf = idibf then isdf_min = idmax+1 (or similar)
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_msd_elem.c,v 1.13 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_1D *msd_node = NULL;

  INT_ found, ibface, ibfn, ieln, ielem, inode,
       jnode, layer, loc, loc1, loc2, mlielin, mmsd_elem, nelpnt;
  INT_ ierr = 0;
  
  // return if there are no sub-domain interface boundaries

  if (ug_max_int (1, nbface, isdfibf) < isdf_min)
    return (0);

  // allocate element flag, element node map, and temporary space

  mmsd_elem = (*msd_elem == NULL) ? 0: 1;
  mlielin = (*lielin == NULL) ? 0: 1;

  if (mmsd_elem == 0) *msd_elem = (INT_1D *) ug_malloc (&ierr, (nelem+1) * sizeof (INT_1D));
  msd_node = (INT_1D *) ug_malloc (&ierr, (nnode+1) * sizeof (INT_1D));
  if (mlielin == 0) *lielin = (INT_1D *) ug_malloc (&ierr, (nnode+2) * sizeof (INT_1D));

  if (mlielin == 0)
  {
    *nelpntd = 0;

    ug_free (*ielin);

    *ielin = NULL;
  }

  // set error message

  if (ierr)
  {
    ug_error_message ("*** ERROR 100339 : unable to allocate required memory ***");
    ierr = 100339;
  }

  // create list of elements surrounding each node

  if (ierr == 0)
    ierr = ug3_ielin (1, nelem, nelpntd, nnode, &nelpnt,
                      iniel, NULL, ielin, *lielin);

  // return on error

  if (ierr)
  {
    if (mmsd_elem == 0)
    {
      ug_free (*msd_elem);
      *msd_elem = NULL;
    }
    ug_free (msd_node);
    if (mlielin == 0)
    {
      *nelpntd = 0;
      ug_free (*ielin);
      ug_free (*lielin);
      *ielin = NULL;
      *lielin = NULL;
    }
    return (ierr);
  }

  // initialize element and node flags

  ug_set_int (1, nelem, 0, *msd_elem);
  ug_set_int (1, nnode, 0, msd_node);

  // set node flags on sub-domain interface boundaries

  for (ibface = 1; ibface <= nbface; ++ibface)
  {
    if (isdfibf[ibface] >= isdf_min)
    {
      for (ibfn = 0; ibfn < 3; ++ibfn)
      {
        inode = inibf[ibface][ibfn];

        msd_node[inode] = 1;
      }
    }
  }

  // flag nlayer_sd layers of elements around sub-domain interface boundaries
  // and consider these the sub-domain interface boundary region

  for (layer = 1; layer <= nlayer_sd; ++layer)
  {
    for (inode = 1; inode <= nnode; ++inode)
    {
      if (msd_node[inode] == layer)
      {
        loc1 = (*lielin)[inode];
        loc2 = (*lielin)[inode+1];

        for (loc = loc1; loc < loc2; ++loc)
        {
          ielem = (*ielin)[loc];

          if ((*msd_elem)[ielem] == 0)
          {
            (*msd_elem)[ielem] = 1;

            for (ieln = 0; ieln < 4; ++ieln)
            {
              jnode = iniel[ielem][ieln];

              if (msd_node[jnode] == 0)
                msd_node[jnode] = layer+1;
            }
          }
        }
      }
    }
  }

  // reset element flag

  // if msd_elem = 1 then the element is within interface region
  // if msd_elem = 0 then the element is not within interface region

  for (ielem = 1; ielem <= nelem; ++ielem)
  {
    found = 0;

    ieln = 0;

    while (ieln < 4 && found == 0)
    {
      inode = iniel[ielem][ieln];

      found = (msd_node[inode]) ? 1: 0;

      ++ieln;
    }

    (*msd_elem)[ielem] = (found) ? 1: 0;
  }

  // set element flag to zero for elements with an index less that ielem0

  ug_set_int (1, ielem0-1, 0, *msd_elem);

  // free element node map and temporary space

  if (mlielin == 0)
  {
    *nelpntd = 0;

    ug_free (*ielin);
    ug_free (*lielin);

    *ielin = NULL;
    *lielin = NULL;
  }

  ug_free (msd_node);

  return (0);
}
