#include "UG3_LIB.h"

INT_ ug3_mq_elem (INT_ nelem,
                  INT_ nnode,
                  INT_ nlayer_q,
                  INT_4D * iniel,
                  INT_1D ** mq_elem,
                  double angqmax,
                  DOUBLE_3D * x)
{

/*
 * Set element flag for regions surrounding low quality elements. Regions are
 * set to nq_elem layers around a low-quality element.
 *
 * Note that isdfibf is a boundary face flag that identifies sub-domain
 * interface boundary regions as those with isdfibf[ibface] >= isdf_min.
 * e.g. if isdfibf = ibcibf then isdf_min = PART_UG3_GBC
 * e.g. if isdfibf = idibf then isdf_min = idmax+1 (or similar)
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_mq_elem.c,v 1.11 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_1D *ielin = NULL;
  INT_1D *lielin = NULL;
  INT_1D *mq_node = NULL;

  INT_ found, ieln, ielem, inode, inode1, inode2, inode3, inode4,
       jnode, layer, loc, loc1, loc2, nelpnt;
  INT_ ierr = 0;
  INT_ nelpntd = 0;

  double cosqmax, cosqmaxs, dc0, dc1, dc45, w, w21, w31, w41, w32, w42, w43;

  dc0 = 0.0;
  dc1 = 1.0;
  dc45 = 45.0;

  cosqmax = cos (angqmax * atan (dc1) / dc45);

  cosqmaxs = cosqmax * fabs (cosqmax);

  // allocate temporary space

  lielin = (INT_1D *) ug_malloc (&ierr, (nnode+2) * sizeof (INT_1D));
  mq_node = (INT_1D *) ug_malloc (&ierr, (nnode+1) * sizeof (INT_1D));

  // return on error

  if (ierr)
  {
    ug_free (lielin);
    ug_free (mq_node);
    ug_error_message ("*** ERROR 100340 : unable to allocate required memory ***");
    return (100340);
  }

  // initialize node flags

  ug_set_int (1, nnode, 0, mq_node);

  // set node flag for low quality elements

  found = 0;

  for (ielem = 1; ielem <= nelem; ++ielem)
  {
    inode1 = iniel[ielem][0];
    inode2 = iniel[ielem][1];
    inode3 = iniel[ielem][2];
    inode4 = iniel[ielem][3];

    ug3_dh_ang_w (x[inode1][0], x[inode1][1], x[inode1][2],
                  x[inode2][0], x[inode2][1], x[inode2][2],
                  x[inode3][0], x[inode3][1], x[inode3][2],
                  x[inode4][0], x[inode4][1], x[inode4][2],
                  &w21, &w31, &w32, &w41, &w42, &w43);

    w = MAX (w21, w31);
    w = MAX (w41, w);
    w = MAX (w32, w);
    w = MAX (w42, w);
    w = MAX (w43, w);
    w = w + cosqmaxs;

    if (w > dc0)
    {
      found = 1;

      mq_node[inode1] = 1;
      mq_node[inode2] = 1;
      mq_node[inode3] = 1;
      mq_node[inode4] = 1;
    }
  }

  // set error flag to exit if no low-quality elements found

  if (found == 0)
    ierr = -1;

  // allocate element flag

  if (ierr == 0)
  {
    *mq_elem = (INT_1D *) ug_malloc (&ierr, (nelem+1) * sizeof (INT_1D));

    if (ierr)
    {
      ug_error_message ("*** ERROR 100346 : unable to allocate required memory ***");
      ierr = 100346;
    }
  }

  // initialize element flag

  if (ierr == 0)
    ug_set_int (1, nelem, 0, *mq_elem);

  // create list of elements surrounding each node

  if (ierr == 0)
    ierr = ug3_ielin (1, nelem, &nelpntd, nnode, &nelpnt,
                      iniel, NULL, &ielin, lielin);

  // exit on error

  if (ierr)
  {
    ug_free (ielin);
    ug_free (lielin);
    ug_free (mq_node);

    return (ierr);
  }

  // flag nlayer_q layers of elements around low-quality elements

  for (layer = 1; layer <= nlayer_q; ++layer)
  {
    for (inode = 1; inode <= nnode; ++inode)
    {
      if (mq_node[inode] == layer)
      {
        loc1 = lielin[inode];
        loc2 = lielin[inode+1];

        for (loc = loc1; loc < loc2; ++loc)
        {
          ielem = ielin[loc];

          if ((*mq_elem)[ielem] == 0)
          {
            (*mq_elem)[ielem] = 1;

            for (ieln = 0; ieln < 4; ++ieln)
            {
              jnode = iniel[ielem][ieln];

              if (mq_node[jnode] == 0)
                mq_node[jnode] = layer+1;
            }
          }
        }
      }
    }
  }

  // reset element flag

  // if mq_elem = 1 then the element is within low-quality region
  // if mq_elem = 0 then the element is not within low-quality region

  for (ielem = 1; ielem <= nelem; ++ielem)
  {
    found = 0;

    ieln = 0;

    while (ieln < 4 && found == 0)
    {
      inode = iniel[ielem][ieln];

      found = (mq_node[inode]) ? 1: 0;

      ++ieln;
    }

    (*mq_elem)[ielem] = (found) ? 1: 0;
  }

  // free temporary space

  ug_free (ielin);
  ug_free (lielin);
  ug_free (mq_node);

  return (0);
}
