#include "UG3_LIB.h"

/*
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_idiel.c,v 1.19 2021/11/26 09:51:27 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

INT_ ug3_idiel (
  INT_ nbface,
  INT_ nelem,
  INT_ nnode,
  INT_1D * ibcibf,
  INT_1D ** idiel,
  INT_1D * ielidibf,
  INT_3D * inibf,
  INT_4D * iniel)
{
  // Set volume ID flag.

  INT_4D *ieliel = NULL;
  INT_1D *list = NULL;

  INT_ ibface, ichk, ichk1, ichk2, id, ielem, iit, j, jchk, jelem;
  INT_ found = 0;
  INT_ ierr = 0;

  // check if element ID flag is already set

  if (*idiel == NULL) {

    // allocate element ID flag and search list

    *idiel = (INT_1D *) ug_malloc (&ierr, (nelem+1) * sizeof (INT_1D));
    list = (INT_1D *) ug_malloc (&ierr, (nelem+1) * sizeof (INT_1D));

    if (ierr) {
      ug_free (list);
      ug_error_message ("*** ERROR 100311 : unable to allocate required memory ***");
      return 100311;
    }

    // set all element IDs to 1

    ug_set_int (1, nelem, 1, *idiel);

    // check for transparent boundary surface faces

    ibface = 1;

    do {

      if (ibcibf[ibface] == TRANSP_UG3_GBC ||
          ibcibf[ibface] == TRANSP_INTRNL_UG3_GBC ||
          ibcibf[ibface] == TMP_TRANSP_UG3_GBC ||
          ibcibf[ibface] == TMP_TRANSP_INTRNL_UG3_GBC) found = 1;

      ibface++;
    }
    while (ibface <= nbface && found == 0);

    // if transparent face found or boundary face element ID map is required then 

    if (found || ielidibf) {

      // get element neighbor map

      ierr = ug3_ieliel2 (nelem, nnode, iniel, &ieliel);

      if (ierr) {
        ug_free (ieliel);
        ug_free (list);
        return ierr;
      }

      // set element neighbor map at boundary faces

      ierr = ug3_ieliel2b (nbface, nelem, nnode, ibcibf, inibf, iniel, ieliel);

      if (ierr) {
        ug_free (ieliel);
        ug_free (list);
        return ierr;
      }
    }

    // if no transparent face found and boundary face element ID map is not
    // required then exit

    if (found == 0 && ielidibf == NULL) {

      ug_free (list);

      return 0;
    }

    // if transparent face found then set all IDs based on region

    if (found) {

      ug_set_int (1, nelem, 0, *idiel);

      id = 1;

      ielem = 1;

      do {

        ichk = 1;

        list[ichk] = ielem;

        (*idiel)[ielem] = id;

        // iterative tree-search for elements to add to current ID group

        ichk2 = 0;

        iit = 1;

        do {

          ichk1 = ichk2+1;
          ichk2 = ichk;

          // loop over unset neighbors of elements added during last iteration

          jchk = ichk1;

          do {

            // get element to check

            jelem = list[jchk];

            // loop over element neighbors

            j = 0;

            do {

              ielem = ieliel[jelem][j];

              // if element neighbor exists and does not have an ID then set it

              if (ielem > 0 && (*idiel)[ielem] == 0) {

                ichk++;

                list[ichk] = ielem;

                (*idiel)[ielem] = id;
              }

              j++;
            }
            while (j < 4);

            // end of loop over new elements to be deleted

            jchk++;
          }
          while (jchk <= ichk2);

          // end of tree search iteration loop

          iit++;
        }
        while (iit <= nelem && ichk > ichk2);

        ielem = 1;

        while (ielem <= nelem && (*idiel)[ielem]) {
          ++ielem;
        }

        id++;
      }
      while (id <= nelem && ielem <= nelem);
    }
  }

  // set boundary face element ID map
  // note that this will only be used for BL generating boundary faces
  // as it is not valid for closed transparent boundary surfaces

  if (ielidibf) {

    // set element containing each boundary face

    ierr = ug3_ielibf2 (nbface, nelem, ieliel, ielidibf);

    // set element ID for each boundary face

    if (ierr == 0) {

      for (ibface = 1; ibface <= nbface; ibface++) {

        ielem = ielidibf[ibface];

        ielidibf[ibface] = (*idiel)[ielem];
      }
    }
  }

  ug_free (ieliel);
  ug_free (list);

  return ierr;
}
