#include "UG3_LIB.h"

INT_ ug3_face_map (INT_ merr,
                   INT_ nbface,
                   INT_ nquad,
                   INT_ nelem,
                   INT_ nelemc5,
                   INT_ nelemc6,
                   INT_ nelemc8,
                   INT_ nsplit,
                   INT_ *ntface,
                   INT_ *nqface,
                   INT_1D * ibcibf,
                   INT_3D * inibf,
                   INT_4D * iniq,
                   INT_4D * iniel,
                   INT_5D * inielc5,
                   INT_6D * inielc6,
                   INT_8D * inielc8,
                   INT_1D * ielc8isplit,
                   INT_9D * inisplit,
                   INT_6D ** initf,
                   INT_7D ** iniqf)
{

/*
 * Determine face map data structure for volume mesh.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_face_map.c,v 1.15 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_1D *iqfkqf = NULL;
  INT_1D *itfktf = NULL;

  INT_ dir, fn, fn1, fn2, fn3, fn4, found,
       ibface, iquad, ielem, ielem2, ielemc5, ielemc6, ielemc8,
       inode1, inode2, inode3, inode4, jnode1, jnode2, jnode3, jnode4,
       jqface, kqface, nqface0, nqfacem,
       jtface, ktface, ntface0, ntfacem, pass;
  INT_ ierr = 0;
  INT_ iqface = 0;
  INT_ itface = 0;
  INT_ ntbqface = 0;
  INT_ ntbtface = 0;

// NOT VALID if we have split face hex elements
  if (merr && nsplit)
  {
    if (ielc8isplit) ielc8isplit = NULL;
    if (inisplit) inisplit = NULL;
    ug_error_message ("*** ERROR 368 : split face hexes not allowed for face map data structure ***");
    return (368);
  }
  else if (nsplit)
    return (-1);

  // set number of transparent boundary faces

  for (ibface = 1; ibface <= nbface; ++ibface)
  {
    if (ibcibf && CHK_TRANSP_UG3_GBC (ibcibf[ibface]))
      ++ntbtface;
  }

  for (iquad = 1; iquad <= nquad; ++iquad)
  {
    if (ibcibf && CHK_TRANSP_UG3_GBC (ibcibf[nbface+iquad]))
      ++ntbqface;
  }

  // allocate space for face map data structure

  ntfacem = ntbtface + nbface + nelem*4 + nelemc5*4 + nelemc6*2;
  nqfacem = ntbqface + nquad + nelemc5 + nelemc6*3 + nelemc8*6;

  *initf = (INT_6D *) ug_malloc (&ierr, (ntfacem+1) * sizeof (INT_6D));
  *iniqf = (INT_7D *) ug_malloc (&ierr, (nqfacem+1) * sizeof (INT_7D));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100341 : unable to allocate required memory ***");
    return (100341);
  }

  // set boundary tria faces

  for (ibface = 1; ibface <= nbface; ++ibface)
  {
    ++itface;

    (*initf)[itface][0] = inibf[ibface][0];
    (*initf)[itface][1] = inibf[ibface][1];
    (*initf)[itface][2] = inibf[ibface][2];
    (*initf)[itface][3] = -ibface;

    if (ibcibf && CHK_TRANSP_UG3_GBC (ibcibf[ibface]))
    {
      (*initf)[itface][5] = ibface;

      ++itface;

      (*initf)[itface][0] = inibf[ibface][0];
      (*initf)[itface][1] = inibf[ibface][1];
      (*initf)[itface][2] = inibf[ibface][2];
      (*initf)[itface][3] = -ibface;
      (*initf)[itface][5] = ibface;
    }
    else
      (*initf)[itface][5] = 0;
  }

  // set boundary quad faces

  for (iquad = 1; iquad <= nquad; ++iquad)
  {
    ++iqface;

    (*iniqf)[iqface][0] = iniq[iquad][0];
    (*iniqf)[iqface][1] = iniq[iquad][1];
    (*iniqf)[iqface][2] = iniq[iquad][2];
    (*iniqf)[iqface][3] = iniq[iquad][3];
    (*iniqf)[iqface][4] = -iquad;

    if (ibcibf && CHK_TRANSP_UG3_GBC (ibcibf[nbface+iquad]))
    {
      (*iniqf)[iqface][6] = iquad;

      ++iqface;

      (*iniqf)[iqface][0] = iniq[iquad][0];
      (*iniqf)[iqface][1] = iniq[iquad][1];
      (*iniqf)[iqface][2] = iniq[iquad][2];
      (*iniqf)[iqface][3] = iniq[iquad][3];
      (*iniqf)[iqface][4] = -iquad;
      (*iniqf)[iqface][6] = iquad;
    }
    else
      (*iniqf)[iqface][6] = 0;
  }

  // set tet element faces

  for (ielem = 1; ielem <= nelem; ++ielem)
  {
    ++itface;

    (*initf)[itface][0] = iniel[ielem][0];
    (*initf)[itface][1] = iniel[ielem][1];
    (*initf)[itface][2] = iniel[ielem][3];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++itface;

    (*initf)[itface][0] = iniel[ielem][1];
    (*initf)[itface][1] = iniel[ielem][2];
    (*initf)[itface][2] = iniel[ielem][3];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++itface;

    (*initf)[itface][0] = iniel[ielem][3];
    (*initf)[itface][1] = iniel[ielem][2];
    (*initf)[itface][2] = iniel[ielem][0];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++itface;

    (*initf)[itface][0] = iniel[ielem][0];
    (*initf)[itface][1] = iniel[ielem][2];
    (*initf)[itface][2] = iniel[ielem][1];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;
  }

  // set pyramid element faces

  for (ielemc5 = 1; ielemc5 <= nelemc5; ++ielemc5)
  {
    ielem = nelem+ielemc5;

    ++itface;

    (*initf)[itface][0] = inielc5[ielemc5][4];
    (*initf)[itface][1] = inielc5[ielemc5][1];
    (*initf)[itface][2] = inielc5[ielemc5][2];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++itface;

    (*initf)[itface][0] = inielc5[ielemc5][4];
    (*initf)[itface][1] = inielc5[ielemc5][2];
    (*initf)[itface][2] = inielc5[ielemc5][3];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++itface;

    (*initf)[itface][0] = inielc5[ielemc5][3];
    (*initf)[itface][1] = inielc5[ielemc5][2];
    (*initf)[itface][2] = inielc5[ielemc5][0];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++itface;

    (*initf)[itface][0] = inielc5[ielemc5][0];
    (*initf)[itface][1] = inielc5[ielemc5][2];
    (*initf)[itface][2] = inielc5[ielemc5][1];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc5[ielemc5][0];
    (*iniqf)[iqface][1] = inielc5[ielemc5][1];
    (*iniqf)[iqface][2] = inielc5[ielemc5][4];
    (*iniqf)[iqface][3] = inielc5[ielemc5][3];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;
  }

  // set prism element faces

  for (ielemc6 = 1; ielemc6 <= nelemc6; ++ielemc6)
  {
    ielem = nelem+nelemc5+ielemc6;

    ++itface;

    (*initf)[itface][0] = inielc6[ielemc6][3];
    (*initf)[itface][1] = inielc6[ielemc6][4];
    (*initf)[itface][2] = inielc6[ielemc6][5];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++itface;

    (*initf)[itface][0] = inielc6[ielemc6][0];
    (*initf)[itface][1] = inielc6[ielemc6][2];
    (*initf)[itface][2] = inielc6[ielemc6][1];
    (*initf)[itface][3] = ielem;
    (*initf)[itface][5] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc6[ielemc6][0];
    (*iniqf)[iqface][1] = inielc6[ielemc6][1];
    (*iniqf)[iqface][2] = inielc6[ielemc6][4];
    (*iniqf)[iqface][3] = inielc6[ielemc6][3];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc6[ielemc6][1];
    (*iniqf)[iqface][1] = inielc6[ielemc6][2];
    (*iniqf)[iqface][2] = inielc6[ielemc6][5];
    (*iniqf)[iqface][3] = inielc6[ielemc6][4];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc6[ielemc6][3];
    (*iniqf)[iqface][1] = inielc6[ielemc6][5];
    (*iniqf)[iqface][2] = inielc6[ielemc6][2];
    (*iniqf)[iqface][3] = inielc6[ielemc6][0];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;
  }

  // set hex element faces

  for (ielemc8 = 1; ielemc8 <= nelemc8; ++ielemc8)
  {
    ielem = nelem+nelemc5+nelemc6+ielemc8;

    ++iqface;

    (*iniqf)[iqface][0] = inielc8[ielemc8][0];
    (*iniqf)[iqface][1] = inielc8[ielemc8][1];
    (*iniqf)[iqface][2] = inielc8[ielemc8][5];
    (*iniqf)[iqface][3] = inielc8[ielemc8][4];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc8[ielemc8][1];
    (*iniqf)[iqface][1] = inielc8[ielemc8][2];
    (*iniqf)[iqface][2] = inielc8[ielemc8][6];
    (*iniqf)[iqface][3] = inielc8[ielemc8][5];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc8[ielemc8][2];
    (*iniqf)[iqface][1] = inielc8[ielemc8][3];
    (*iniqf)[iqface][2] = inielc8[ielemc8][7];
    (*iniqf)[iqface][3] = inielc8[ielemc8][6];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc8[ielemc8][4];
    (*iniqf)[iqface][1] = inielc8[ielemc8][7];
    (*iniqf)[iqface][2] = inielc8[ielemc8][3];
    (*iniqf)[iqface][3] = inielc8[ielemc8][0];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc8[ielemc8][4];
    (*iniqf)[iqface][1] = inielc8[ielemc8][5];
    (*iniqf)[iqface][2] = inielc8[ielemc8][6];
    (*iniqf)[iqface][3] = inielc8[ielemc8][7];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;

    ++iqface;

    (*iniqf)[iqface][0] = inielc8[ielemc8][3];
    (*iniqf)[iqface][1] = inielc8[ielemc8][2];
    (*iniqf)[iqface][2] = inielc8[ielemc8][1];
    (*iniqf)[iqface][3] = inielc8[ielemc8][0];
    (*iniqf)[iqface][4] = ielem;
    (*iniqf)[iqface][6] = 0;
  }

  // set total number of tria and quad faces

  ntfacem = itface;
  nqfacem = iqface;

  // reset tria face connectivity so that node ordering is sequential

  for (itface = 1; itface <= ntfacem; ++itface)
  {
    dir = 0;

    fn1 = 0;

    do
    {
      fn2 = (fn1 < 2) ? fn1+1: 0;
      fn3 = (fn2 < 2) ? fn2+1: 0;

      inode1 = (*initf)[itface][fn1];
      inode2 = (*initf)[itface][fn2];
      inode3 = (*initf)[itface][fn3];

      if (inode1 < inode2 && inode1 < inode3)
        dir = (inode2 < inode3) ? 1: -1;

      ++fn1;
    }
    while (fn1 < 3 && dir == 0);

    if (dir == 1)
    {
      (*initf)[itface][0] = inode1;
      (*initf)[itface][1] = inode2;
      (*initf)[itface][2] = inode3;
      (*initf)[itface][4] = 0;
    }
    else
    {
      (*initf)[itface][0] = inode1;
      (*initf)[itface][1] = inode3;
      (*initf)[itface][2] = inode2;
      (*initf)[itface][4] = 1;
    }
  }

  // sort tria faces sequentially

  if (ntfacem)
    qsort (&((*initf)[1][0]), (size_t) ntfacem, sizeof (INT_6D), ug3_tface_sort);

  // reset quad face connectivity so that node ordering is sequential

  for (iqface = 1; iqface <= nqfacem; ++iqface)
  {
    dir = 0;

    fn1 = 0;

    do
    {
      fn2 = (fn1 < 3) ? fn1+1: 0;
      fn3 = (fn2 < 3) ? fn2+1: 0;
      fn4 = (fn3 < 3) ? fn3+1: 0;

      inode1 = (*iniqf)[iqface][fn1];
      inode2 = (*iniqf)[iqface][fn2];
      inode3 = (*iniqf)[iqface][fn3];
      inode4 = (*iniqf)[iqface][fn4];

      if (inode1 < inode2 && inode1 < inode3 && inode1 < inode4)
        dir = (inode2 < inode4) ? 1: -1;

      ++fn1;
    }
    while (fn1 < 4 && dir == 0);

    if (dir == 1)
    {
      (*iniqf)[iqface][0] = inode1;
      (*iniqf)[iqface][1] = inode2;
      (*iniqf)[iqface][2] = inode3;
      (*iniqf)[iqface][3] = inode4;
      (*iniqf)[iqface][5] = 0;
    }
    else
    {
      (*iniqf)[iqface][0] = inode1;
      (*iniqf)[iqface][1] = inode4;
      (*iniqf)[iqface][2] = inode3;
      (*iniqf)[iqface][3] = inode2;
      (*iniqf)[iqface][5] = 1;
    }
  }

  // sort quad faces sequentially

  if (nqfacem)
    qsort (&((*iniqf)[1][0]), (size_t) nqfacem, sizeof (INT_7D), ug3_qface_sort);

  // combine duplicate tria faces

  jtface = 0;
  ktface = 0;

  for (itface = 1; itface <= ntfacem; ++itface)
  {
    ++jtface;

    inode1 = (*initf)[itface][0];
    inode2 = (*initf)[itface][1];
    inode3 = (*initf)[itface][2];

    ielem2 = (itface < ntfacem &&
              inode1 == (*initf)[itface+1][0] &&
              inode2 == (*initf)[itface+1][1] &&
              inode3 == (*initf)[itface+1][2]) ? (*initf)[itface+1][3]: 0;

    if (ielem2)
    {
      fn = (*initf)[itface][4];

      (*initf)[jtface][0] = inode1;
      (*initf)[jtface][1] = inode2;
      (*initf)[jtface][2] = inode3;
      (*initf)[jtface][3+fn] = (*initf)[itface][3];
      (*initf)[jtface][4-fn] = ielem2;
      (*initf)[jtface][5] = (*initf)[itface][5];

      ++itface;
    }
    else
    {
      (*initf)[jtface][0] = -inode1;
      (*initf)[jtface][1] = inode2;
      (*initf)[jtface][2] = inode3;
      (*initf)[jtface][3] = (*initf)[itface][3];
      (*initf)[jtface][4] = (*initf)[itface][4];
      (*initf)[jtface][5] = (*initf)[itface][5];

      ++ktface;
    }
  }

  *ntface = jtface;
  ntface0 = ktface;

  // combine duplicate quad faces

  jqface = 0;
  kqface = 0;

  for (iqface = 1; iqface <= nqfacem; ++iqface)
  {
    ++jqface;

    inode1 = (*iniqf)[iqface][0];
    inode2 = (*iniqf)[iqface][1];
    inode3 = (*iniqf)[iqface][2];
    inode4 = (*iniqf)[iqface][3];

    ielem2 = (iqface < nqfacem &&
              inode1 == (*iniqf)[iqface+1][0] &&
              inode2 == (*iniqf)[iqface+1][1] &&
              inode3 == (*iniqf)[iqface+1][2] &&
              inode4 == (*iniqf)[iqface+1][3]) ? (*iniqf)[iqface+1][4]: 0;

    if (ielem2)
    {
      fn = (*iniqf)[iqface][5];

      (*iniqf)[jqface][0] = inode1;
      (*iniqf)[jqface][1] = inode2;
      (*iniqf)[jqface][2] = inode3;
      (*iniqf)[jqface][3] = inode4;
      (*iniqf)[jqface][4+fn] = (*iniqf)[iqface][4];
      (*iniqf)[jqface][5-fn] = ielem2;
      (*iniqf)[jqface][6] = (*iniqf)[iqface][6];

      ++iqface;
    }
    else
    {
      (*iniqf)[jqface][0] = -inode1;
      (*iniqf)[jqface][1] = inode2;
      (*iniqf)[jqface][2] = inode3;
      (*iniqf)[jqface][3] = inode4;
      (*iniqf)[jqface][4] = (*iniqf)[iqface][4];
      (*iniqf)[jqface][5] = (*iniqf)[iqface][5];
      (*iniqf)[jqface][6] = (*iniqf)[iqface][6];

      ++kqface;
    }
  }

  *nqface = jqface;
  nqface0 = kqface;

  // allocate temporary faces for split quad faces

  itfktf = (INT_1D *) ug_malloc (&ierr, (ntface0+1) * sizeof (INT_1D));
  iqfkqf = (INT_1D *) ug_malloc (&ierr, (nqface0+1) * sizeof (INT_1D));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100342 : unable to allocate required memory ***");
    ug_free (iqfkqf);
    ug_free (itfktf);
    return (100342);
  }

  // set list of unmatched tria faces

  ktface = 0;

  for (itface = 1; itface <= *ntface; ++itface)
  {
    if ((*initf)[itface][0] < 0)
    {
      (*initf)[itface][0] = - (*initf)[itface][0];

      ++ktface;

      itfktf[ktface] = itface;
    }
  }

  // set list of unmatched quad faces

  kqface = 0;

  for (iqface = 1; iqface <= *nqface; ++iqface)
  {
    if ((*iniqf)[iqface][0] < 0)
    {
      (*iniqf)[iqface][0] = - (*iniqf)[iqface][0];

      ++kqface;

      iqfkqf[kqface] = iqface;
    }
  }

  // find unmatched tria faces that match unmatched quad faces

  for (kqface = 1; kqface <= nqface0; ++kqface)
  {
    iqface = iqfkqf[kqface];

    jnode1 = (*iniqf)[iqface][0];
    jnode2 = (*iniqf)[iqface][1];
    jnode3 = (*iniqf)[iqface][2];
    jnode4 = (*iniqf)[iqface][3];
    ielem2 = (*iniqf)[iqface][4];

    fn1 = 0;
    fn2 = 1;
    fn3 = 2;

    found = 0;

    pass = 1;

    ktface = 1;

    do
    {
      itface = itfktf[ktface];

      if (itface)
      {
        if ((*initf)[itface][0] == jnode1)
        {
          inode2 = (*initf)[itface][1];
          inode3 = (*initf)[itface][2];

          if (pass == 1)
          {
            if ((inode2 == jnode2 && inode3 == jnode3) ||
                (inode3 == jnode2 && inode2 == jnode3))
            {
              ++found;

              jnode2 = (*iniqf)[iqface][2];
              jnode3 = (*iniqf)[iqface][3];
            }
            else if ((inode2 == jnode2 && inode3 == jnode4) ||
                     (inode3 == jnode2 && inode2 == jnode4))
            {
              ++found;

              fn1 = (jnode2 < jnode3 && jnode2 < jnode4) ? 1:
                    (jnode3 < jnode2 && jnode3 < jnode4) ? 2: 3; 
              fn2 = (fn1 < 3) ? fn1+1: 1;
              fn3 = (fn2 < 3) ? fn2+1: 1;

              jnode1 = (*iniqf)[iqface][fn1];
              jnode2 = (*iniqf)[iqface][fn2];
              jnode3 = (*iniqf)[iqface][fn3];
            }
          }

          else if ((inode2 == jnode2 && inode3 == jnode3) ||
                   (inode2 == jnode3 && inode3 == jnode2))
            ++found;

          if (pass == found)
          {
            if ((*initf)[itface][4])
            {
              (*initf)[itface][1] = inode3;
              (*initf)[itface][2] = inode2;
            }

            (*initf)[itface][4] = ielem2;
            (*initf)[itface][5] = (*iniqf)[iqface][6];

            itfktf[ktface] = 0;

            ++pass;
          }
        }
      }

      ++ktface;
    }
    while (ktface <= ntface0 && found < 2);

// NOT TRUE if we have split face hex elements
// NEED to add section that use the split data to match faces
// AND the possible matches include 1-quad to 4-quads

    if (merr && found != 2)
    {
      ug_free (iqfkqf);
      ug_free (itfktf);
      ug_error_message ("*** ERROR 365 : element connectivity found with non-matching element neighbor faces ***");
      return (365);
    }
    else if (found != 2)
    {
      ug_free (iqfkqf);
      ug_free (itfktf);
      return (-1);
    }

    (*iniqf)[iqface][0] = 0;
  }

  // remove quad faces that were matched to tria faces

  jqface = 0;

  for (iqface = 1; iqface <= *nqface; ++iqface)
  {
    if ((*iniqf)[iqface][0])
    {
      ++jqface;

      for (fn = 0; fn < 7; ++fn)
      {
        (*iniqf)[jqface][fn] = (*iniqf)[iqface][fn];
      }
    }
  }

  *nqface = jqface;

  // remove duplicate transparent tria faces and add to element face entry

  if (ntbtface)
  {
    jtface = 0;

    for (itface = 1; itface <= *ntface; ++itface)
    {
      ++jtface;

      if ((*initf)[itface][3] < 0 && (*initf)[itface][4] < 0)
      {
        ibface = (*initf)[itface][5];

        dir = 0;

        fn1 = 0;

        do
        {
          fn2 = (fn1 < 2) ? fn1+1: 0;
          fn3 = (fn2 < 2) ? fn2+1: 0;

          inode1 = inibf[ibface][fn1];
          inode2 = inibf[ibface][fn2];
          inode3 = inibf[ibface][fn3];

          if (inode1 < inode2 && inode1 < inode3)
            dir = (inode2 < inode3) ? 1: -1;

          ++fn1;
        }
        while (fn1 < 3 && dir == 0);

        ++itface;

        (*initf)[itface][5] = (dir == 1) ? ibface: -ibface;
      }

      for (fn = 0; fn < 6; ++fn)
      {
        (*initf)[jtface][fn] = (*initf)[itface][fn];
      }
    }

    *ntface = jtface;
  }

  // remove duplicate transparent quad faces and add to element face entry

  if (ntbqface)
  {
    jqface = 0;

    for (iqface = 1; iqface <= *nqface; ++iqface)
    {
      ++jqface;

      if ((*iniqf)[iqface][4] < 0 && (*iniqf)[iqface][5] < 0)
      {
        iquad = (*iniqf)[iqface][6];

        dir = 0;

        fn1 = 0;

        do
        {
          fn2 = (fn1 < 3) ? fn1+1: 0;
          fn3 = (fn2 < 3) ? fn2+1: 0;
          fn4 = (fn3 < 3) ? fn3+1: 0;

          inode1 = iniq[iquad][fn1];
          inode2 = iniq[iquad][fn2];
          inode3 = iniq[iquad][fn3];
          inode4 = iniq[iquad][fn4];

          if (inode1 < inode2 && inode1 < inode3 && inode1 < inode4)
            dir = (inode2 < inode4) ? 1: -1;

          ++fn1;
        }
        while (fn1 < 4 && dir == 0);

        ++iqface;

        (*iniqf)[iqface][6] = (dir == 1) ? iquad: -iquad;
      }

      for (fn = 0; fn < 7; ++fn)
      {
        (*iniqf)[jqface][fn] = (*iniqf)[iqface][fn];
      }
    }

    *nqface = jqface;
  }

  ug_free (iqfkqf);
  ug_free (itfktf);

  // reallocate face map data structure to exact size

  *initf = (INT_6D *) ug_realloc (&ierr, *initf, ((*ntface)+1) * sizeof (INT_6D));
  *iniqf = (INT_7D *) ug_realloc (&ierr, *iniqf, ((*nqface)+1) * sizeof (INT_7D));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100343 : unable to allocate required memory ***");
    return (100343);
  }

  return (0);
}

int ug3_tface_sort (const void *a1_, const void *a2_)
{
  // tria face sort for qsort routine

  INT_6D *a1;
  INT_6D *a2;

  int i = 0;

  a1 = (INT_6D *) a1_;
  a2 = (INT_6D *) a2_;

       if (a1[0][0] < a2[0][0]) i = -1;
  else if (a1[0][0] > a2[0][0]) i =  1;
  else if (a1[0][1] < a2[0][1]) i = -1;
  else if (a1[0][1] > a2[0][1]) i =  1;
  else if (a1[0][2] < a2[0][2]) i = -1;
  else if (a1[0][2] > a2[0][2]) i =  1;
  else if (a1[0][3] < a2[0][3]) i = -1;
  else if (a1[0][3] > a2[0][3]) i =  1;

  return (i);
}

int ug3_qface_sort (const void *a1_, const void *a2_)
{
  // quad face sort for qsort routine

  INT_7D *a1;
  INT_7D *a2;

  int i = 0;

  a1 = (INT_7D *) a1_;
  a2 = (INT_7D *) a2_;

       if (a1[0][0] < a2[0][0]) i = -1;
  else if (a1[0][0] > a2[0][0]) i =  1;
  else if (a1[0][1] < a2[0][1]) i = -1;
  else if (a1[0][1] > a2[0][1]) i =  1;
  else if (a1[0][2] < a2[0][2]) i = -1;
  else if (a1[0][2] > a2[0][2]) i =  1;
  else if (a1[0][3] < a2[0][3]) i = -1;
  else if (a1[0][3] > a2[0][3]) i =  1;
  else if (a1[0][4] < a2[0][4]) i = -1;
  else if (a1[0][4] > a2[0][4]) i =  1;

  return (i);
}
