#include "UG3_LIB.h"

/*
 * Routines to compute edge length information for tet, prism, pyramid, and/or
 * hex elements.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_dxchk.c,v 1.3 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

double ug3_dxref
 (INT_ nn,
  INT_1D * in,
  DOUBLE_3D * x)

{

  // Get bounding-box reference length.

  double xmin, xmax;
  double dxref = 0;

  INT_ i, j;

  if (nn < 2)
    return (0);

  for (j = 0; j < 3; ++j)
  {
    xmax = MAX (x[in[0]][j], x[in[1]][j]);
    xmin = MIN (x[in[0]][j], x[in[1]][j]);

    for (i = 2; i < nn; ++i)
    {
      xmax = MAX (x[in[i]][j], xmax);
      xmin = MIN (x[in[i]][j], xmin);
    }

    dxref = MAX (xmax - xmin, dxref);
  }

  return (dxref);

}

double ug3_dxs
 (INT_ inode1,
  INT_ inode2,
  DOUBLE_3D * x)

{

  // Get distance squared.

  double dx1, dx2, dx3, dxs;

  dx1 = x[inode2][0] - x[inode1][0];
  dx2 = x[inode2][1] - x[inode1][1];
  dx3 = x[inode2][2] - x[inode1][2];

  dxs = dx1 * dx1 + dx2 * dx2 + dx3 * dx3;

  return (dxs);

}

double ug3_tet_dxmins (INT_ inode1,
                       INT_ inode2,
                       INT_ inode3,
                       INT_ inode4,
                       DOUBLE_3D * x)
{

  // Determine the minimum edge-length squared.

  double dxmins;

  dxmins = ug3_dxs (inode1, inode2, x);
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode3, x));
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode4, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode3, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode4, x));
  dxmins = MIN (dxmins, ug3_dxs (inode3, inode4, x));

  return (dxmins);
}

double ug3_pyramid_dxmins (INT_ inode1,
                           INT_ inode2,
                           INT_ inode3,
                           INT_ inode4,
                           INT_ inode5,
                           DOUBLE_3D * x)
{

  // Determine the minimum edge-length squared.

  double dxmins;

  dxmins = ug3_dxs (inode1, inode2, x);
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode3, x));
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode4, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode3, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode5, x));
  dxmins = MIN (dxmins, ug3_dxs (inode3, inode4, x));
  dxmins = MIN (dxmins, ug3_dxs (inode3, inode5, x));
  dxmins = MIN (dxmins, ug3_dxs (inode4, inode5, x));

  return (dxmins);
}

double ug3_prism_dxmins (INT_ inode1,
                         INT_ inode2,
                         INT_ inode3,
                         INT_ inode4,
                         INT_ inode5,
                         INT_ inode6,
                         DOUBLE_3D * x)
{

  // Determine the minimum edge-length squared.

  double dxmins;

  dxmins = ug3_dxs (inode1, inode2, x);
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode3, x));
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode4, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode3, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode5, x));
  dxmins = MIN (dxmins, ug3_dxs (inode3, inode6, x));
  dxmins = MIN (dxmins, ug3_dxs (inode4, inode5, x));
  dxmins = MIN (dxmins, ug3_dxs (inode4, inode6, x));
  dxmins = MIN (dxmins, ug3_dxs (inode5, inode6, x));

  return (dxmins);
}

double ug3_hex_dxmins (INT_ inode1,
                       INT_ inode2,
                       INT_ inode3,
                       INT_ inode4,
                       INT_ inode5,
                       INT_ inode6,
                       INT_ inode7,
                       INT_ inode8,
                       DOUBLE_3D * x)
{

  // Determine the minimum edge-length squared.

  double dxmins;

  dxmins = ug3_dxs (inode1, inode2, x);
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode4, x));
  dxmins = MIN (dxmins, ug3_dxs (inode1, inode5, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode3, x));
  dxmins = MIN (dxmins, ug3_dxs (inode2, inode6, x));
  dxmins = MIN (dxmins, ug3_dxs (inode3, inode4, x));
  dxmins = MIN (dxmins, ug3_dxs (inode3, inode7, x));
  dxmins = MIN (dxmins, ug3_dxs (inode4, inode8, x));
  dxmins = MIN (dxmins, ug3_dxs (inode5, inode6, x));
  dxmins = MIN (dxmins, ug3_dxs (inode5, inode8, x));
  dxmins = MIN (dxmins, ug3_dxs (inode6, inode7, x));
  dxmins = MIN (dxmins, ug3_dxs (inode7, inode8, x));

  return (dxmins);
}

INT_ ug3_tet_dxchk (INT_ ielem,
                    INT_4D * iniel,
                    double tol,
                    double *dxmin,
                    double *dxtol,
                    DOUBLE_3D * x)
{
  // check minimum-edge-lengths for tet elements

  double dxmins;

  *dxtol = tol * ug3_dxref (4, iniel[ielem], x);

  dxmins = ug3_tet_dxmins (iniel[ielem][0], iniel[ielem][1], iniel[ielem][2], iniel[ielem][3], x);

  *dxmin = sqrt (dxmins);

  if (*dxmin < *dxtol)
    return (1);

  return (0);
}

INT_ ug3_pyramid_dxchk (INT_ ielemc5,
                        INT_5D * inielc5,
                        double tol,
                        double *dxmin,
                        double *dxtol,
                        DOUBLE_3D * x)
{
  // check minimum-edge-lengths for pyramid elements

  double dxmins;

  *dxtol = tol * ug3_dxref (5, inielc5[ielemc5], x);

  dxmins = ug3_pyramid_dxmins (inielc5[ielemc5][0], inielc5[ielemc5][1], inielc5[ielemc5][2], inielc5[ielemc5][3], inielc5[ielemc5][4], x);

  *dxmin = sqrt (dxmins);

  if (*dxmin < *dxtol)
    return (1);

  return (0);
}

INT_ ug3_prism_dxchk (INT_ ielemc6,
                      INT_6D * inielc6,
                      double tol,
                      double *dxmin,
                      double *dxtol,
                      DOUBLE_3D * x)
{
  // check minimum-edge-lengths for prism elements

  double dxmins;

  *dxtol = tol * ug3_dxref (6, inielc6[ielemc6], x);

  dxmins = ug3_prism_dxmins (inielc6[ielemc6][0], inielc6[ielemc6][1], inielc6[ielemc6][2], inielc6[ielemc6][3], inielc6[ielemc6][4], inielc6[ielemc6][5], x);

  *dxmin = sqrt (dxmins);

  if (*dxmin < *dxtol)
    return (1);

  return (0);
}

INT_ ug3_hex_dxchk (INT_ ielemc8,
                    INT_8D * inielc8,
                    double tol,
                    double *dxmin,
                    double *dxtol,
                    DOUBLE_3D * x)
{
  // check minimum-edge-lengths for hex elements

  double dxmins;

  *dxtol = tol * ug3_dxref (8, inielc8[ielemc8], x);

  dxmins = ug3_hex_dxmins (inielc8[ielemc8][0], inielc8[ielemc8][1], inielc8[ielemc8][2], inielc8[ielemc8][3], inielc8[ielemc8][4], inielc8[ielemc8][5], inielc8[ielemc8][6], inielc8[ielemc8][7], x);

  *dxmin = sqrt (dxmins);

  if (*dxmin < *dxtol)
    return (1);

  return (0);
}
