#include "UG3_LIB.h"

/*
 * Determine dihedral element angle functions.
 *
 * angleAB	: dihedral angle between two faces sharing edge AB
 *
 * wAB = cos (angleAB) = A1.A2  / ( |A1|.|A2| )
 *
 * angleAB	: dihedral angle between two faces sharing edge AB
 * A1           : area for one of the two faces
 * A2           : area for one of the two faces
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_dh_ang_w.c,v 1.7 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

void ug3_tet_dh_ang_w (INT_ inode1,
                       INT_ inode2,
                       INT_ inode3,
                       INT_ inode4,
                       double *w21,
                       double *w31,
                       double *w32,
                       double *w41,
                       double *w42,
                       double *w43, 
                       DOUBLE_3D * x)
{

  double a124s, a1241, a1242, a1243,
         a132s, a1321, a1322, a1323,
         a143s, a1431, a1432, a1433,
         a234s, a2341, a2342, a2343,
         dc1,
         dx211, dx212, dx213,
         dx311, dx312, dx313,
         dx411, dx412, dx413,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43;

  dc1 = 1.0;

  // coordinates

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];

  // delta vectors

  dx211 = x21 - x11;
  dx212 = x22 - x12;
  dx213 = x23 - x13;
  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx411 = x41 - x11;
  dx412 = x42 - x12;
  dx413 = x43 - x13;

  // area vectors

  a1241 = dx212 * dx413 - dx213 * dx412;
  a1242 = dx213 * dx411 - dx211 * dx413;
  a1243 = dx211 * dx412 - dx212 * dx411;
  a1321 = dx312 * dx213 - dx313 * dx212;
  a1322 = dx313 * dx211 - dx311 * dx213;
  a1323 = dx311 * dx212 - dx312 * dx211;
  a1431 = dx412 * dx313 - dx413 * dx312;
  a1432 = dx413 * dx311 - dx411 * dx313;
  a1433 = dx411 * dx312 - dx412 * dx311;
  a2341 = - a1241 - a1321 - a1431;
  a2342 = - a1242 - a1322 - a1432;
  a2343 = - a1243 - a1323 - a1433;

  // area dot products

  a124s = a1241 * a1241 + a1242 * a1242 + a1243 * a1243;
  a132s = a1321 * a1321 + a1322 * a1322 + a1323 * a1323;
  a143s = a1431 * a1431 + a1432 * a1432 + a1433 * a1433;
  a234s = a2341 * a2341 + a2342 * a2342 + a2343 * a2343;

  // angle function for each edge

  *w21 = (- a1241 * a1321 - a1242 * a1322 - a1243 * a1323) / sqrt (a124s * a132s);
  *w31 = (- a1431 * a1321 - a1432 * a1322 - a1433 * a1323) / sqrt (a143s * a132s);
  *w32 = (- a2341 * a1321 - a2342 * a1322 - a2343 * a1323) / sqrt (a234s * a132s);
  *w41 = (- a1431 * a1241 - a1432 * a1242 - a1433 * a1243) / sqrt (a143s * a124s);
  *w42 = (- a2341 * a1241 - a2342 * a1242 - a2343 * a1243) / sqrt (a234s * a124s);
  *w43 = (- a2341 * a1431 - a2342 * a1432 - a2343 * a1433) / sqrt (a234s * a143s);

  *w21 = MAX (MIN (*w21, dc1), -dc1);
  *w31 = MAX (MIN (*w31, dc1), -dc1);
  *w32 = MAX (MIN (*w32, dc1), -dc1);
  *w41 = MAX (MIN (*w41, dc1), -dc1);
  *w42 = MAX (MIN (*w42, dc1), -dc1);
  *w43 = MAX (MIN (*w43, dc1), -dc1);

  return;
}

void ug3_pyramid_dh_ang_w (INT_ inode1,
                           INT_ inode2,
                           INT_ inode3,
                           INT_ inode4,
                           INT_ inode5,
                           double *w21, 
                           double *w31,
                           double *w32,
                           double *w41,
                           double *w43,
                           double *w52,
                           double *w53,
                           double *w54,
                           DOUBLE_3D * x)
{

  double a132s, a1321, a1322, a1323,
         a143s, a1431, a1432, a1433,
         a235s, a2351, a2352, a2353,
         a345s, a3451, a3452, a3453,
         a1254s, a12541, a12542, a12543,
         dc1,
         dx311, dx312, dx313,
         dx321, dx322, dx323,
         dx421, dx422, dx423,
         dx431, dx432, dx433,
         dx511, dx512, dx513,
         dx531, dx532, dx533,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53;

  dc1 = 1.0;

  // coordinates

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];

  // delta vectors

  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx321 = x31 - x21;
  dx322 = x32 - x22;
  dx323 = x33 - x23;
  dx421 = x41 - x21;
  dx422 = x42 - x22;
  dx423 = x43 - x23;
  dx431 = x41 - x31;
  dx432 = x42 - x32;
  dx433 = x43 - x33;
  dx511 = x51 - x11;
  dx512 = x52 - x12;
  dx513 = x53 - x13;
  dx531 = x51 - x31;
  dx532 = x52 - x32;
  dx533 = x53 - x33;

  // area vectors

  a1321 = dx322 * dx313 - dx323 * dx312;
  a1322 = dx323 * dx311 - dx321 * dx313;
  a1323 = dx321 * dx312 - dx322 * dx311;
  a1431 = dx432 * dx313 - dx433 * dx312;
  a1432 = dx433 * dx311 - dx431 * dx313;
  a1433 = dx431 * dx312 - dx432 * dx311;
  a2351 = dx322 * dx533 - dx323 * dx532;
  a2352 = dx323 * dx531 - dx321 * dx533;
  a2353 = dx321 * dx532 - dx322 * dx531;
  a3451 = dx432 * dx533 - dx433 * dx532;
  a3452 = dx433 * dx531 - dx431 * dx533;
  a3453 = dx431 * dx532 - dx432 * dx531;
  a12541 = dx512 * dx423 - dx513 * dx422;
  a12542 = dx513 * dx421 - dx511 * dx423;
  a12543 = dx511 * dx422 - dx512 * dx421;

  // area dot products

  a132s = a1321 * a1321 + a1322 * a1322 + a1323 * a1323;
  a143s = a1431 * a1431 + a1432 * a1432 + a1433 * a1433;
  a235s = a2351 * a2351 + a2352 * a2352 + a2353 * a2353;
  a345s = a3451 * a3451 + a3452 * a3452 + a3453 * a3453;
  a1254s = a12541 * a12541 + a12542 * a12542 + a12543 * a12543;

  // angle function for each edge

  *w21 = (- a1321 * a12541 - a1322 * a12542 - a1323 * a12543) / sqrt (a132s * a1254s);
  *w31 = (- a1431 * a1321 - a1432 * a1322 - a1433 * a1323) / sqrt (a143s * a132s);
  *w32 = (- a1321 * a2351 - a1322 * a2352 - a1323 * a2353) / sqrt (a132s * a235s);
  *w41 = (- a1431 * a12541 - a1432 * a12542 - a1433 * a12543) / sqrt (a143s * a1254s);
  *w43 = (- a1431 * a3451 - a1432 * a3452 - a1433 * a3453) / sqrt (a143s * a345s);
  *w52 = (- a2351 * a12541 - a2352 * a12542 - a2353 * a12543) / sqrt (a235s * a1254s);
  *w53 = (- a2351 * a3451 - a2352 * a3452 - a2353 * a3453) / sqrt (a235s * a345s);
  *w54 = (- a3451 * a12541 - a3452 * a12542 - a3453 * a12543) / sqrt (a345s * a1254s);

  *w21 = MAX (MIN (*w21, dc1), -dc1);
  *w31 = MAX (MIN (*w31, dc1), -dc1);
  *w32 = MAX (MIN (*w32, dc1), -dc1);
  *w41 = MAX (MIN (*w41, dc1), -dc1);
  *w43 = MAX (MIN (*w43, dc1), -dc1);
  *w52 = MAX (MIN (*w52, dc1), -dc1);
  *w53 = MAX (MIN (*w53, dc1), -dc1);
  *w54 = MAX (MIN (*w54, dc1), -dc1);

  return;
}

void ug3_prism_dh_ang_w (INT_ inode1,
                         INT_ inode2,
                         INT_ inode3,
                         INT_ inode4,
                         INT_ inode5,
                         INT_ inode6,
                         double *w21, 
                         double *w31,
                         double *w32,
                         double *w41,
                         double *w52,
                         double *w54,
                         double *w63,
                         double *w64,
                         double *w65,
                         DOUBLE_3D * x)
{

  double a132s, a1321, a1322, a1323,
         a456s, a4561, a4562, a4563,
         a1254s, a12541, a12542, a12543,
         a2365s, a23651, a23652, a23653,
         a3146s, a31461, a31462, a31463,
         dc1,
         dx311, dx312, dx313,
         dx321, dx322, dx323,
         dx421, dx422, dx423,
         dx431, dx432, dx433,
         dx511, dx512, dx513,
         dx531, dx532, dx533,
         dx541, dx542, dx543,
         dx611, dx612, dx613,
         dx621, dx622, dx623,
         dx641, dx642, dx643,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53,
         x61, x62, x63;

  dc1 = 1.0;

  // coordinates

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];
  x61 = x[inode6][0];
  x62 = x[inode6][1];
  x63 = x[inode6][2];

  // delta vectors

  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx321 = x31 - x21;
  dx322 = x32 - x22;
  dx323 = x33 - x23;
  dx421 = x41 - x21;
  dx422 = x42 - x22;
  dx423 = x43 - x23;
  dx431 = x41 - x31;
  dx432 = x42 - x32;
  dx433 = x43 - x33;
  dx511 = x51 - x11;
  dx512 = x52 - x12;
  dx513 = x53 - x13;
  dx531 = x51 - x31;
  dx532 = x52 - x32;
  dx533 = x53 - x33;
  dx541 = x51 - x41;
  dx542 = x52 - x42;
  dx543 = x53 - x43;
  dx611 = x61 - x11;
  dx612 = x62 - x12;
  dx613 = x63 - x13;
  dx621 = x61 - x21;
  dx622 = x62 - x22;
  dx623 = x63 - x23;
  dx641 = x61 - x41;
  dx642 = x62 - x42;
  dx643 = x63 - x43;

  // area vectors

  a1321 = dx322 * dx313 - dx323 * dx312;
  a1322 = dx323 * dx311 - dx321 * dx313;
  a1323 = dx321 * dx312 - dx322 * dx311;
  a4561 = dx542 * dx643 - dx543 * dx642;
  a4562 = dx543 * dx641 - dx541 * dx643;
  a4563 = dx541 * dx642 - dx542 * dx641;
  a12541 = dx512 * dx423 - dx513 * dx422;
  a12542 = dx513 * dx421 - dx511 * dx423;
  a12543 = dx511 * dx422 - dx512 * dx421;
  a23651 = dx622 * dx533 - dx623 * dx532;
  a23652 = dx623 * dx531 - dx621 * dx533;
  a23653 = dx621 * dx532 - dx622 * dx531;
  a31461 = dx432 * dx613 - dx433 * dx612;
  a31462 = dx433 * dx611 - dx431 * dx613;
  a31463 = dx431 * dx612 - dx432 * dx611;

  // area dot products

  a132s = a1321 * a1321 + a1322 * a1322 + a1323 * a1323;
  a456s = a4561 * a4561 + a4562 * a4562 + a4563 * a4563;
  a1254s = a12541 * a12541 + a12542 * a12542 + a12543 * a12543;
  a2365s = a23651 * a23651 + a23652 * a23652 + a23653 * a23653;
  a3146s = a31461 * a31461 + a31462 * a31462 + a31463 * a31463;

  // angle function for each edge

  *w21 = (- a12541 * a1321 - a12542 * a1322 - a12543 * a1323) / sqrt (a1254s * a132s);
  *w31 = (- a31461 * a1321 - a31462 * a1322 - a31463 * a1323) / sqrt (a3146s * a132s);
  *w32 = (- a23651 * a1321 - a23652 * a1322 - a23653 * a1323) / sqrt (a2365s * a132s);
  *w41 = (- a31461 * a12541 - a31462 * a12542 - a31463 * a12543) / sqrt (a3146s * a1254s);
  *w52 = (- a23651 * a12541 - a23652 * a12542 - a23653 * a12543) / sqrt (a2365s * a1254s);
  *w54 = (- a12541 * a4561 - a12542 * a4562 - a12543 * a4563) / sqrt (a1254s * a456s);
  *w63 = (- a23651 * a31461 - a23652 * a31462 - a23653 * a31463) / sqrt (a2365s * a3146s);
  *w64 = (- a31461 * a4561 - a31462 * a4562 - a31463 * a4563) / sqrt (a3146s * a456s);
  *w65 = (- a23651 * a4561 - a23652 * a4562 - a23653 * a4563) / sqrt (a2365s * a456s);

  *w21 = MAX (MIN (*w21, dc1), -dc1);
  *w31 = MAX (MIN (*w31, dc1), -dc1);
  *w32 = MAX (MIN (*w32, dc1), -dc1);
  *w41 = MAX (MIN (*w41, dc1), -dc1);
  *w52 = MAX (MIN (*w52, dc1), -dc1);
  *w54 = MAX (MIN (*w54, dc1), -dc1);
  *w63 = MAX (MIN (*w63, dc1), -dc1);
  *w64 = MAX (MIN (*w64, dc1), -dc1);
  *w65 = MAX (MIN (*w65, dc1), -dc1);

  return;
}

void ug3_hex_dh_ang_w (INT_ inode1,
                       INT_ inode2,
                       INT_ inode3,
                       INT_ inode4,
                       INT_ inode5,
                       INT_ inode6,
                       INT_ inode7,
                       INT_ inode8,
                       double *w21, 
                       double *w32,
                       double *w41,
                       double *w43,
                       double *w51,
                       double *w62,
                       double *w65,
                       double *w73,
                       double *w76,
                       double *w84,
                       double *w85,
                       double *w87,
                       DOUBLE_3D * x)
{

  double a1265s, a12651, a12652, a12653,
         a1432s, a14321, a14322, a14323,
         a1584s, a15841, a15842, a15843,
         a2376s, a23761, a23762, a23763,
         a3487s, a34871, a34872, a34873,
         a5678s, a56781, a56782, a56783,
         dc1,
         dx311, dx312, dx313,
         dx421, dx422, dx423,
         dx521, dx522, dx523,
         dx541, dx542, dx543,
         dx611, dx612, dx613,
         dx631, dx632, dx633,
         dx721, dx722, dx723,
         dx741, dx742, dx743,
         dx751, dx752, dx753,
         dx811, dx812, dx813,
         dx831, dx832, dx833,
         dx861, dx862, dx863,
         x11, x12, x13,
         x21, x22, x23,
         x31, x32, x33,
         x41, x42, x43,
         x51, x52, x53,
         x61, x62, x63,
         x71, x72, x73,
         x81, x82, x83;

  dc1 = 1.0;

  // coordinates

  x11 = x[inode1][0];
  x12 = x[inode1][1];
  x13 = x[inode1][2];
  x21 = x[inode2][0];
  x22 = x[inode2][1];
  x23 = x[inode2][2];
  x31 = x[inode3][0];
  x32 = x[inode3][1];
  x33 = x[inode3][2];
  x41 = x[inode4][0];
  x42 = x[inode4][1];
  x43 = x[inode4][2];
  x51 = x[inode5][0];
  x52 = x[inode5][1];
  x53 = x[inode5][2];
  x61 = x[inode6][0];
  x62 = x[inode6][1];
  x63 = x[inode6][2];
  x71 = x[inode7][0];
  x72 = x[inode7][1];
  x73 = x[inode7][2];
  x81 = x[inode8][0];
  x82 = x[inode8][1];
  x83 = x[inode8][2];

  // delta vectors

  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx421 = x41 - x21;
  dx422 = x42 - x22;
  dx423 = x43 - x23;
  dx521 = x51 - x21;
  dx522 = x52 - x22;
  dx523 = x53 - x23;
  dx541 = x51 - x41;
  dx542 = x52 - x42;
  dx543 = x53 - x43;
  dx611 = x61 - x11;
  dx612 = x62 - x12;
  dx613 = x63 - x13;
  dx631 = x61 - x31;
  dx632 = x62 - x32;
  dx633 = x63 - x33;
  dx721 = x71 - x21;
  dx722 = x72 - x22;
  dx723 = x73 - x23;
  dx741 = x71 - x41;
  dx742 = x72 - x42;
  dx743 = x73 - x43;
  dx751 = x71 - x51;
  dx752 = x72 - x52;
  dx753 = x73 - x53;
  dx811 = x81 - x11;
  dx812 = x82 - x12;
  dx813 = x83 - x13;
  dx831 = x81 - x31;
  dx832 = x82 - x32;
  dx833 = x83 - x33;
  dx861 = x81 - x61;
  dx862 = x82 - x62;
  dx863 = x83 - x63;

  // area vectors

  a12651 = dx612 * dx523 - dx613 * dx522;
  a12652 = dx613 * dx521 - dx611 * dx523;
  a12653 = dx611 * dx522 - dx612 * dx521;
  a14321 = dx422 * dx313 - dx423 * dx312;
  a14322 = dx423 * dx311 - dx421 * dx313;
  a14323 = dx421 * dx312 - dx422 * dx311;
  a15841 = dx542 * dx813 - dx543 * dx812;
  a15842 = dx543 * dx811 - dx541 * dx813;
  a15843 = dx541 * dx812 - dx542 * dx811;
  a23761 = dx722 * dx633 - dx723 * dx632;
  a23762 = dx723 * dx631 - dx721 * dx633;
  a23763 = dx721 * dx632 - dx722 * dx631;
  a34871 = dx832 * dx743 - dx833 * dx742;
  a34872 = dx833 * dx741 - dx831 * dx743;
  a34873 = dx831 * dx742 - dx832 * dx741;
  a56781 = dx752 * dx863 - dx753 * dx862;
  a56782 = dx753 * dx861 - dx751 * dx863;
  a56783 = dx751 * dx862 - dx752 * dx861;

  // area dot products

  a1265s = a12651 * a12651 + a12652 * a12652 + a12653 * a12653;
  a1432s = a14321 * a14321 + a14322 * a14322 + a14323 * a14323;
  a1584s = a15841 * a15841 + a15842 * a15842 + a15843 * a15843;
  a2376s = a23761 * a23761 + a23762 * a23762 + a23763 * a23763;
  a3487s = a34871 * a34871 + a34872 * a34872 + a34873 * a34873;
  a5678s = a56781 * a56781 + a56782 * a56782 + a56783 * a56783;

  // angle function for each edge

  *w21 = (- a12651 * a14321 - a12652 * a14322 - a12653 * a14323) / sqrt (a1265s * a1432s);
  *w32 = (- a23761 * a14321 - a23762 * a14322 - a23763 * a14323) / sqrt (a2376s * a1432s);
  *w41 = (- a15841 * a14321 - a15842 * a14322 - a15843 * a14323) / sqrt (a1584s * a1432s);
  *w43 = (- a34871 * a14321 - a34872 * a14322 - a34873 * a14323) / sqrt (a3487s * a1432s);
  *w51 = (- a15841 * a12651 - a15842 * a12652 - a15843 * a12653) / sqrt (a1584s * a1265s);
  *w62 = (- a23761 * a12651 - a23762 * a12652 - a23763 * a12653) / sqrt (a2376s * a1265s);
  *w65 = (- a12651 * a56781 - a12652 * a56782 - a12653 * a56783) / sqrt (a1265s * a5678s);
  *w73 = (- a23761 * a34871 - a23762 * a34872 - a23763 * a34873) / sqrt (a2376s * a3487s);
  *w76 = (- a23761 * a56781 - a23762 * a56782 - a23763 * a56783) / sqrt (a2376s * a5678s);
  *w84 = (- a34871 * a15841 - a34872 * a15842 - a34873 * a15843) / sqrt (a3487s * a1584s);
  *w85 = (- a15841 * a56781 - a15842 * a56782 - a15843 * a56783) / sqrt (a1584s * a5678s);
  *w87 = (- a34871 * a56781 - a34872 * a56782 - a34873 * a56783) / sqrt (a3487s * a5678s);

  *w21 = MAX (MIN (*w21, dc1), -dc1);
  *w32 = MAX (MIN (*w32, dc1), -dc1);
  *w41 = MAX (MIN (*w41, dc1), -dc1);
  *w43 = MAX (MIN (*w43, dc1), -dc1);
  *w51 = MAX (MIN (*w51, dc1), -dc1);
  *w62 = MAX (MIN (*w62, dc1), -dc1);
  *w65 = MAX (MIN (*w65, dc1), -dc1);
  *w73 = MAX (MIN (*w73, dc1), -dc1);
  *w76 = MAX (MIN (*w76, dc1), -dc1);
  *w84 = MAX (MIN (*w84, dc1), -dc1);
  *w85 = MAX (MIN (*w85, dc1), -dc1);
  *w87 = MAX (MIN (*w87, dc1), -dc1);

  return;
}

void ug3_dh_ang_w (double x11,
                   double x12,
                   double x13,
                   double x21,
                   double x22,
                   double x23,
                   double x31,
                   double x32,
                   double x33,
                   double x41,
                   double x42,
                   double x43,
                   double *w21, 
                   double *w31, 
                   double *w32, 
                   double *w41, 
                   double *w42, 
                   double *w43)
{

/*
 * Determine dihedral element angle functions from a given set of tet element
 * coordinates.
 *
 * wAB = cos (angleAB) * |cos (angleAB)| = A1.A2  * |A1.A2| / ( A1.A1 * A2.A2 )
 *
 * angleAB	: dihedral angle between two faces sharing edge AB
 * A1           : area for one of the two faces
 * A2           : area for one of the two faces
 */

  double a124s, a1241, a1242, a1243,
         a132s, a1321, a1322, a1323,
         a143s, a1431, a1432, a1433,
         a234s, a2341, a2342, a2343,
         dx211, dx212, dx213,
         dx311, dx312, dx313,
         dx411, dx412, dx413;

  // delta vectors

  dx211 = x21 - x11;
  dx212 = x22 - x12;
  dx213 = x23 - x13;
  dx311 = x31 - x11;
  dx312 = x32 - x12;
  dx313 = x33 - x13;
  dx411 = x41 - x11;
  dx412 = x42 - x12;
  dx413 = x43 - x13;

  // area vectors

  a1241 = dx212 * dx413 - dx213 * dx412;
  a1242 = dx213 * dx411 - dx211 * dx413;
  a1243 = dx211 * dx412 - dx212 * dx411;
  a1321 = dx312 * dx213 - dx313 * dx212;
  a1322 = dx313 * dx211 - dx311 * dx213;
  a1323 = dx311 * dx212 - dx312 * dx211;
  a1431 = dx412 * dx313 - dx413 * dx312;
  a1432 = dx413 * dx311 - dx411 * dx313;
  a1433 = dx411 * dx312 - dx412 * dx311;
  a2341 = - a1241 - a1321 - a1431;
  a2342 = - a1242 - a1322 - a1432;
  a2343 = - a1243 - a1323 - a1433;

  // area dot products

  a124s = a1241 * a1241 + a1242 * a1242 + a1243 * a1243;
  a132s = a1321 * a1321 + a1322 * a1322 + a1323 * a1323;
  a143s = a1431 * a1431 + a1432 * a1432 + a1433 * a1433;
  a234s = a2341 * a2341 + a2342 * a2342 + a2343 * a2343;

  // angle function for each edge

  *w21 = a1241 * a1321 - a1242 * a1322 - a1243 * a1323;
  *w31 = a1431 * a1321 - a1432 * a1322 - a1433 * a1323;
  *w32 = a2341 * a1321 - a2342 * a1322 - a2343 * a1323;
  *w41 = a1431 * a1241 - a1432 * a1242 - a1433 * a1243;
  *w42 = a2341 * a1241 - a2342 * a1242 - a2343 * a1243;
  *w43 = a2341 * a1431 - a2342 * a1432 - a2343 * a1433;

  // edge area dot products

  *w21 = a1241 * a1321 + a1242 * a1322 + a1243 * a1323;
  *w31 = a1431 * a1321 + a1432 * a1322 + a1433 * a1323;
  *w32 = a2341 * a1321 + a2342 * a1322 + a2343 * a1323;
  *w41 = a1431 * a1241 + a1432 * a1242 + a1433 * a1243;
  *w42 = a2341 * a1241 + a2342 * a1242 + a2343 * a1243;
  *w43 = a2341 * a1431 + a2342 * a1432 + a2343 * a1433;

  // angle function for each edge

  *w21 = *w21 * fabs (*w21) / (a124s * a132s);
  *w31 = *w31 * fabs (*w31) / (a143s * a132s);
  *w32 = *w32 * fabs (*w32) / (a234s * a132s);
  *w41 = *w41 * fabs (*w41) / (a143s * a124s);
  *w42 = *w42 * fabs (*w42) / (a234s * a124s);
  *w43 = *w43 * fabs (*w43) / (a234s * a143s);

  return;
}
