#include "UG3_LIB.h"

/*
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_conv_2d_mesh_3d.c,v 1.10 2023/11/29 01:11:32 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

INT_ ug3_conv_2d_mesh_3d (
  INT_ mesh_3d,
  INT_ mixed_3d,
  INT_ mmsg,
  INT_ nplane,
  double dang,
  double dz,
  INT_ nbedge_2d,
  INT_ ntria_2d,
  INT_ nquad_2d,
  INT_ nnode_2d,
  INT_1D *idibe_2d,
  INT_2D *inibe_2d,
  INT_4D *iniq_2d,
  INT_3D *init_2d,
  DOUBLE_2D *x_2d,
  INT_ *nbface,
  INT_ *nquad,
  INT_ *nelem,
  INT_ *nelemc5,
  INT_ *nelemc6,
  INT_ *nelemc8,
  INT_ *nnode,
  INT_1D **idibf,
  INT_3D **inibf,
  INT_4D **iniq,
  INT_4D **iniel,
  INT_5D **inielc5,
  INT_6D **inielc6,
  INT_8D **inielc8,
  DOUBLE_3D **x)
{
  // Convert a 2D planar mesh to an axisymmetric or stacked plane 3D mesh.

  // mesh_3d=0	: do nothing
  // mesh_3d=1	: convert to a axisymmetric 3D volume mesh
  // mesh_3d=2	: convert to a stacked plane 3D volume mesh

  // mixed_3d=0	: use tet-elements for 3D mesh
  // mixed_3d=1	: use prism-elements for 3D mesh and if mesh_3d=2 also use
  // 		  tet-elements and pyramid-elements along center line

  // dang	: angle between rotated planes with axisymmetric volume mesh
  // dz		: distance between planes with stacked plane volume mesh

  // nplanes	: number of planes for 3D volume mesh

  INT_1D *idibf_save = NULL;
  INT_1D *itin_2d = NULL;
  INT_1D *jnin = NULL;
  INT_1D *litin_2d = NULL;

  DOUBLE_3D *x_save = NULL;

  INT_ i, ibedge_2d, ibface, inode, inode_2d, loc, plane;
  INT_ i1 = 0;
  INT_ i2 = 0;
  INT_ i3 = 0;
  INT_ i4 = 0;
  INT_ i5 = 0;
  INT_ i6 = 0;
  INT_ id_max = 1;
  INT_ ielem = 0;
  INT_ ielemc6 = 0;
  INT_ ierr = 0;
  INT_ iquad = 0;
  INT_ jbface = 0;
  INT_ jelem = 0;
  INT_ jelemc5 = 0;
  INT_ jelemc6 = 0;
  INT_ jnode = 0;
  INT_ jquad = 0;
  INT_ nbfacei = 0;
  INT_ nbfaced = 0;
  INT_ nbfpnt = 0;
  INT_ nbfpntd = 0;
  INT_ nelemd = 0;
  INT_ nelemc5d = 0;
  INT_ nnodeb_2d = 0;
  INT_ nquad_2d_ = 0;

  double ang, angi, dang_;

  *nquad = 0;
  *nelem = 0;
  *nelemc5 = 0;
  *nelemc6 = 0;
  *nelemc8 = 0;
  *nnode = 0;

  *idibf = NULL;
  *inibf = NULL;
  *iniq = NULL;
  *iniel = NULL;
  *inielc5 = NULL;
  *inielc6 = NULL;
  *inielc8 = NULL;
  *x = NULL;

  // exit if 3D volume mesh option is off

  if (mesh_3d == 0) return 0;

  // set number of boundary edge nodes

  ug2_nnodeb (nbedge_2d, &nnodeb_2d, inibe_2d);

  // allocate initial boundary face connectivity for 3D volume mesh
  // and work data

  *nbface = ntria_2d;

  *inibf = (INT_3D *) ug_malloc (&ierr, ((*nbface)+1) * sizeof (INT_3D));
  if (mesh_3d == 1) jnin = (INT_1D *) ug_malloc (&ierr, (nnode_2d+1) * sizeof (INT_1D));

  if (ierr) {
    ug_error_message ("*** ERROR 100207 : unable to allocate required memory ***");
    ierr = 100207;
    goto cleanup;
  }

  // initialize boundary face connectivity for 3D volume mesh 

  for (ibface = 1; ibface <= ntria_2d; ibface++) {
    (*inibf)[ibface][0] = init_2d[ibface][0];
    (*inibf)[ibface][1] = init_2d[ibface][1];
    (*inibf)[ibface][2] = init_2d[ibface][2];
  }

  // convert quad faces to tria faces

  nquad_2d_ = nquad_2d;

  ierr = ug2_qtria (nbface, &nquad_2d_, inibf, iniq_2d, x_2d);

  if (ierr) goto cleanup;

  // flag nodes on center line

  if (mesh_3d == 1) {

    memset (jnin, 0, (nnode_2d+1)*sizeof (INT_1D));

    for (inode = 1; inode <= nnode_2d; inode++) {
      if (x_2d[inode][1] == 0.0) jnin[inode] = 1;
    }

    for (ibface = 1; ibface <= ntria_2d; ibface++) {

      i = 0;

      if (jnin[(*inibf)[ibface][0]] == 1) i++;
      if (jnin[(*inibf)[ibface][1]] == 1) i++;
      if (jnin[(*inibf)[ibface][2]] == 1) i++;

           if (i == 1) i1++;
      else if (i == 2) i2++;
    }
  }

  // allocate data

  nbfacei = *nbface;

  if (mixed_3d == 0) {
    *nbface = 2*nbfacei + 2*(nplane-1)*nnodeb_2d;
    *nelem = 3*(nplane-1)*nbfacei;
    nbfaced = *nbface;
    nelemd = *nelem;
  }
  else {
    *nbface = 2*nbfacei;
    *nquad = (nplane-1)*nnodeb_2d;
    *nelemc6 = (nplane-1)*nbfacei;
    nbfaced = *nbface + 2*(nplane-1);
    nelemd = (nplane-1)*i2;
    nelemc5d = (nplane-1)*i1;
  }

  *nnode = nplane * nnode_2d;

  dang_ = dang * atan(1.0) / 45.0;
  angi = atan(1.0) + atan(1.0);

  *idibf = (INT_1D *) ug_malloc (&ierr, (nbfaced+(*nquad)+1) * sizeof (INT_1D));
  *inibf = (INT_3D *) ug_realloc (&ierr, *inibf, (nbfaced+1) * sizeof (INT_3D));
  if (*nquad) *iniq = (INT_4D *) ug_realloc (&ierr, *iniq, ((*nquad)+1) * sizeof (INT_4D));
  if (nelemd) *iniel = (INT_4D *) ug_malloc (&ierr, (nelemd+1) * sizeof (INT_4D));
  if (nelemc5) *inielc5 = (INT_5D *) ug_malloc (&ierr, (nelemc5d+1) * sizeof (INT_6D));
  if (*nelemc6) *inielc6 = (INT_6D *) ug_malloc (&ierr, ((*nelemc6)+1) * sizeof (INT_6D));

  litin_2d = (INT_1D *) ug_malloc (&ierr, (nnode_2d+2) * sizeof (INT_1D));

  *x = (DOUBLE_3D *) ug_malloc (&ierr, ((*nnode)+1) * sizeof (DOUBLE_3D));

  if (ierr) {
    ug_error_message ("*** ERROR 100207 : unable to allocate required memory ***");
    ierr = 100207;
    goto cleanup;
  }

  // set map of boundary faces surrounding each node

  ierr = ug2_ielin (1, nbfacei, &nbfpntd, nnode_2d, &nbfpnt, 
                    *inibf, NULL, &itin_2d, litin_2d);

  if (ierr) goto cleanup;

  // set coordinates for axisymmetric 3D volume mesh

  if (mesh_3d == 1) {

    for (plane = 0; plane < nplane; plane++) {

      ang = angi + dang_ * (double) plane;

      for (inode_2d = 1; inode_2d <= nnode_2d; inode_2d++) {

        inode = inode_2d + plane * nnode_2d;

        (*x)[inode][0] = x_2d[inode_2d][0];
        (*x)[inode][1] = x_2d[inode_2d][1] * sin(ang);
        (*x)[inode][2] = x_2d[inode_2d][1] * cos(ang);
      }
    }
  }

  // set coordinates for stacked plane 3D volume mesh

  else {

    for (plane = 0; plane < nplane; plane++) {

      for (inode_2d = 1; inode_2d <= nnode_2d; inode_2d++) {

        inode = inode_2d + plane * nnode_2d;

        (*x)[inode][0] = x_2d[inode_2d][0];
        (*x)[inode][1] = x_2d[inode_2d][1];
        (*x)[inode][2] = - dz * plane;
      }
    }
  }

  // set boundary face connectivity and ID for 3D volume mesh

  if (idibe_2d) id_max = ug_max_int (1, nbedge_2d, idibe_2d);

  // start and ending planes for 3D mesh

  for (ibface = 1; ibface <= nbfacei; ibface++) {

    i2 = (*inibf)[ibface][1];
    i3 = (*inibf)[ibface][2];

    (*idibf)[ibface] = id_max+1;
    (*inibf)[ibface][1] = i3;
    (*inibf)[ibface][2] = i2;

    (*idibf)[ibface+nbfacei] = id_max+2;
    (*inibf)[ibface+nbfacei][0] = (*inibf)[ibface][0] + (nplane-1) * nnode_2d;
    (*inibf)[ibface+nbfacei][1] = i2 + (nplane-1) * nnode_2d;
    (*inibf)[ibface+nbfacei][2] = i3 + (nplane-1) * nnode_2d;
  }

  // tria-faces on boundary-edges for 3D mesh

  if (mixed_3d == 0) {

    ibface = 2*nbfacei;

    for (plane = 1; plane < nplane; plane++) {

      for (ibedge_2d = 1; ibedge_2d <= nbedge_2d; ibedge_2d++) {

        i1 = inibe_2d[ibedge_2d][0];
        i2 = inibe_2d[ibedge_2d][1];

        if (i1 < i2) {

          ibface++;

          (*idibf)[ibface] = (idibe_2d) ? idibe_2d[ibedge_2d]: 1;
          (*inibf)[ibface][0] = i1 + (plane-1) * nnode_2d;
          (*inibf)[ibface][1] = i2 + (plane-1) * nnode_2d;
          (*inibf)[ibface][2] = i1 + plane * nnode_2d;

          ibface++;

          (*idibf)[ibface] = (idibe_2d) ? idibe_2d[ibedge_2d]: 1;
          (*inibf)[ibface][0] = i2 + (plane-1) * nnode_2d;
          (*inibf)[ibface][1] = i2 + plane * nnode_2d;
          (*inibf)[ibface][2] = i1 + plane * nnode_2d;
        }

        else {

          ibface++;

          (*idibf)[ibface] = (idibe_2d) ? idibe_2d[ibedge_2d]: 1;
          (*inibf)[ibface][0] = i1 + (plane-1) * nnode_2d;
          (*inibf)[ibface][1] = i2 + (plane-1) * nnode_2d;
          (*inibf)[ibface][2] = i2 + plane * nnode_2d;

          ibface++;

          (*idibf)[ibface] = (idibe_2d) ? idibe_2d[ibedge_2d]: 1;
          (*inibf)[ibface][0] = i1 + (plane-1) * nnode_2d;
          (*inibf)[ibface][1] = i2 + plane * nnode_2d;
          (*inibf)[ibface][2] = i1 + plane * nnode_2d;
        }
      }
    }
  }

  // quad-faces on boundary-edges for 3D mesh

  else {

    for (plane = 1; plane < nplane; plane++) {

      for (ibedge_2d = 1; ibedge_2d <= nbedge_2d; ibedge_2d++) {

        iquad++;

        (*idibf)[2*nbfacei+iquad] = (idibe_2d) ? idibe_2d[ibedge_2d]: 1;

        (*iniq)[iquad][0] = inibe_2d[ibedge_2d][0] + (plane-1) * nnode_2d;
        (*iniq)[iquad][1] = inibe_2d[ibedge_2d][1] + (plane-1) * nnode_2d;
        (*iniq)[iquad][2] = inibe_2d[ibedge_2d][1] + plane * nnode_2d;
        (*iniq)[iquad][3] = inibe_2d[ibedge_2d][0] + plane * nnode_2d;
      }
    }
  }

  // set element connectivity and ID for 3D tet-element volume mesh

  if (mixed_3d == 0) {

    for (plane = 0; plane < nplane-1; plane++) {

      for (inode_2d = 1; inode_2d <= nnode_2d; inode_2d++) {

        for (loc = litin_2d[inode_2d]; loc < litin_2d[inode_2d+1]; loc++) {

          ibface = itin_2d[loc];

          if (inode_2d == (*inibf)[ibface][0]) {
            i1 = inode_2d;
            i2 = (*inibf)[ibface][1];
            i3 = (*inibf)[ibface][2];
          }
          else if (inode_2d == (*inibf)[ibface][1]) {
            i1 = inode_2d;
            i2 = (*inibf)[ibface][2];
            i3 = (*inibf)[ibface][0];
          }
          else {
            i1 = inode_2d;
            i2 = (*inibf)[ibface][0];
            i3 = (*inibf)[ibface][1];
          }

          ielem++;

          (*iniel)[ielem][0] = i1 + plane * nnode_2d;
          (*iniel)[ielem][1] = (i1 < i2) ? i2 + plane * nnode_2d:
                                           i2 + (plane+1) * nnode_2d;
          (*iniel)[ielem][2] = (i1 < i3) ? i3 + plane * nnode_2d:
                                           i3 + (plane+1) * nnode_2d;
          (*iniel)[ielem][3] = i1 + (plane+1) * nnode_2d;
        }
      }
    }
  }

  // set element connectivity and ID for 3D prism-element volume mesh

  else {

    for (plane = 0; plane < nplane-1; plane++) {

      for (ibface = 1; ibface <= nbfacei; ibface++) {

        ielemc6++;

        (*inielc6)[ielemc6][0] = (*inibf)[ibface][0] + plane * nnode_2d;
        (*inielc6)[ielemc6][1] = (*inibf)[ibface][1] + plane * nnode_2d;
        (*inielc6)[ielemc6][2] = (*inibf)[ibface][2] + plane * nnode_2d;
        (*inielc6)[ielemc6][3] = (*inibf)[ibface][0] + (plane+1) * nnode_2d;
        (*inielc6)[ielemc6][4] = (*inibf)[ibface][1] + (plane+1) * nnode_2d;
        (*inielc6)[ielemc6][5] = (*inibf)[ibface][2] + (plane+1) * nnode_2d;
      }
    }
  }

  // remove duplicate coordinates in axisymmetric 3D volume mesh

  if (mesh_3d == 1) {

    // allocate work data

    idibf_save = (INT_1D *) ug_malloc (&ierr, (nbfaced+(*nquad)+1) * sizeof (INT_1D));
    jnin = (INT_1D *) ug_realloc (&ierr, jnin, ((*nnode)+1) * sizeof (INT_1D));
    x_save = (DOUBLE_3D *) ug_malloc (&ierr, ((*nnode)+1) * sizeof (DOUBLE_3D));

    if (ierr) {
      ug_error_message ("*** ERROR 100207 : unable to allocate required memory ***");
      ierr = 100207;
      goto cleanup;
    }

    // count nodes on center line for axisymmetric case and
    // flag all nodes on center line

    memset (jnin, 0, ((*nnode)+1)*sizeof (INT_1D));

    for (inode = 1; inode <= nnode_2d; inode++) {
      if (x_2d[inode][1] == 0.0) jnin[inode] = 1;
    }

    // remove duplicate nodes on center line

    for (inode = 1; inode <= nnode_2d; inode++) {

      if (jnin[inode] == 1) {

        jnode++;

        jnin[inode] = jnode;

        x_save[jnode][0] = (*x)[inode][0];
        x_save[jnode][1] = (*x)[inode][1];
        x_save[jnode][2] = (*x)[inode][2];

        for (plane = 1; plane < nplane; plane++)
          jnin[inode+plane*nnode_2d] = jnode;
      }
    }

    for (inode = 1; inode <= *nnode; inode++) {

      if (jnin[inode] == 0) {

        jnode++;

        jnin[inode] = jnode;

        x_save[jnode][0] = (*x)[inode][0];
        x_save[jnode][1] = (*x)[inode][1];
        x_save[jnode][2] = (*x)[inode][2];
      }
    }

    *nnode = jnode;

    for (inode = 1; inode <= *nnode; inode++) {
      (*x)[inode][0] = x_save[inode][0];
      (*x)[inode][1] = x_save[inode][1];
      (*x)[inode][2] = x_save[inode][2];
    }

    // copy face ID

    for (i = 1; i <= (*nbface)+(*nquad); i++)
      idibf_save[i] = (*idibf)[i];

    // reset tria-face connectivity along center line

    for (ibface = 1; ibface <= *nbface; ibface++) {

      i1 = jnin[(*inibf)[ibface][0]];
      i2 = jnin[(*inibf)[ibface][1]];
      i3 = jnin[(*inibf)[ibface][2]];

      if (i1 != i2 && i1 != i3 && i2 != i3) {

        jbface++;

        (*idibf)[jbface] = idibf_save[ibface];

        (*inibf)[jbface][0] = i1;
        (*inibf)[jbface][1] = i2;
        (*inibf)[jbface][2] = i3;
      }
    }

    *nbface = jbface;

    // reset tet-element connectivity

    jelem = 0;

    for (ielem = 1; ielem <= *nelem; ielem++) {

      i1 = jnin[(*iniel)[ielem][0]];
      i2 = jnin[(*iniel)[ielem][1]];
      i3 = jnin[(*iniel)[ielem][2]];
      i4 = jnin[(*iniel)[ielem][3]];

      if (i1 != i2 && i1 != i3 && i1 != i4 && 
          i2 != i3 && i2 != i4 && i3 != i4) {

        jelem++;

        (*iniel)[jelem][0] = i1;
        (*iniel)[jelem][1] = i2;
        (*iniel)[jelem][2] = i3;
	(*iniel)[jelem][3] = i4;
      }
    }

    *nelem = jelem;

    // for mixed element type mesh

    if (mixed_3d) {

      // replace quad-faces with tria-faces on boundary-edges at center line

      for (iquad = 1; iquad <= *nquad; iquad++) {

        i1 = jnin[(*iniq)[iquad][0]];
        i2 = jnin[(*iniq)[iquad][1]];
        i3 = jnin[(*iniq)[iquad][2]];
        i4 = jnin[(*iniq)[iquad][3]];

        if (i1 == i4 && i2 != i3) {

          jbface++;

          (*idibf)[jbface] = idibf_save[2*nbfacei+iquad];

          (*inibf)[jbface][0] = i1;
          (*inibf)[jbface][1] = i2;
          (*inibf)[jbface][2] = i3;
        }

        else if (i1 != i4 && i2 == i3) {

          jbface++;

          (*idibf)[jbface] = idibf_save[2*nbfacei+iquad];

          (*inibf)[jbface][0] = i1;
          (*inibf)[jbface][1] = i2;
          (*inibf)[jbface][2] = i4;
        }
      }

      *nbface = jbface;

      // reset quad-faces on boundary-edges for 3D mesh

      for (iquad = 1; iquad <= *nquad; iquad++) {

        i1 = jnin[(*iniq)[iquad][0]];
        i2 = jnin[(*iniq)[iquad][1]];
        i3 = jnin[(*iniq)[iquad][2]];
        i4 = jnin[(*iniq)[iquad][3]];

        if (i1 != i4 && i2 != i3) {

          jquad++;

          (*idibf)[(*nbface)+jquad] = idibf_save[2*nbfacei+iquad];

          (*iniq)[jquad][0] = i1;
          (*iniq)[jquad][1] = i2;
          (*iniq)[jquad][2] = i3;
          (*iniq)[jquad][3] = i4;
        }
      }

      *nquad = jquad;

      // reset prism-element connectivity

      jelemc6 = 0;

      for (ielemc6 = 1; ielemc6 <= *nelemc6; ielemc6++) {

        i1 = jnin[(*inielc6)[ielemc6][0]];
        i2 = jnin[(*inielc6)[ielemc6][1]];
        i3 = jnin[(*inielc6)[ielemc6][2]];
        i4 = jnin[(*inielc6)[ielemc6][3]];
        i5 = jnin[(*inielc6)[ielemc6][4]];
        i6 = jnin[(*inielc6)[ielemc6][5]];

        if (i1 != i4 && i2 != i5 && i3 != i6) {

          jelemc6++;

          (*inielc6)[jelemc6][0] = i1;
          (*inielc6)[jelemc6][1] = i2;
          (*inielc6)[jelemc6][2] = i3;
	  (*inielc6)[jelemc6][3] = i4;
	  (*inielc6)[jelemc6][4] = i5;
	  (*inielc6)[jelemc6][5] = i6;
        }

        // replace prism-elements with pyramid-elements and tet-elements

        else {

          i = 0;

          if (i1 == i4) i++;
          if (i2 == i5) i++;
          if (i3 == i6) i++;

          if (i == 1) {

            jelemc5++;

            if (i1 == i4) {
              (*inielc5)[jelemc5][0] = i2;
              (*inielc5)[jelemc5][1] = i3;
              (*inielc5)[jelemc5][2] = i1;
	      (*inielc5)[jelemc5][3] = i5;
	      (*inielc5)[jelemc5][4] = i6;
            }
            else if (i2 == i5) {
              (*inielc5)[jelemc5][0] = i3;
              (*inielc5)[jelemc5][1] = i1;
              (*inielc5)[jelemc5][2] = i2;
	      (*inielc5)[jelemc5][3] = i6;
	      (*inielc5)[jelemc5][4] = i4;
            }
            else { // if (i3 == i6)
              (*inielc5)[jelemc5][0] = i1;
              (*inielc5)[jelemc5][1] = i2;
              (*inielc5)[jelemc5][2] = i3;
	      (*inielc5)[jelemc5][3] = i4;
	      (*inielc5)[jelemc5][4] = i5;
            }
          }

          else { // if (i == 2) {

            jelem++;

            (*iniel)[jelem][0] = i1;
            (*iniel)[jelem][1] = i2;
            (*iniel)[jelem][2] = i3;
            (*iniel)[jelem][3] = (i1 != i4) ? i4: (i2 != i5) ? i5: i6;
          }
        }
      }

      *nelem = jelem;
      *nelemc5 = jelemc5;
      *nelemc6 = jelemc6;
    }
  }

  cleanup:

  // cleanup work data

  ug_free (idibf_save);
  ug_free (itin_2d);
  ug_free (jnin);
  ug_free (litin_2d);
  ug_free (x_save);

  return (ierr);
}
