#include "UG3_LIB.h"

static INT_ mbv_mode_ = 1;
static INT_ bv_ls_set_param_ = 1;

static double ang_bv_disc_ = 30.0;
static double ang_bv_sharp_ = 179.99;
static double ang_bv_smooth_ = 2.0;
static double bv_det_tol_ = 1.0e-6;
static double bv_diff_tol_ = 1.0e-6;
static double bv_ev_sharp_ = 1.0e-6;
static double cosdiscs_ = 0.0;
static double cossharps_ = 0.0;
static double cossmooths_ = 0.0;

static void ug3_bv_ls_set_param (void);

static INT_ ug3_bv_surf_loop (INT_ *kbface,
                              INT_ inode,
                              INT_ nsurfd,
                              INT_ set_start,
                              INT_3D * ibfibf,
                              INT_3D * inibf,
                              DOUBLE_3D * bfv,
                              DOUBLE_3D * bv_surf,
                              DOUBLE_3D * x);

static void ug3_bv_ls_surf (INT_ nsurf,
                            DOUBLE_3D * bv_surf,
                            DOUBLE_3D bv);

static void ug3_bv_avg (INT_ kbface,
                        INT_ inode,
                        INT_3D * ibfibf,
                        INT_3D * inibf,
                        DOUBLE_3D * bfv,
                        DOUBLE_3D bv,
                        DOUBLE_3D * x);

static INT_ ug3_bv_most (INT_ kbface,
                         INT_ mmsg, 
                         INT_ nbface, 
                         INT_ inode,
                         INT_3D * ibfibf,
                         INT_3D * inibf,
                         DOUBLE_3D * bfv,
                         DOUBLE_3D bv,
                         DOUBLE_3D * x);

static INT_ ug3_MostNormalVertexNormal (INT_ iVer,
                                        INT_ mmsg,
                                        INT_ nbrNor,
                                        DOUBLE_3D *vecNor,
                                        DOUBLE_3D mosNor);

INT_ ug3_bv_ls (INT_ *ibface,
                INT_ inode,
                INT_ nbface,
                INT_3D * ibfibf,
                INT_3D * inibf,
                DOUBLE_3D bv,
                DOUBLE_3D * bfv,
                DOUBLE_3D * x)
{

/*
 * Determine boundary normal vector at a given node from a least-squares
 * optimization method that uses the boundary face normals for each of the
 * boundary faces that surround the node.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_bv_ls.c,v 1.32 2022/11/21 00:25:14 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  DOUBLE_3D *bv_surf = NULL;

  INT_ found, nsurf;
  INT_ ierr = 0;
  INT_ mmsg = 0;
  INT_ nsurfd = 100;

  // set parameters for determining face normal vectors

  ug3_bv_ls_set_param ();

  // if initial starting face if not set then set initial staring face

  if (*ibface == 0)
  {
    *ibface = 0;

    do
    {
      ++(*ibface);

      found = (inode == inibf[*ibface][0] ||
               inode == inibf[*ibface][1] ||
               inode == inibf[*ibface][2]) ? 1: 0;
    }
    while (*ibface <= nbface && found == 0);

    if (found == 0)
    {
      ug_error_message ("*** ERROR 334 : no valid starting face found for LS normal ***");
      return (334);
    }
  }

  // LS normal vector

  if (mbv_mode_ == 1)
  {
    // allocate temporary space for surface normal vectors

    bv_surf = (DOUBLE_3D *) ug_malloc (&ierr, (nsurfd+1) * sizeof (DOUBLE_3D));

    if (ierr)
    {
      ug_error_message ("*** ERROR 100333 : unable to allocate required memory ***");
      return (100333);
    }

    // set starting boundary face

    nsurf = ug3_bv_surf_loop (ibface, inode, nsurfd, 1, ibfibf, inibf,
                              bfv, bv_surf, x);

    // loop counter clockwise around node and
    // determine total number of unique surface normal vectors and
    // set surface normal vector for each surface

    nsurf = ug3_bv_surf_loop (ibface, inode, nsurfd, 0, ibfibf, inibf,
                              bfv, bv_surf, x);

    // if there is sufficient space for number of surfaces then
    // determine least-squares normal vector from surface normal vectors
  
    if (nsurf > 0) 
      ug3_bv_ls_surf (nsurf, bv_surf, bv);

    // otherwise determine simple average normal vector
  
    else
      ug3_bv_avg (*ibface, inode, ibfibf, inibf, bfv, bv, x);

    // free temporary space

    ug_free (bv_surf);
  }

  // most visible normal vector

  else if (mbv_mode_ == 2)
    ierr = ug3_bv_most (*ibface, mmsg, nbface, inode, ibfibf, inibf, bfv, bv, x);  

  // if there was an error with most visible or if averaging mode is selected
  // then use simple average normal vector

  if ((ierr && mbv_mode_ == 2) || mbv_mode_ == 3)
    ug3_bv_avg (*ibface, inode, ibfibf, inibf, bfv, bv, x);

  return (0);
}

void ug3_bv_ls_def_param (INT_ mbv_mode,
                          double ang_bv_disc,
                          double ang_bv_sharp,
                          double ang_bv_smooth,
                          double bv_det_tol,
                          double bv_diff_tol,
                          double bv_ev_sharp)
{
  // define max dbfvaiation angle between adjacent boundary face normal vectors
  // and sharp edge boundary face normal vector correction coefficient

  if (mbv_mode != mbv_mode_)
    mbv_mode_ = mbv_mode;

  if (ang_bv_disc != ang_bv_disc_)
  {
    bv_ls_set_param_ = 1;
    ang_bv_disc_ = ang_bv_disc;
  }

  if (ang_bv_sharp != ang_bv_sharp_)
  {
    bv_ls_set_param_ = 1;
    ang_bv_sharp_ = ang_bv_sharp;
  }

  if (ang_bv_smooth != ang_bv_smooth_)
  {
    bv_ls_set_param_ = 1;
    ang_bv_smooth_ = ang_bv_smooth;
  }

  if (bv_det_tol != bv_det_tol_)
    bv_det_tol_ = bv_det_tol;

  if (bv_diff_tol != bv_diff_tol_)
    bv_diff_tol_ = bv_diff_tol;

  if (bv_ev_sharp != bv_ev_sharp_)
    bv_ev_sharp_ = bv_ev_sharp;

  return;
}

void ug3_bv_ls_set_param (void)
{
  // set parameters for determining face normal vectors

  double cosdisc, cossharp, cossmooth, dc1, dc45;

  dc1 = 1.0;
  dc45 = 45.0;

  if (bv_ls_set_param_ == 1)
  {
    bv_ls_set_param_ = 0;

    cosdisc = cos (ang_bv_disc_ * atan (dc1) / dc45);
    cosdiscs_ = cosdisc * fabs (cosdisc);

    cossharp = cos (ang_bv_sharp_ * atan (dc1) / dc45);
    cossharps_ = cossharp * fabs (cossharp);

    cossmooth = cos (ang_bv_smooth_ * atan (dc1) / dc45);
    cossmooths_ = cossmooth * fabs (cossmooth);
  }

  return;
}

INT_ ug3_bv_surf_loop (INT_ *kbface,
                       INT_ inode,
                       INT_ nsurfd,
                       INT_ set_start,
                       INT_3D * ibfibf,
                       INT_3D * inibf,
                       DOUBLE_3D * bfv,
                       DOUBLE_3D * bv_surf,
                       DOUBLE_3D * x)
{
  // if set_start=1 then set starting boundary face for node
  // if set_start=0 set surface boundary normal vector for each surface that
  // surrounds a node by looping counter clockwise (or clockwise) around node

  DOUBLE_3D bv;

  INT_ ibface, ibfn, ibfn1, ibfn2, ibfn3, inode2, inode3, isurf,
       jbface, jsurf, kbface1, kbface2, ksurf, loop, nsurf;

  double bfvi1, bfvi2, bfvi3, bfvj1, bfvj2, bfvj3,
         d1, d2, d3, dc1, dx211, dx212, dx213, dx311, dx312, dx313,
         w, x11, x12, x13, x21, x22, x23, x31, x32, x33;

  dc1 = 1.0;

  // if in set start mode then loop clockwise and check for a bounding edge

  if (set_start)
  {
    // set initial boundary face

    ibface = *kbface;

    // set nodes of face

    ibfn1 = (inibf[ibface][0] == inode) ? 0:
            (inibf[ibface][1] == inode) ? 1: 2;
    ibfn2 = (ibfn1 == 2) ? 0 : ibfn1 + 1;
    ibfn3 = (ibfn2 == 2) ? 0 : ibfn2 + 1;

    inode2 = inibf[ibface][ibfn2];
    inode3 = inibf[ibface][ibfn3];

    // set direction of next boundary face to check

    ibfn = ibfn3;

    // save current boundary face as starting boundary face

    jbface = ibface;

    // set next boundary face

    ibface = ibfibf[ibface][ibfn];

    // loop around the given node

    while (ibface > 0 && ibface != *kbface)
    {
      // set nodes for face

      inode3 = inode2;

      ibfn3 = (inibf[ibface][0] == inode3) ? 0:
              (inibf[ibface][1] == inode3) ? 1: 2;
      ibfn2 = (ibfn3 == 0) ? 2 : ibfn3 - 1;

      ibfn = ibfn3;

      inode2 = inibf[ibface][ibfn2];

      // save current boundary face as starting boundary face

      jbface = ibface;

      // set next boundary face

      ibface = ibfibf[ibface][ibfn];
    }

    // if the local configuration of boundary faces surrounding the given node
    // is open then set the starting boundary face and exit
    // if the configuration is open then continue

    if (ibface <= 0)
    {
      *kbface = jbface;

      return (0);
    }
  }

  // set initial boundary face

  ibface = *kbface;

  // set nodes of face

  ibfn1 = (inibf[ibface][0] == inode) ? 0:
          (inibf[ibface][1] == inode) ? 1: 2;
  ibfn2 = (ibfn1 == 2) ? 0 : ibfn1 + 1;
  ibfn3 = (ibfn2 == 2) ? 0 : ibfn2 + 1;

  inode2 = inibf[ibface][ibfn2];
  inode3 = inibf[ibface][ibfn3];

  // if boundary face unit normals are set then get boundary face unit normal

  if (bfv)
  {
    bfvi1 = bfv[ibface][0];
    bfvi2 = bfv[ibface][1];
    bfvi3 = bfv[ibface][2];
  }

  // if boundary face unit normals are not set then
  // set coordinates of face, set delta vectors of face,
  // and determine boundary face unit normal

  else //if (bfv == NULL)
  {
    x11 = x[inode][0];
    x12 = x[inode][1];
    x13 = x[inode][2];
    x21 = x[inode2][0];
    x22 = x[inode2][1];
    x23 = x[inode2][2];
    x31 = x[inode3][0];
    x32 = x[inode3][1];
    x33 = x[inode3][2];

    dx211 = x21 - x11;
    dx212 = x22 - x12;
    dx213 = x23 - x13;
    dx311 = x31 - x11;
    dx312 = x32 - x12;
    dx313 = x33 - x13;

    bfvi1 = dx212 * dx313 - dx213 * dx312;
    bfvi2 = dx213 * dx311 - dx211 * dx313;
    bfvi3 = dx211 * dx312 - dx212 * dx311;

    w = dc1 / sqrt (bfvi1 * bfvi1 + bfvi2 * bfvi2 + bfvi3 * bfvi3);

    bfvi1 = w * bfvi1;
    bfvi2 = w * bfvi2;
    bfvi3 = w * bfvi3;
  }

  // set direction of next boundary face to check

  ibfn = ibfn2;

  // if not in set start mode then
  // initialize surface normal vector for starting surface

  if (set_start == 0)
  {
    isurf = 0;
    ksurf = 0;

    bv_surf[isurf][0] = bfvi1;
    bv_surf[isurf][1] = bfvi2;
    bv_surf[isurf][2] = bfvi3;
  }

  // set next boundary face

  ibface = ibfibf[ibface][ibfn];

  // loop around the given node

  loop = (ibface <= 0 || ibface == *kbface) ? 0: 1;

  kbface1 = 0;
  kbface2 = 0;

  while (loop)
  {
    // set nodes for face

    inode2 = inode3;

    ibfn2 = (inibf[ibface][0] == inode2) ? 0:
            (inibf[ibface][1] == inode2) ? 1: 2;
    ibfn3 = (ibfn2 == 2) ? 0 : ibfn2 + 1;

    inode3 = inibf[ibface][ibfn3];

    // set direction of next boundary face to check

    ibfn = ibfn2;

    // set boundary face unit normal for previous face

    bfvj1 = bfvi1;
    bfvj2 = bfvi2;
    bfvj3 = bfvi3;

    // if boundary face unit normals are set then get normal vector

    if (bfv)
    {
      bfvi1 = bfv[ibface][0];
      bfvi2 = bfv[ibface][1];
      bfvi3 = bfv[ibface][2];
    }

    // if boundary face unit normals are not set then
    // determine boundary face unit normal

    else //if (bfv == NULL)
    {
      x31 = x[inode3][0];
      x32 = x[inode3][1];
      x33 = x[inode3][2];

      dx211 = dx311;
      dx212 = dx312;
      dx213 = dx313;
      dx311 = x31 - x11;
      dx312 = x32 - x12;
      dx313 = x33 - x13;

      bfvi1 = dx212 * dx313 - dx213 * dx312;
      bfvi2 = dx213 * dx311 - dx211 * dx313;
      bfvi3 = dx211 * dx312 - dx212 * dx311;

      w = dc1 / sqrt (bfvi1 * bfvi1 + bfvi2 * bfvi2 + bfvi3 * bfvi3);

      bfvi1 = w * bfvi1;
      bfvi2 = w * bfvi2;
      bfvi3 = w * bfvi3;
    }

    // set signed vector dot product squared for adjacent faces

    w = bfvi1 * bfvj1 + bfvi2 * bfvj2 + bfvi3 * bfvj3;
    w = w * fabs (w);

    // if not in set start mode then set surface normal vector

    if (set_start == 0)
    {
      // if we are not at the end and
      // if boundary face is part of a smooth surface then
      // add boundary face unit normal vector to the surface normal vector

      if (loop == 1 && w >= cossmooths_)
      {
        bv_surf[isurf][0] = bv_surf[isurf][0] + bfvi1;
        bv_surf[isurf][1] = bv_surf[isurf][1] + bfvi2;
        bv_surf[isurf][2] = bv_surf[isurf][2] + bfvi3;
      }

      // if boundary face is not smooth then create a new surface

      else if (w < cossmooths_)
      {
        // if we are not at the end then
        // create and initialize new surface normal vector

        if (loop == 1)
        {
          ++isurf;

          if (isurf >= nsurfd)
            return (-1);

          bv_surf[isurf][0] = bfvi1;
          bv_surf[isurf][1] = bfvi2;
          bv_surf[isurf][2] = bfvi3;
        }

        // if the edge is sharp then correct surface normal vectors

        if (w <= cossharps_)
        {
          // set edge vector

          dx211 = x[inode2][0] - x[inode][0];
          dx212 = x[inode2][1] - x[inode][1];
          dx213 = x[inode2][2] - x[inode][2];

          // correct previous surface normal vector for sharp edge

          d1 = bfvj2 * dx213 - bfvj3 * dx212;
          d2 = bfvj3 * dx211 - bfvj1 * dx213;
          d3 = bfvj1 * dx212 - bfvj2 * dx211;

          w = bv_ev_sharp_ / sqrt (d1 * d1 + d2 * d2 + d3 * d3);

          jsurf = (loop == 1) ? isurf-1: isurf;

          bv_surf[jsurf][0] = bv_surf[jsurf][0] + w * d1;
          bv_surf[jsurf][1] = bv_surf[jsurf][1] + w * d2;
          bv_surf[jsurf][2] = bv_surf[jsurf][2] + w * d3;

          // correct current surface normal vector for sharp edge

          d1 = dx212 * bfvi3 - dx213 * bfvi2;
          d2 = dx213 * bfvi1 - dx211 * bfvi3;
          d3 = dx211 * bfvi2 - dx212 * bfvi1;

          w = bv_ev_sharp_ / sqrt (d1 * d1 + d2 * d2 + d3 * d3);

          jsurf = (loop == 1) ? isurf: 0;

          bv_surf[jsurf][0] = bv_surf[jsurf][0] + w * d1;
          bv_surf[jsurf][1] = bv_surf[jsurf][1] + w * d2;
          bv_surf[jsurf][2] = bv_surf[jsurf][2] + w * d3;
        }

        // if the edge is discontinuous then evaluate surface normal vector
        // here we evaluate the surface normal vector for the current continuous
        // surface using a least-squares unit normal vector

        if (w <= cosdiscs_)
        {
          // if the current continuous surface is composed of two or more
          // surface segments then evaluate the normal vector for the complete
          // continuous surface

          jsurf = (loop == 1) ? isurf-1: isurf;

          if (jsurf > ksurf)
          {
            nsurf = jsurf - ksurf + 1;

            // determine least-squares unit normal vector

            ug3_bv_ls_surf (nsurf, &(bv_surf[ksurf]), bv);

            bv_surf[ksurf][0] = bv[0];
            bv_surf[ksurf][1] = bv[1];
            bv_surf[ksurf][2] = bv[2];

            // if we are not at the end then reset the location of the next
            // surface segment and start a new continuous surface
            // this eliminates all of the now unused ones surface segments used
            // for the least-squares evaluation

            if (loop == 1)
            {
              ++ksurf;

              bv_surf[ksurf][0] = bv_surf[isurf][0];
              bv_surf[ksurf][1] = bv_surf[isurf][1];
              bv_surf[ksurf][2] = bv_surf[isurf][2];
            }

            isurf = ksurf;
          }

          // start a new continuous surface

          else
            ksurf = isurf;
        }
      }
    }

    // if in set start mode and a non-smooth or discontinuous boundary face was
    // found then set possible starting boundary face values

    else
    {
      if (w <= cosdiscs_)
        kbface1 = ibface;

      else if (w < cossmooths_)
        kbface2 = ibface;
    }

    // if we are not at the end then
    // check if we should continue the loop and set next boundary face

    if (loop == 1)
    {
      // otherwise set next boundary face

      ibface = ibfibf[ibface][ibfn];

      // if we are not in set start mode and if we are at the end then
      // set loop flag so that only the end is evaluated

      if (set_start == 0 && ibface == *kbface)
          loop = -1;

      // if the loop is complete then set loop flag to exit 

      else if (ibface <= 0 || ibface == *kbface)
        loop = 0;
    }

    // if we are at the end then set flag to exit the loop

    else //if (loop == -1)
      loop = 0;
  }

  // if in set start mode then set starting face to the either the last
  // discontinuous or smooth face junction found
  // priority is given to the last discontinuous face junction if found
  // if neither are found don't change starting face value

  if (set_start)
  {
    if (kbface1)
      *kbface = kbface1;
    else if (kbface2)
      *kbface = kbface2;
  }

  // set number of continuous surfaces found

  nsurf = (set_start) ? 0: isurf+1;

  return (nsurf);
}

void ug3_bv_ls_surf (INT_ nsurf,
                     DOUBLE_3D * bv_surf,
                     DOUBLE_3D bv)
{
  // determine least-squares optimized normal vector from a set of surface
  // normal vectors

  INT_ isurf, jsurf, ksurf, nsurfm;

  double a11, a12, a13, b1, b2, b3, bu1, bu2, c1, c2, c3,
         c11, c12, c13, c21, c22, c23, c31, c32, c33, d1, d2, d3, dc0, dc1, det,
         tu11, tu12, tu13, tu21, tu22, tu23, w;

  dc0 = 0.0;
  dc1 = 1.0;

  // normalize surface normal vectors for each surface

  for (isurf = 0; isurf < nsurf; ++isurf)
  {
    w = dc1 / sqrt (bv_surf[isurf][0] * bv_surf[isurf][0]
                  + bv_surf[isurf][1] * bv_surf[isurf][1]
                  + bv_surf[isurf][2] * bv_surf[isurf][2]);

    bv_surf[isurf][0] = w * bv_surf[isurf][0];
    bv_surf[isurf][1] = w * bv_surf[isurf][1];
    bv_surf[isurf][2] = w * bv_surf[isurf][2];
  }

  // check for and eliminate redundant surface normal vectors

  ksurf = -1;

  for (isurf = 0; isurf < nsurf; ++isurf)
  {
    if (bv_surf[isurf][0] != dc0 || bv_surf[isurf][1] != dc0 || bv_surf[isurf][2] != dc0)
    {
      for (jsurf = isurf+1; jsurf < nsurf; ++jsurf)
      {
        // check for redundant surface normal vectors

        d1 = fabs (bv_surf[jsurf][0] - bv_surf[isurf][0]);
        d2 = fabs (bv_surf[jsurf][1] - bv_surf[isurf][1]);
        d3 = fabs (bv_surf[jsurf][2] - bv_surf[isurf][2]);

        w = MAX (d1, d2);
        w = MAX (w, d3);

        // if vectors are redundant then average and normalize them

        if (w <= bv_diff_tol_)
        {
          bv_surf[isurf][0] = bv_surf[isurf][0] + bv_surf[jsurf][0];
          bv_surf[isurf][1] = bv_surf[isurf][1] + bv_surf[jsurf][1];
          bv_surf[isurf][2] = bv_surf[isurf][2] + bv_surf[jsurf][2];

          w = dc1 / sqrt (bv_surf[isurf][0] * bv_surf[isurf][0]
                        + bv_surf[isurf][1] * bv_surf[isurf][1]
                        + bv_surf[isurf][2] * bv_surf[isurf][2]);

          bv_surf[isurf][0] = w * bv_surf[isurf][0];
          bv_surf[isurf][1] = w * bv_surf[isurf][1];
          bv_surf[isurf][2] = w * bv_surf[isurf][2];

          bv_surf[jsurf][0] = dc0;
          bv_surf[jsurf][1] = dc0;
          bv_surf[jsurf][2] = dc0;
        }
      }

      // if previous redundant vectors were found then shift data locations

      ++ksurf;

      if (ksurf < isurf)
      {
        bv_surf[ksurf][0] = bv_surf[isurf][0];
        bv_surf[ksurf][1] = bv_surf[isurf][1];
        bv_surf[ksurf][2] = bv_surf[isurf][2];
      }
    }
  }

  // set final number of unique surfaces

  nsurfm = ksurf+1;

  // set average normal vector to sum of unique surface normal vectors
  // if there is only one surface then this is always used
  // otherwise this is only used when the determinant is too small

  bv[0] = dc0;
  bv[1] = dc0;
  bv[2] = dc0;

  for (isurf = 0; isurf < nsurfm; ++isurf)
  {
    bv[0] = bv[0] + bv_surf[isurf][0];
    bv[1] = bv[1] + bv_surf[isurf][1];
    bv[2] = bv[2] + bv_surf[isurf][2];
  }

  // if there are only two surfaces then
  // set normal vector using 2D least-squares optimization

  if (nsurfm == 2)
  {
    // determine vector normal to the plane of the surface vectors

    b1 = bv_surf[0][1] * bv_surf[1][2] - bv_surf[1][1] * bv_surf[0][2];
    b2 = bv_surf[1][0] * bv_surf[0][2] - bv_surf[0][0] * bv_surf[1][2];
    b3 = bv_surf[0][0] * bv_surf[1][1] - bv_surf[1][0] * bv_surf[0][1];

    // determine the transformation vectors for surface vector plane

    ug3_proj_vector (b1, b2, b3, &tu11, &tu12, &tu13, &tu21, &tu22, &tu23);

    // project surface 0 vector to plane

    ug3_proj_uv_from_xyz (&bu1, &bu2,
                          bv_surf[0][0], bv_surf[0][1], bv_surf[0][2],
                          tu11, tu12, tu13, tu21, tu22, tu23);

    // set least-squares coefficients

    c1 = bu1;
    c2 = bu2;

    c11 = bu1 * bu1;
    c12 = bu1 * bu2;
    c22 = bu2 * bu2;

    // project surface 1 vector to plane

    ug3_proj_uv_from_xyz (&bu1, &bu2,
                          bv_surf[1][0], bv_surf[1][1], bv_surf[1][2],
                          tu11, tu12, tu13, tu21, tu22, tu23);

    // set and sum least-squares coefficients

    c1 = c1 + bu1;
    c2 = c2 + bu2;

    c11 = c11 + bu1 * bu1;
    c12 = c12 + bu1 * bu2;
    c22 = c22 + bu2 * bu2;

    // set symmetric coefficient

    c21 = c12;

    // normalize row 1 least-squares coefficients

    w = MAX (fabs (c11), fabs (c12));
    w = dc1 / w;

    c1 = w * c1;

    c11 = w * c11;
    c12 = w * c12;

    // normalize row 2 least-squares coefficients

    w = MAX (fabs (c21), fabs (c22));
    w = dc1 / w;

    c2 = w * c2;

    c21 = w * c21;
    c22 = w * c22;

    det = c11 * c22 - c21 * c12;

    // if determinant is not too small then
    // solve 2D least-squares system of equations for vector in the plane

    if (fabs (det) > bv_det_tol_)
    {
      w = dc1 / det;

      bu1 = w * (c1 * c22 - c2 * c12);
      bu2 = w * (c11 * c2 - c21 * c1);

      // set final least-squares normal vector by transforming vector in the
      // plane to xyz physical space

      ug3_proj_xyz_from_uv (bu1, bu2,
                            &b1, &b2, &b3,
                            tu11, tu12, tu13, tu21, tu22, tu23);

      bv[0] = b1;
      bv[1] = b2;
      bv[2] = b3;
    }
  }

  // if there are three or more surfaces then
  // set normal vector using 3D least-squares optimization

  else if (nsurfm >= 3)
  {
    // initialize least-squares coefficients

    c1 = dc0;
    c2 = dc0;
    c3 = dc0;

    c11 = dc0;
    c12 = dc0;
    c13 = dc0;
    c22 = dc0;
    c23 = dc0;
    c33 = dc0;

    //loop over all surfaces

    for (isurf = 0; isurf < nsurfm; ++isurf)
    {
      // set and sum least-squares coefficients

      c1 = c1 + bv_surf[isurf][0];
      c2 = c2 + bv_surf[isurf][1];
      c3 = c3 + bv_surf[isurf][2];

      c11 = c11 + bv_surf[isurf][0] * bv_surf[isurf][0];
      c12 = c12 + bv_surf[isurf][0] * bv_surf[isurf][1];
      c13 = c13 + bv_surf[isurf][0] * bv_surf[isurf][2];
      c22 = c22 + bv_surf[isurf][1] * bv_surf[isurf][1];
      c23 = c23 + bv_surf[isurf][1] * bv_surf[isurf][2];
      c33 = c33 + bv_surf[isurf][2] * bv_surf[isurf][2];
    }

    // set symmetric coefficient

    c21 = c12;
    c31 = c13;
    c32 = c23;

    // normalize row 1 least-squares coefficients

    w = MAX (fabs (c11), fabs (c12));
    w = MAX (fabs (c13), w);
    w = dc1 / w;

    c1 = w * c1;

    c11 = w * c11;
    c12 = w * c12;
    c13 = w * c13;

    // normalize row 2 least-squares coefficients

    w = MAX (fabs (c21), fabs (c22));
    w = MAX (fabs (c23), w);
    w = dc1 / w;

    c2 = w * c2;

    c21 = w * c21;
    c22 = w * c22;
    c23 = w * c23;

    // normalize row 3 least-squares coefficients

    w = MAX (fabs (c31), fabs (c32));
    w = MAX (fabs (c33), w);
    w = dc1 / w;

    c3 = w * c3;

    c31 = w * c31;
    c32 = w * c32;
    c33 = w * c33;

    // determine determinant

    a11 = c22 * c33 - c32 * c23;
    a12 = c21 * c33 - c31 * c23;
    a13 = c21 * c32 - c31 * c22;

    det = c11 * a11 - c12 * a12 + c13 * a13;

    // if determinant is not too small then
    // solve 3D least-squares system of equations for normal vector

    if (fabs (det) > bv_det_tol_)
    {
      w = dc1 / det;

      bv[0] = w * (c1 * a11
                 - c2 * (c12 * c33 - c32 * c13)
                 + c3 * (c12 * c23 - c22 * c13));
      bv[1] = w * (- c1 * a12
                   + c2 * (c11 * c33 - c31 * c13)
                   - c3 * (c11 * c23 - c21 * c13));
      bv[2] = w * (c1 * a13
                 - c2 * (c11 * c32 - c31 * c12)
                 + c3 * (c11 * c22 - c21 * c12));
    }
  }

  // normalize least-squares normal vector

  w = dc1 / sqrt (bv[0] * bv[0] + bv[1] * bv[1] + bv[2] * bv[2]);

  bv[0] = w * bv[0];
  bv[1] = w * bv[1];
  bv[2] = w * bv[2];

  return;
}

void ug3_bv_avg (INT_ kbface,
                 INT_ inode,
                 INT_3D * ibfibf,
                 INT_3D * inibf,
                 DOUBLE_3D * bfv,
                 DOUBLE_3D bv,
                 DOUBLE_3D * x)
{
  // set surface boundary normal vector to the average of the normals for each
  // surface face that surrounds the given node

  INT_ ibface, ibfn1, ibfn2, ibfn3, inode2, inode3;

  double bfv1, bfv2, bfv3, dc0, dc1, dx211, dx212, dx213, dx311, dx312, dx313,
         w, x11, x12, x13, x21, x22, x23, x31, x32, x33;

  dc0 = 0.0;
  dc1 = 1.0;

  // set initial boundary face

  ibface = kbface;

  // initialize surface normal vector sum

  bv[0] = dc0;
  bv[1] = dc0;
  bv[2] = dc0;

  // set node coordinates

  if (bfv == NULL)
  {
    x11 = x[inode][0];
    x12 = x[inode][1];
    x13 = x[inode][2];
  }

  // loop around the given node

  do
  {
    // set nodes for face

    ibfn1 = (inibf[ibface][0] == inode) ? 0:
            (inibf[ibface][1] == inode) ? 1: 2;
    ibfn2 = (ibfn1 == 2) ? 0 : ibfn1 + 1;
    ibfn3 = (ibfn2 == 2) ? 0 : ibfn2 + 1;

    // if boundary face normals are set then get boundary face normals and
    // add to the surface normal vector sum

    if (bfv)
    {
      bv[0] = bv[0] + bfv[ibface][0];
      bv[1] = bv[1] + bfv[ibface][1];
      bv[2] = bv[2] + bfv[ibface][2];
    }

    // if boundary face normals are not set then
    // set coordinates of face, set delta vectors of face, and
    // determine boundary face normals and
    // add to the surface normal vector sum

    else //if (bfv == NULL)
    {
      inode2 = inibf[ibface][ibfn2];
      inode3 = inibf[ibface][ibfn3];

      x21 = x[inode2][0];
      x22 = x[inode2][1];
      x23 = x[inode2][2];
      x31 = x[inode3][0];
      x32 = x[inode3][1];
      x33 = x[inode3][2];

      dx211 = x21 - x11;
      dx212 = x22 - x12;
      dx213 = x23 - x13;
      dx311 = x31 - x11;
      dx312 = x32 - x12;
      dx313 = x33 - x13;

      bfv1 = dx212 * dx313 - dx213 * dx312;
      bfv2 = dx213 * dx311 - dx211 * dx313;
      bfv3 = dx211 * dx312 - dx212 * dx311;

      w = dc1 / sqrt (bfv1 * bfv1 + bfv2 * bfv2 + bfv3 * bfv3);

      bv[0] = bv[0] + w * bfv1;
      bv[1] = bv[1] + w * bfv2;
      bv[2] = bv[2] + w * bfv3;
    }

    // set next boundary face

    ibface = ibfibf[ibface][ibfn2];
  }
  while (ibface > 0 && ibface != kbface);

  // normalize average surface normal vectors

  w = dc1 / sqrt (bv[0] * bv[0] + bv[1] * bv[1] + bv[2] * bv[2]);

  bv[0] = w * bv[0];
  bv[1] = w * bv[1];
  bv[2] = w * bv[2];

  return;
}

static INT_ ug3_bv_most (INT_ kbface,
                         INT_ mmsg, 
                         INT_ nbface, 
                         INT_ inode,
                         INT_3D * ibfibf,
                         INT_3D * inibf,
                         DOUBLE_3D * bfv,
                         DOUBLE_3D bv,
                         DOUBLE_3D * x)
{
  // set surface boundary normal vector to the average of the normals for each
  // surface face that surrounds the given node

  CHAR_UG_MAX Text;

  INT_ ibface, ibfn1, ibfn2, ibfn3, inode2, inode3;
  INT_ nbrbvf;
  INT_ ierr = 0;
  INT_ nbrbvfd = 128;
 
  double bfv1, bfv2, bfv3, dc0, dc1, dx211, dx212, dx213, dx311, dx312, dx313,
         nrm, w, x11, x12, x13, x21, x22, x23, x31, x32, x33;

  DOUBLE_3D *bvface = NULL;

  dc0 = 0.0;
  dc1 = 1.0;

  // set initial boundary face

  ibface = kbface;

  // initialize surface normal vector sum

  bv[0] = dc0;
  bv[1] = dc0;
  bv[2] = dc0;

  // set node coordinates

  if (bfv == NULL)
  {
    x11 = x[inode][0];
    x12 = x[inode][1];
    x13 = x[inode][2];
  }

  // allocate temporary space

  bvface = (DOUBLE_3D *) ug_malloc (&ierr, (nbrbvfd+2) * sizeof (DOUBLE_3D));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100337 : unable to allocate required memory ***");
    return (100337);
  }

  // loop around the given node : get all faces normals 
  nbrbvf = -1;

  do
  {
    // set nodes for face
    if ( ibface < 1 || ibface > nbface ) {
      snprintf (Text, sizeof(Text), "*** ERROR 335 : face index %d out of bound = [1,%d] ***", (int) ibface, (int) nbface);
      ug_error_message (Text);
      return (335);
    }

    ibfn1 = (inibf[ibface][0] == inode) ? 0:
            (inibf[ibface][1] == inode) ? 1: 2;
    ibfn2 = (ibfn1 == 2) ? 0 : ibfn1 + 1;
    ibfn3 = (ibfn2 == 2) ? 0 : ibfn2 + 1;

    // if boundary face normals are set then get boundary face normals and
    // add to the surface normal vector sum
    
    nbrbvf++;
    if ( nbrbvf >= nbrbvfd ) {
      ug_free (bvface);
      snprintf (Text, sizeof(Text), "*** ERROR 350 : too many faces in vertex %d ball ***", (int) inode);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "*** number of faces is greater than %d ***", (int) nbrbvf);
      ug_message (Text);
      return (350);
    }

    if (bfv)
    {
      bvface[nbrbvf][0] = bfv[ibface][0];
      bvface[nbrbvf][1] = bfv[ibface][1];
      bvface[nbrbvf][2] = bfv[ibface][2];
    }

    // if boundary face normals are not set then
    // set coordinates of face, set delta vectors of face, and
    // determine boundary face normals and
    // add to the surface normal vector sum

    else //if (bfv == NULL)
    {
      inode2 = inibf[ibface][ibfn2];
      inode3 = inibf[ibface][ibfn3];

      x21 = x[inode2][0];
      x22 = x[inode2][1];
      x23 = x[inode2][2];
      x31 = x[inode3][0];
      x32 = x[inode3][1];
      x33 = x[inode3][2];

      dx211 = x21 - x11;
      dx212 = x22 - x12;
      dx213 = x23 - x13;
      dx311 = x31 - x11;
      dx312 = x32 - x12;
      dx313 = x33 - x13;

      bfv1 = dx212 * dx313 - dx213 * dx312;
      bfv2 = dx213 * dx311 - dx211 * dx313;
      bfv3 = dx211 * dx312 - dx212 * dx311;

      w = dc1 / sqrt (bfv1 * bfv1 + bfv2 * bfv2 + bfv3 * bfv3);

      bvface[nbrbvf][0] = w * bfv1;
      bvface[nbrbvf][1] = w * bfv2;
      bvface[nbrbvf][2] = w * bfv3;
    }

    // set next boundary face

    ibface = ibfibf[ibface][ibfn2];
  }
  while (ibface > 0 && ibface != kbface);
  
  // compute the most visible normal
  
  nbrbvf++;   // should increment the nbr of faces because we start at -1
  
  ierr = ug3_MostNormalVertexNormal(inode, mmsg, nbrbvf, bvface, bv);

  ug_free (bvface);
  
  if (ierr)
    return (ierr);

  // normalize surface normal vectors
  
  nrm = sqrt (bv[0] * bv[0] + bv[1] * bv[1] + bv[2] * bv[2]);  
  
  if ( nrm < 1.e-30 ) {
    snprintf (Text, sizeof(Text),"*** ERROR 351 : an output vertex normal is nul for node %d ***", (int) inode);
    ug_error_message (Text);
    return (351);
  }
  if ( isnan(nrm) ) {
    snprintf (Text, sizeof(Text), "*** ERROR 352 : an output vertex normal is nan (%le %le %le) for node %d ***", bv[0], bv[1], bv[2], (int) inode);
    ug_error_message (Text);
    return (352);
  }

  ierr = 0;

  w = dc1 / nrm;
  
  bv[0] = w * bv[0];
  bv[1] = w * bv[1];
  bv[2] = w * bv[2];

  return 0;
}

static INT_ ug3_MostNormalVertexNormal (INT_ iVer,
                                        INT_ mmsg,
                                        INT_ nbrNor,
                                        DOUBLE_3D *vecNor,
                                        DOUBLE_3D mosNor)
{
  CHAR_UG_MAX Text;
  INT_   ii, jj, iconv, iter, niter;
  INT_   kk, ll;
  double relax1, relax2, tol1, tol2, scal;
  double epsil, epsil2, tolcircle, tolphimax;
  double rnx, rny, rnz, rnl, cfact, cangle, phi, phimax, csca;
  double rnx1, rny1, rnz1, rnxold, rnyold, rnzold, rsum, rax, ray, raz;
  double scalmin, scal1;
  double rnl1, phimax1, rnxb, rnyb, rnzb;
  double rnrx, rnry, rnrz;
  double scalmin1, scalt, rx12, ry12, rz12, rx23, ry23, rz23;
  double rx31, ry31, rz31, denom, rxc, ryc, rzc;
  
  INT_   ierr = 0;
  INT_   mnolo = 800;
  DOUBLE_1D *rangle = NULL;
  DOUBLE_1D *rfactor = NULL;
  DOUBLE_1D *rfactor2 = NULL;
  
  //--- Init
  epsil     =  1.e-12;
  epsil2    =  1.e-6;
  tolcircle =  1.e-6;
  tolphimax =  0.1;
  iconv     =  0;

  //--- Parameters
  niter  = 500;           // max iter number
  relax1 = 0.4;           // relaxation parameter   
  relax2 = 1. - relax1;   // relaxation parameter  
  tol1   = 1.e-4;          // convergence       
  tol2   = 1.e-2;          // verification between both methods  

  rnrx   = 0.;
  rnry   = 0.;
  rnrz   = 0.;
  
  if ( nbrNor > mnolo ) {
    snprintf (Text, sizeof(Text), "*** ERROR 353 : too many faces in vertex %d ball ***", (int) iVer);
    ug_error_message (Text);
    snprintf (Text, sizeof(Text), "*** number of faces is %d ***", (int) nbrNor);
    ug_error_message (Text);
    mosNor[0] = 0.;
    mosNor[1] = 0.;
    mosNor[2] = 0.;
    return (353);
  }
  if ( nbrNor == 0 ) {
    snprintf (Text, sizeof(Text), "*** ERROR 354 : No face in vertex %d ball ***", (int) iVer);
    ug_error_message (Text);
    snprintf (Text, sizeof(Text), "*** number of faces is %d ***", (int) nbrNor);
    ug_error_message (Text);
    mosNor[0] = 0.;
    mosNor[1] = 0.;
    mosNor[2] = 0.;
    return (354);
  }
  
  //--- check input face normals
  for (ii=0; ii<nbrNor; ++ii) {
    rnl = vecNor[ii][0]*vecNor[ii][0] + vecNor[ii][1]*vecNor[ii][1] + vecNor[ii][2]*vecNor[ii][2];
    if ( rnl < 1.e-60 ) {
      snprintf (Text, sizeof(Text),"*** ERROR 355 : an input face normal is nul for node %d ***", (int) iVer);
      ug_error_message (Text);
      return (355);
    }
    if ( fabs(rnl-1.) > 1.e-12 ) {
      snprintf (Text, sizeof(Text),"*** ERROR 356 : an input face normal is not 1 for node %d ***", (int) iVer);
      ug_error_message (Text);
      return (356);
    }
    if ( isnan(rnl) ) {
      snprintf (Text, sizeof(Text),"*** ERROR 357 : an input face normal is nan for node %d ***", (int) iVer);
      ug_error_message (Text);
      return (357);
    }
  }

  // allocate temporary space for surface normal vectors

  rangle = (DOUBLE_1D *) ug_malloc (&ierr, (mnolo+1) * sizeof (DOUBLE_1D));
  rfactor = (DOUBLE_1D *) ug_malloc (&ierr, (mnolo+1) * sizeof (DOUBLE_1D));
  rfactor2 = (DOUBLE_1D *) ug_malloc (&ierr, (mnolo+1) * sizeof (DOUBLE_1D));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100338 : unable to allocate required memory ***");
    return (100338);
  }
  
  //  ---------------------------------------------------
  //  -----  First method: Variation on Pirzadeh
  //  ---------------------------------------------------

  //--- initialization  
  rnx   = 0.;
  rny   = 0.;
  rnz   = 0.;
  cfact = 1. / (double)nbrNor;
  
  for (ii=0; ii<nbrNor; ++ii) {
    rfactor[ii] = cfact;
    
    rnx += vecNor[ii][0];
    rny += vecNor[ii][1];
    rnz += vecNor[ii][2];
  }
  rnl = sqrt(rnx*rnx + rny*rny + rnz*rnz);
  if ( rnl < 1.e-30 ) {
    ug_free (rangle);
    ug_free (rfactor);
    ug_free (rfactor2);
    snprintf (Text, sizeof(Text),"*** ERROR 358 : rnl is zero => the vertex faces ball is closed ***");
    ug_error_message (Text);
    return (358);
  }
  rnl = 1. / rnl;
  rnx = rnl * rnx;
  rny = rnl * rny;
  rnz = rnl * rnz;

  //--- evaluating initial angle
  cangle = 0.;
  phimax = 0.;
  for (ii=0; ii<nbrNor; ++ii) {
    csca = rnx*vecNor[ii][0] + rny*vecNor[ii][1] + rnz*vecNor[ii][2];
    phi  = acos( MAX(-1., MIN(1., csca)) );
    if ( phi > phimax )
      phimax = phi;
      
    rangle[ii] = phi;
    cangle    += phi;
  }
  rnxold    = rnx;
  rnyold    = rny;
  rnzold    = rnz;

  if ( phimax < tolphimax ) {
    //- normal normal to all faces
    iter = 1;
    goto EndIteLoop2;
    // goto 2999;
  }

  //--- iteration loop
  for (iter=1; iter<=niter; ++iter) {

    //--- correcting the factors
    rsum = 0.;
    for (ii=0; ii<nbrNor; ++ii) {
      rfactor2[ii]  = rfactor[ii] * (rangle[ii]/cangle);
      rsum         += rfactor2[ii];
    }

    //--- new weighting factor
    for (ii=0; ii<nbrNor; ++ii) 
      rfactor[ii] = rfactor2[ii] / rsum;

    //--- new vector
    rnx1 = 0.;
    rny1 = 0.;
    rnz1 = 0.;
    for (ii=0; ii<nbrNor; ++ii) {
      rnx1 += rfactor[ii] * vecNor[ii][0];
      rny1 += rfactor[ii] * vecNor[ii][1];
      rnz1 += rfactor[ii] * vecNor[ii][2];
    }
    rnl1 = sqrt(rnx1*rnx1 + rny1*rny1 + rnz1*rnz1);
    rnl1 = 1. / rnl1;
    rnx1 = rnx1 * rnl1;
    rny1 = rny1 * rnl1;
    rnz1 = rnz1 * rnl1;

    //--- new relaxed vector
    rnx = relax1*rnx1 + relax2*rnxold;
    rny = relax1*rny1 + relax2*rnyold;
    rnz = relax1*rnz1 + relax2*rnzold;
     
    rnl = sqrt(rnx*rnx + rny*rny + rnz*rnz);
    if ( rnl < 1.e-30 ) {
      ug_free (rangle);
      ug_free (rfactor);
      ug_free (rfactor2);
      snprintf (Text, sizeof(Text),"*** ERROR 359 : rnl is zero ***");
      ug_error_message (Text);
      return (359);
    }
    rnl = 1. / rnl;
    rnx = rnx * rnl;
    rny = rny * rnl;
    rnz = rnz * rnl;
     
    //--- evaluating angle
    cangle = 0.;
    phimax = 0.;
    
    for (ii=0; ii<nbrNor; ++ii) {
      csca = rnx*vecNor[ii][0] + rny*vecNor[ii][1] + rnz*vecNor[ii][2];
      phi  = acos( MAX(-1., MIN(1., csca)) );
      if ( phi > phimax )
        phimax = phi;
      rangle[ii]  = phi;
      cangle     += phi;
    }

    if ( phimax < tolphimax )
      goto EndIteLoop2;
      //goto 2999
     
    //--- computing difference
    rax = rnx - rnxold;
    ray = rny - rnyold;
    raz = rnz - rnzold;
    rnl = sqrt(rax*rax + ray*ray + raz*raz);
    if ( rnl < tol1 )
      goto EndIteLoop1;
      //goto 1401;

    //--- update old vector and angle 
    rnxold    = rnx;
    rnyold    = rny;
    rnzold    = rnz;
  }    // end iteration loop

  EndIteLoop1:
  //1401 continue 

  if ( iter == (niter+1) )
    iconv = 1;
  else
    iconv = 0;

  EndIteLoop2:
  //2999 continue

  phimax1 = phimax;
  
//  ---------------------------------------------------
//  -----  Second method: Aubry 'most visible' normal
//  ---------------------------------------------------
  
  //--- find the smallest circumcenter
  //--- build all the cercles with two points
  
  //--- if only one surface
  if ( nbrNor == 1 ) {
    rnrx = vecNor[0][0];
    rnry = vecNor[0][1];
    rnrz = vecNor[0][2];
    goto Verification;
    //goto 8000
  }

  //--- initialization
  rnx1 = 0.;
  rny1 = 0.;
  rnz1 = 0.;
  for (ii=0; ii<nbrNor; ++ii) {
    rnx1 += vecNor[ii][0];
    rny1 += vecNor[ii][1];
    rnz1 += vecNor[ii][2];
  }
  rnl1 = sqrt(rnx1*rnx1 + rny1*rny1 + rnz1*rnz1);
  rnl1 = 1. / rnl1;
  rnx1 = rnl1 * rnx1;
  rny1 = rnl1 * rny1;
  rnz1 = rnl1 * rnz1;
     
  //--- evaluating initial angle
  cangle = 0.;
  phimax = 0.; 
  for (ii=0; ii<nbrNor; ++ii) {
    csca = rnx1*vecNor[ii][0] + rny1*vecNor[ii][1] + rnz1*vecNor[ii][2];
    phi  = acos( MAX(-1., MIN(1., csca)) );
    if ( phi > phimax )
      phimax = phi;
    cangle += phi;
  }

  if (phimax < tolphimax ) {   
    //- normal normal to all faces
    rnrx = rnx1;
    rnry = rny1;
    rnrz = rnz1;
    goto Verification;
    //goto 8000
  }

  scalmin  = -1.;
  scalmin1 = -1.;
  
  for (ii=0; ii<nbrNor; ++ii) {
    for (jj=ii+1; jj<nbrNor; ++jj) {

      //--- compute the bisector (center in the plane)
      //--- RQ: this could be zero if two faces are flip over. We get this case on the spar: (1,0,0) and (-1,0,0)
      rnxb = vecNor[ii][0] + vecNor[jj][0];
      rnyb = vecNor[ii][1] + vecNor[jj][1];
      rnzb = vecNor[ii][2] + vecNor[jj][2];
      rnl  = sqrt(rnxb*rnxb + rnyb*rnyb + rnzb*rnzb);
      if ( rnl < 1.e-30 ) {
        ug_free (rangle);
        ug_free (rfactor);
        ug_free (rfactor2);
        snprintf (Text, sizeof(Text),"*** ERROR 360 : rnl is zero => two opposite normals in the vertex faces ball ***");
        ug_error_message (Text);
        return (360);
      }
      rnl  = 1. / rnl;
      rnxb = rnxb * rnl;
      rnyb = rnyb * rnl;
      rnzb = rnzb * rnl;

      //--- compute the scalar product (radius in the plane)
      scal = rnxb*vecNor[ii][0] + rnyb*vecNor[ii][1] + rnzb*vecNor[ii][2];

      //--- check that this radius is smaller that the current smallest radius
      if ( scal < scalmin1 )
        goto NexVec1;
        //goto 4199;

      //--- set the tolerance
      scal1 = scal - tolcircle;
     
      //--- compute the maximal distance
      for (kk=0; kk<nbrNor; ++kk) {
        if ( kk != ii && kk != jj ) {
          scalt = rnxb*vecNor[kk][0] + rnyb*vecNor[kk][1] + rnzb*vecNor[kk][2];
          if ( scalt < scal1 )
            goto NexVec1;
            //goto 4199
        }
      }

      //--- have found a circumcercle, store it            
      scalmin  = scal;
      scalmin1 = scalmin + tolcircle;
      rnrx     = rnxb;
      rnry     = rnyb;
      rnrz     = rnzb;

      NexVec1:
      continue;
    }
  }

  //--- check all the combinations of three points
  for (ii=0; ii<nbrNor; ++ii) {
    for (jj=ii+1; jj<nbrNor; ++jj) {
      for (kk=jj+1; kk<nbrNor; ++kk) {

        //--- compute the center of the circle 
        rx12 = vecNor[ii][0] - vecNor[jj][0];
        ry12 = vecNor[ii][1] - vecNor[jj][1];
        rz12 = vecNor[ii][2] - vecNor[jj][2];
        rnl  = fabs(rx12) + fabs(ry12) + fabs(rz12);
        if ( rnl < epsil2 )
          goto NexVec2;
          //goto 5399
        
        rx23 = vecNor[jj][0] - vecNor[kk][0];
        ry23 = vecNor[jj][1] - vecNor[kk][1];
        rz23 = vecNor[jj][2] - vecNor[kk][2];
        rnl  = fabs(rx23) + fabs(ry23) + fabs(rz23);
        if ( rnl < epsil2 )
          goto NexVec2;
          //goto 5399
        
        rx31 = vecNor[kk][0] - vecNor[ii][0];
        ry31 = vecNor[kk][1] - vecNor[ii][1];
        rz31 = vecNor[kk][2] - vecNor[ii][2];
        rnl  = fabs(rx31) + fabs(ry31) + fabs(rz31);
        if ( rnl < epsil2 )
          goto NexVec2;
          //goto 5399

        //---Three rotating denom by three cases
        
        //- first number permutation
        denom = rx23*ry12 - rx12*ry23;
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rxc   = -(rz23*ry12 - rz12*ry23)*denom;
          ryc   =  (rz23*rx12 - rz12*rx23)*denom;  
          rnl   = sqrt(1. + rxc*rxc + ryc*ryc);
          rzc   = 1. / rnl;
          rxc   = rxc*rzc;
          ryc   = ryc*rzc;
          goto ValidCircumCircle;
        }
      
        denom = rx23*rz12 - rx12*rz23;
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rxc   = -(ry23*rz12 - ry12*rz23)*denom;
          rzc   =  (ry23*rx12 - ry12*rx23)*denom;
          rnl   = sqrt(1. + rxc*rxc + rzc*rzc);
          ryc   = 1. / rnl;
          rxc   = rxc*ryc;
          rzc   = rzc*ryc;
          goto ValidCircumCircle;
        }

        denom = rz23*ry12 - rz12*ry23;
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rzc   = -(rx23*ry12 - rx12*ry23)*denom;
          ryc   =  (rx23*rz12 - rx12*rz23)*denom;
          rnl   = sqrt(1. + rzc*rzc + ryc*ryc);
          rxc   = 1. / rnl;
          rzc   = rzc*rxc;
          ryc   = ryc*rxc;
          goto ValidCircumCircle;
        }

        //--- second number permutation
        denom = rx31*ry23 - rx23*ry31;
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rxc   = -(rz31*ry23 - rz23*ry31)*denom;
          ryc   =  (rz31*rx23 - rz23*rx31)*denom;
          rnl   = sqrt(1. + rxc*rxc + ryc*ryc);
          rzc   = 1. / rnl;
          rxc   = rxc*rzc;
          ryc   = ryc*rzc;
          goto ValidCircumCircle;
        }
      
        denom = rx31*rz23 - rx23*rz31;
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rxc   = -(ry31*rz23 - ry23*rz31)*denom;
          rzc   =  (ry31*rx23 - ry23*rx31)*denom;
          rnl   = sqrt(1. + rxc*rxc + rzc*rzc);
          ryc   = 1. / rnl;
          rxc   = rxc*ryc;
          rzc   = rzc*ryc;
          goto ValidCircumCircle;
        }
      
        denom = rz31*ry23 - rz23*ry31;
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rzc   = -(rx31*ry23 - rx23*ry31)*denom;
          ryc   =  (rx31*rz23 - rx23*rz31)*denom;
          rnl   = sqrt(1. + rzc*rzc + ryc*ryc);
          rxc   = 1. / rnl;
          rzc   = rzc*rxc;
          ryc   = ryc*rxc;
          goto ValidCircumCircle;
        }

        //--- third number permutation
        denom = rx12*ry31 - rx31*ry12;               
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rxc   = -(rz12*ry31 - rz31*ry12)*denom;
          ryc   =  (rz12*rx31 - rz31*rx12)*denom;
          rnl   = sqrt(1. + rxc*rxc + ryc*ryc);
          rzc   = 1. / rnl;
          rxc   = rxc*rzc;
          ryc   = ryc*rzc;
          goto ValidCircumCircle;
        }

        denom = rx12*rz31 - rx31*rz12;               
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rxc   = -(ry12*rz31 - ry31*rz12)*denom;
          rzc   =  (ry12*rx31 - ry31*rx12)*denom;
          rnl   = sqrt(1. + rxc*rxc + rzc*rzc);
          ryc   = 1. / rnl;
          rxc   = rxc*ryc;
          rzc   = rzc*ryc;
          goto ValidCircumCircle;
        }

        denom = rz12*ry31 - rz31*ry12;              
        if ( fabs(denom) > epsil ) {
          denom = 1. / denom;
          rzc   = -(rx12*ry31 - rx31*ry12)*denom;
          ryc   =  (rx12*rz31 - rx31*rz12)*denom;
          rnl   = sqrt(1. + rzc*rzc + ryc*ryc);
          rxc   = 1. / rnl;
          rzc   = rzc*rxc;
          ryc   = ryc*rxc;
          goto ValidCircumCircle;
        }

        //--- at this point, the three normals must be aligned with a huge radius
        //--- go home (ie to the next vector)
        goto NexVec2;
        //goto 5399   


        ValidCircumCircle:  

        //--- compute the radius and orientation
        scal1 = rxc*vecNor[kk][0] + ryc*vecNor[kk][1] + rzc*vecNor[kk][2];
        if ( scal1 < 0. ) {
          rxc   = -rxc;
          ryc   = -ryc;
          rzc   = -rzc;
          scal1 = -scal1;
        }

        scal  = scal1;

        //--- check that this radius is smaller that the current smallest radius
        if ( scal < scalmin1 )
          goto NexVec2;
          //goto 5399

        //--- set the tolerance
        scal1 = scal - tolcircle;
     
        //--- compute the maximal distance
        for (ll=0; ll<nbrNor; ++ll) {
          if ( ll != ii && ll != jj && ll != kk ) {
            scalt = rxc*vecNor[ll][0] + ryc*vecNor[ll][1] + rzc*vecNor[ll][2];
            if ( scalt < scal1 )
              goto NexVec2;
              //goto 5399
          }
        }
     
        //--- have found a circumcercle, store it           
        scalmin  = scal;
        scalmin1 = scal + tolcircle;
        rnrx     = rxc;
        rnry     = ryc;
        rnrz     = rzc;

        NexVec2:
        continue;
        //5399  continue

      }  // end for kk 
    }    // end for jj
  }      // end for ii


  //--- verification
  Verification:
  //8000 continue     

  
  if (mmsg == 2) {
    snprintf (Text, sizeof(Text),"  number of normal %d  for vertex %d", (int) nbrNor, (int) iVer);
    ug_message (Text);
    snprintf (Text, sizeof(Text),"  most visible normal %le %le %le", rnrx, rnry, rnrz); 
    ug_message (Text);
  }
  phimax = 0.;
  for (ii=0; ii<nbrNor; ++ii) {
    csca = rnrx*vecNor[ii][0] + rnry*vecNor[ii][1] + rnrz*vecNor[ii][2];
    if (mmsg == 2) {
      snprintf (Text, sizeof(Text),"  -- POINT %d VISIBILITY %le   PT-NOR %le %le %le   FA-NOR %le %le %le\n", (int) iVer, csca, 
        rnrx, rnry, rnrz, vecNor[ii][0], vecNor[ii][1], vecNor[ii][2]);
      ug_message (Text);
    }
    
    if ( csca < 0. ) { 
      snprintf (Text, sizeof(Text),"  ## WARNING MOST VISIBLE: POINT %d NEG. VISIBILITY %le   PT-NOR %le %le %le   FA-NOR %le %le %le", (int) iVer, csca, 
        rnrx, rnry, rnrz, vecNor[ii][0], vecNor[ii][1], vecNor[ii][2]);
      ug_message (Text);
    }
   
    phi = acos( MAX(-1., MIN(1., csca)) );
    if ( phi > phimax )
      phimax = phi;
    rangle[ii]  = phi;
    cangle     += phi;
  }
  
  if (mmsg == 2) {
    snprintf (Text, sizeof(Text),"  Most normal %le %le %le   Maximal angle %le   ", rnrx, rnry, rnrz, phimax);
    ug_message (Text);
  }
  
  if ( (phimax-phimax1) > tol2 ) {
    if ( iconv == 0 ) {
      ug_free (rangle);
      ug_free (rfactor);
      ug_free (rfactor2);
      snprintf (Text, sizeof(Text),"*** ERROR 361 : most visible: comparison ***");
      ug_error_message (Text);
      snprintf (Text, sizeof(Text),"*** most visible normal failed, vertex %d ***", (int) iVer);
      ug_error_message (Text);
      snprintf (Text, sizeof(Text),"*** phi %le %le ***", phimax, phimax1);
      ug_error_message (Text);
      mosNor[0] = 0.;
      mosNor[1] = 0.;
      mosNor[2] = 0.;
      return (361);
    }
  }
  
  if ( fabs(rnrx) + fabs(rnry) + fabs(rnrz) < 1.e-30 ) {
    ug_free (rangle);
    ug_free (rfactor);
    ug_free (rfactor2);
    snprintf (Text, sizeof(Text),"*** ERROR 362 : most visible: failed ***");
    ug_error_message (Text);
    return (362);
  }

  //--- set normal to iVer
  mosNor[0] = rnrx;
  mosNor[1] = rnry;
  mosNor[2] = rnrz;

  // free temporary space

  ug_free (rangle);
  ug_free (rfactor);
  ug_free (rfactor2);
 
  return (0);
}
