#include "UG3_LIB.h"

INT_ ug3_bv3 (
  INT_ nbface,
  INT_ nnode,
  INT_1D * ibcibf,
  INT_3D * inibf,
  DOUBLE_3D * bv,
  DOUBLE_3D * x)
{

/*
 * Determine a normal vector for each boundary node and then project boundary
 * normal vector on curves between non BL intersecting surfaces and adjacent BL
 * intersecting surfaces onto the adjacent BL intersecting surface.
 * 
 * UG3 LIB : Unstructured Grid - General Purpose Routine Library
 * 3D Version : $Id: ug3_bv3.c,v 1.4 2021/02/07 01:45:08 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  INT_1D *mbvi = NULL;

  DOUBLE_3D *bvi = NULL;

  INT_ found, ibface, inode, inode1, inode2, inode3;
  INT_ ierr = 0;

  double bfv1, bfv2, bfv3, bv1, bv2, bv3, bvi1, bvi2, bvi3,
         dx211, dx212, dx213, dx311, dx312, dx313,
         w, x11, x12, x13, x21, x22, x23, x31, x32, x33;

  // determine standard normal vector for each boundary node

  ug3_bv2 (nbface, nnode, inibf, bv, x);

  // check if there are any BL intersecting surfaces

  ibface = 1;

  do {

    found = CHK_BL_INT_UG3_GBC (ibcibf[ibface]);

    ibface++;
  }
  while (ibface <= nbface && found == 0);

  // return if there are no BL intersecting surfaces

  if (found == 0)
    return 0;

  // allocated temporary space

  mbvi = (INT_1D *) ug_malloc (&ierr, (nnode+1) * sizeof (INT_1D));
  bvi = (DOUBLE_3D *) ug_malloc (&ierr, (nnode+1) * sizeof (DOUBLE_3D));

  if (ierr) {
    ug_free (mbvi);
    ug_free (bvi);
    ug_error_message ("*** ERROR 100353 : unable to allocate required memory ***");
    return 100353;
  }

  // set flag to -1 at all nodes on BL intersecting surfaces

  memset (mbvi, 0, (nnode+1) * sizeof (INT_1D));

  for (ibface = 1; ibface <= nbface; ibface++) {

    if (CHK_BL_INT_UG3_GBC (ibcibf[ibface])) {

      inode1 = inibf[ibface][0];
      inode2 = inibf[ibface][1];
      inode3 = inibf[ibface][2];
  
      mbvi[inode1] = -1;
      mbvi[inode2] = -1;
      mbvi[inode3] = -1;
    }
  }

  // set flag to 1 at all nodes on curves between non BL intersecting
  // surfaces and adjacent BL intersecting surfaces

  for (ibface = 1; ibface <= nbface; ibface++) {

    if (CHK_BL_INT_UG3_GBC (ibcibf[ibface]) == 0) {

      inode1 = inibf[ibface][0];
      inode2 = inibf[ibface][1];
      inode3 = inibf[ibface][2];
  
      if (mbvi[inode1] == -1) mbvi[inode1] = 1;
      if (mbvi[inode2] == -1) mbvi[inode2] = 1;
      if (mbvi[inode3] == -1) mbvi[inode3] = 1;
    }
  }

  // reset flag to 0 if it is -1
  // the resulting flag is 1 if the nodes is on an intersecting curves and
  // 0 otherwise

  for (inode = 1; inode <= nnode; inode++) {

    if (mbvi[inode] == -1) mbvi[inode] = 0;
  }

  // initialize BL intersecting surface normal

  memset (bvi, 0, (nnode+1) * sizeof (DOUBLE_3D));

  for (inode = 1; inode <= nnode; inode++) {

    bvi[inode][0] = 0;
    bvi[inode][1] = 0;
    bvi[inode][2] = 0;
  }

  // determine boundary surface normal on BL intersecting surfaces
  // only those on intersecting curves are needed

  for (ibface = 1; ibface <= nbface; ibface++) {

    // if the face is on a BL intersecting surface

    if (CHK_BL_INT_UG3_GBC (ibcibf[ibface])) {

      inode1 = inibf[ibface][0];
      inode2 = inibf[ibface][1];
      inode3 = inibf[ibface][2];
   
      // if the face contains is a node on an intersecting curve

      if (mbvi[inode1] || mbvi[inode2] || mbvi[inode3]) {

        x11 = x[inode1][0];
        x12 = x[inode1][1];
        x13 = x[inode1][2];
        x21 = x[inode2][0];
        x22 = x[inode2][1];
        x23 = x[inode2][2];
        x31 = x[inode3][0];
        x32 = x[inode3][1];
        x33 = x[inode3][2];

        dx211 = x21 - x11;
        dx212 = x22 - x12;
        dx213 = x23 - x13;
        dx311 = x31 - x11;
        dx312 = x32 - x12;
        dx313 = x33 - x13;

        // determine face normal

        bfv1 = dx212 * dx313 - dx213 * dx312;
        bfv2 = dx213 * dx311 - dx211 * dx313;
        bfv3 = dx211 * dx312 - dx212 * dx311;

        w = 1.0 / sqrt (bfv1 * bfv1 + bfv2 * bfv2 + bfv3 * bfv3);

        bfv1 = w * bfv1;
        bfv2 = w * bfv2;
        bfv3 = w * bfv3;

        // sum face normals on intersecting curve nodes

        if (mbvi[inode1]) {
          bvi[inode1][0] = bvi[inode1][0] + bfv1;
          bvi[inode1][1] = bvi[inode1][1] + bfv2;
          bvi[inode1][2] = bvi[inode1][2] + bfv3;
        }

        if (mbvi[inode2]) {
          bvi[inode2][0] = bvi[inode2][0] + bfv1;
          bvi[inode2][1] = bvi[inode2][1] + bfv2;
          bvi[inode2][2] = bvi[inode2][2] + bfv3;
        }

        if (mbvi[inode3]) {
          bvi[inode3][0] = bvi[inode3][0] + bfv1;
          bvi[inode3][1] = bvi[inode3][1] + bfv2;
          bvi[inode3][2] = bvi[inode3][2] + bfv3;
        }
      }
    }
  }

  // correct boundary surface normal on intersecting curves

  for (inode = 1; inode <= nnode; inode++) {

    // only consider nodes on intersecting curves

    if (mbvi[inode]) {

      bv1 = bv[inode][0];
      bv2 = bv[inode][1];
      bv3 = bv[inode][2];

      bvi1 = bvi[inode][0];
      bvi2 = bvi[inode][1];
      bvi3 = bvi[inode][2];

      w = 1.0 / sqrt (bvi1 * bvi1 + bvi2 * bvi2 + bvi3 * bvi3);

      // normalize BL intersecting surface normal

      bvi1 = w * bvi1;
      bvi2 = w * bvi2;
      bvi3 = w * bvi3;

      // subtract component of standard boundary surface normal that is normal
      // to the BL intersecting surface

      w = bvi1 * bv1 + bvi2 * bv2 + bvi3 * bv3;

      bv1 = bv1 - bvi1 * w;
      bv2 = bv2 - bvi2 * w;
      bv3 = bv3 - bvi3 * w;

      // normalize the boundary surface normal

      w = 1.0 / sqrt (bv1 * bv1 + bv2 * bv2 + bv3 * bv3);

      bv[inode][0] = w * bv1;
      bv[inode][1] = w * bv2;
      bv[inode][2] = w * bv3;
    }
  }

  // free temporary space

  ug_free (mbvi);
  ug_free (bvi);

  return 0;

}
