#include "UG2_LIB.h"

INT_ ug2_write_grid_file
 (char File_Name[],
  INT_ Message_Flag,
  INT_ Number_of_Bnd_Edges,
  INT_ Number_of_Nodes,
  INT_ Number_of_Quads,
  INT_ Number_of_Trias,
  INT_2D * Bnd_Edge_Connectivity,
  INT_1D * Bnd_Edge_Grid_BC_Flag,
  INT_1D * Bnd_Edge_ID_Flag,
  INT_4D * Quad_Connectivity,
  INT_3D * Tria_Connectivity,
  DOUBLE_2D * Coordinates,
  DOUBLE_1D * Initial_Normal_Spacing,
  DOUBLE_1D * BL_Thickness)

{

/*
 * Write 2D grid data to a standard UG 2D grid file.
 * 
 * UG2 LIB : Unstructured Grid - General Purpose Routine Library
 * 2D Version : $Id: ug2_write_grid_file.c,v 1.8 2024/04/22 06:31:40 marcum Exp $
 * Copyright 1994-2020, David L. Marcum
 */

  FILE *Grid_File;

  CHAR_UG_MAX drive, dname, fname, ext, File_Type, Text;
  CHAR_51 Name_Text;

  INT_1D *Face_ID_Flag = NULL;

  INT_ File_Mode, File_Format;
  INT_ ierr = 0;

  // get file info

  ierr = ug_file_format_check (File_Name);

  if (ierr)
    return (ierr);

  ug_file_info (File_Name, File_Type, &File_Format, &File_Mode);

  ug_splitpath (File_Name, drive, dname, fname, ext);

  strcat (fname, ext);
  strcpy (Name_Text, "");
  strncat (Name_Text, fname, 50);

  if (ug_file_status_monitor_flag())
  {
    snprintf (Text, sizeof(Text), "ug2_write_grid_file: writing file with File_Name=%s File_Type=%s File_Format=%i", Name_Text, File_Type, (int) File_Format);
    ug_message (Text);
  }

  // output grid dimensions

  if (Message_Flag >= 1 && File_Mode == UG_FIO_STANDARD_FILE_MODE)
  {
    ug_message (" ");
    ug_message ("UG2      : OUTPUT GRID");
    ug_message (" ");
    snprintf (Text, sizeof(Text), "UG2      : Grid File Name    = %s", Name_Text);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG2      : Boundary Edges    =%10i", (int) Number_of_Bnd_Edges);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG2      : Nodes             =%10i", (int) Number_of_Nodes);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG2      : Quad Faces        =%10i", (int) Number_of_Quads);
    ug_message (Text);
    snprintf (Text, sizeof(Text), "UG2      : Tria Faces        =%10i", (int) Number_of_Trias);
    ug_message (Text);
    ug_message (" ");
  }

  // write grid file

  if (strcmp (File_Type, ".ugrid") == 0)
  {
    // open file

    if (File_Mode == UG_FIO_TMP_FILE_MODE)
    {
      strcpy (fname, File_Name);
      strcat (fname, "+");

      if (ug_file_status_monitor_flag())
        ug_message ("ug2_write_grid_file: opening TMP file");

      Grid_File = ug_fopen (File_Name, "tmp");
    }
    else
    {
      ug_backup_file (File_Name);

      if (ug_file_status_monitor_flag())
        ug_message ("ug2_write_grid_file: opening regular file");

      Grid_File = ug_fopen (File_Name, "w");
    }

    if (Grid_File == NULL)
    {
      ug_error_message ("*** ERROR 236 : unable to open output grid file ***");
      return (236);
    }

    ierr = ug2_write_ugrid (Grid_File, File_Format,
                            Number_of_Nodes, Number_of_Quads, Number_of_Trias,
                            Quad_Connectivity, Tria_Connectivity,
                            Coordinates);

    if (ierr == 0)
      ierr = ug2_write_gdata (Grid_File, File_Format,
                              Number_of_Bnd_Edges, Number_of_Nodes,
                              Number_of_Quads, Number_of_Trias,
                              Bnd_Edge_Connectivity,
                              Bnd_Edge_Grid_BC_Flag, Bnd_Edge_ID_Flag,
                              Initial_Normal_Spacing, BL_Thickness);

    ug_fclose (Grid_File);
  }

  else if (strcmp (File_Type, ".mesh") == 0 ||
           strcmp (File_Type, ".meshb") == 0) {

    ug_backup_file (File_Name);

    ierr = ug2_write_mesh  (File_Name, 
                            Number_of_Nodes, Number_of_Bnd_Edges,
                            Number_of_Quads, Number_of_Trias,
                            Bnd_Edge_ID_Flag, Bnd_Edge_Connectivity,
                            Face_ID_Flag, Quad_Connectivity, Tria_Connectivity,
                            Coordinates);
  }

  else
  { 
    ug_error_message ("*** ERROR 236 : unknown output grid file type ***");
    return (236);
  }

  // exit on error

  if (ierr)
    return (236);

  return (0);

}
