#include "UG2_LIB.h"

INT_ ug2_read_grid_file
 (char File_Name[],
  INT_ Message_Flag,
  INT_ M_BL_Thickness,
  INT_ M_Initial_Normal_Spacing,
  INT_ M_Bnd_Edge_Grid_BC_Flag,
  INT_ M_Bnd_Edge_ID_Flag,
  INT_ Bnd_Only_Flag,
  INT_ No_Quad_Flag,
  INT_ *Number_of_Bnd_Edges,
  INT_ *Number_of_Nodes,
  INT_ *Number_of_Quads,
  INT_ *Number_of_Trias,
  INT_2D ** Bnd_Edge_Connectivity,
  INT_1D ** Bnd_Edge_Grid_BC_Flag,
  INT_1D ** Bnd_Edge_ID_Flag,
  INT_4D ** Quad_Connectivity,
  INT_3D ** Tria_Connectivity,
  DOUBLE_2D ** Coordinates,
  DOUBLE_1D ** Initial_Normal_Spacing,
  DOUBLE_1D ** BL_Thickness)

{

/*
 * Read 2D grid data from a standard UG 2D grid file.
 * 
 * UG2 LIB : Unstructured Grid - General Purpose Routine Library
 * 2D Version : $Id: ug2_read_grid_file.c,v 1.6 2022/11/21 00:11:08 marcum Exp $
 * Copyright 1994-2020, David L. Marcum
 */

  FILE *Grid_File;

  CHAR_UG_MAX drive, dname, fname, ext, File_Type, Text;
  CHAR_51 Name_Text;

  INT_1D *Face_ID_Flag = NULL;
  INT_1D *Map = NULL;

  INT_ File_Mode, File_Format, Number_of_Bnd_Nodes, N_Bnd_Edge_Grid_BC_Flag,
       N_Bnd_Edge_ID_Flag, N_BL_Thickness, N_Initial_Normal_Spacing,
       Read_Task_Flag;
  INT_ ierr = 0;

  double dc0;

  dc0 = 0.0;

  ierr = ug_file_format_check (File_Name);

  if (ierr)
    return (ierr);

  ug_file_info (File_Name, File_Type, &File_Format, &File_Mode);

  ug_splitpath (File_Name, drive, dname, fname, ext);

  strcat (fname, ext);
  strcpy (Name_Text, "");
  strncat (Name_Text, fname, 50);

  // output header

  if (Message_Flag >= 1 && File_Mode == UG_FIO_STANDARD_FILE_MODE)
  {
    ug_message (" ");
    ug_message ("UG2      : INPUT GRID");
    ug_message (" ");
    ug_message ("UG2      : Reading Data");
    ug_message (" ");
  }

  // initialize grid dimensions

  *Number_of_Nodes = 0;
  *Number_of_Quads = 0;
  *Number_of_Trias = 0;

  if (Bnd_Only_Flag)
    Number_of_Bnd_Nodes = -1;
  else
    Number_of_Bnd_Nodes = 0;

  // open file

  if (ug_file_status_monitor_flag())
  {
    snprintf (Text, sizeof(Text), "ug2_read_grid_file: reading file with File_Name=%s File_Type=%s File_Format=%i", Name_Text, File_Type, (int) File_Format);
    ug_message (Text);
  }

  if (strcmp (File_Type, ".ugrid") == 0)
  {
    if (File_Mode == UG_FIO_TMP_FILE_MODE)
    {
      if (ug_file_status_monitor_flag())
        ug_message ("ug2_read_grid_file: opening TMP file");

      Grid_File = ug_fopen (File_Name, "r_tmp");
    }
    else
    {
      if (ug_file_status_monitor_flag())
        ug_message ("ug2_read_grid_file: opening regular file");

      Grid_File = ug_fopen (File_Name, "r");
    }

    if (Grid_File == NULL)
    {
      ug_error_message ("*** ERROR 237 : error opening UG2 grid file ***");
      return (237);
    }
  }

  // make two passes
  // the first pass reads the dimensions and the second reads the data

  Read_Task_Flag = 1;

  do
  {
    // read grid file

    if (strcmp (File_Type, ".ugrid") == 0)
    {
      ierr = ug2_read_ugrid (Grid_File, 
                             File_Format, Read_Task_Flag,
                             Number_of_Bnd_Nodes, Number_of_Nodes, 
                             Number_of_Quads, Number_of_Trias,
                             *Quad_Connectivity, *Tria_Connectivity,
                             *Coordinates);

      if (ierr == 0)
        ierr = ug2_read_gdata (Grid_File, 
                               File_Format, Read_Task_Flag,
                               Number_of_Bnd_Edges,
                               &Number_of_Bnd_Nodes, Number_of_Nodes,
                               Number_of_Quads, Number_of_Trias,
                               *Bnd_Edge_Connectivity,
                               *Bnd_Edge_Grid_BC_Flag, *Bnd_Edge_ID_Flag,
                               *Initial_Normal_Spacing, *BL_Thickness);

      // rewind file for next pass
      // or close file on last pass

      if (Read_Task_Flag == 1 || File_Mode == UG_FIO_TMP_FILE_MODE)
        ug_rewind (Grid_File);
      else
        ug_fclose (Grid_File);
    }

    else if (strcmp (File_Type, ".mesh") == 0 ||
             strcmp (File_Type, ".meshb") == 0)
      ierr = ug2_read_mesh (File_Name, 
                            Read_Task_Flag,
                            &Number_of_Bnd_Nodes,
                            Number_of_Nodes,
                            Number_of_Bnd_Edges,
                            Number_of_Quads,
                            Number_of_Trias,
                            *Bnd_Edge_ID_Flag,
                            *Bnd_Edge_Connectivity,
                            Face_ID_Flag,
                            *Quad_Connectivity,
                            *Tria_Connectivity,
                            *Coordinates);

    else
    {
      Read_Task_Flag = 0;
      ug_error_message ("*** ERROR 237 : unknown input grid file type ***");
      ierr = 237;
    }

    if (ierr)
    {
      if (Read_Task_Flag == 1 && strcmp (File_Type, ".ugrid") == 0) ug_fclose (Grid_File);
      return (237);
    }

    // zero volume grid dimensions if option to read boundary only is set

    if (Bnd_Only_Flag)
    {
      *Number_of_Quads = 0;
      *Number_of_Trias = 0;
    }

    // output grid information

    if (Read_Task_Flag == 1)
    {
      if (Message_Flag >= 1 && File_Mode == UG_FIO_STANDARD_FILE_MODE)
      {
        snprintf (Text, sizeof(Text), "UG2      : Grid File Name    = %s", Name_Text);
        ug_message (Text);
        snprintf (Text, sizeof(Text), "UG2      : Boundary Edges    =%10i", (int) *Number_of_Bnd_Edges);
        ug_message (Text);
        snprintf (Text, sizeof(Text), "UG2      : Nodes             =%10i", (int) *Number_of_Nodes);
        ug_message (Text);
        snprintf (Text, sizeof(Text), "UG2      : Quad Elements     =%10i", (int) *Number_of_Quads);
        ug_message (Text);
        snprintf (Text, sizeof(Text), "UG2      : Tria Elements     =%10i", (int) *Number_of_Trias);
        ug_message (Text);
      }

      // allocate grid data

      ierr = 0;

      N_Bnd_Edge_Grid_BC_Flag = (M_Bnd_Edge_Grid_BC_Flag == 1) ? 1 : 0;
      N_Bnd_Edge_ID_Flag = (M_Bnd_Edge_ID_Flag == 1) ? 1 : 0;
      N_Initial_Normal_Spacing = (M_Initial_Normal_Spacing == 1) ? 1 : 0;
      N_BL_Thickness = (M_BL_Thickness == 1) ? 1 : 0;

      *Bnd_Edge_Connectivity = (INT_2D *) ug_malloc (&ierr,
                                                     ((*Number_of_Bnd_Edges)+1)
                                                   * sizeof (INT_2D));
      *Bnd_Edge_Grid_BC_Flag = (INT_1D *) ug_malloc (&ierr,
                                                     ((*Number_of_Bnd_Edges)+1)
                                                   * N_Bnd_Edge_Grid_BC_Flag
                                                   * sizeof (INT_1D));
      *Bnd_Edge_ID_Flag = (INT_1D *) ug_malloc (&ierr,
                                                ((*Number_of_Bnd_Edges)+1)
                                               * N_Bnd_Edge_ID_Flag
                                               * sizeof (INT_1D));
      *Quad_Connectivity = (INT_4D *) ug_malloc (&ierr,
                                                 ((*Number_of_Quads)+1)
                                               * sizeof (INT_4D));
      *Tria_Connectivity = (INT_3D *) ug_malloc (&ierr,
                                                 ((*Number_of_Trias)+1)
                                               * sizeof (INT_3D));
      *Coordinates = (DOUBLE_2D *) ug_malloc (&ierr,
                                              ((*Number_of_Nodes)+1)
                                            * sizeof (DOUBLE_2D));
      *Initial_Normal_Spacing = (DOUBLE_1D *) ug_malloc (&ierr,
                                                         ((*Number_of_Nodes)+1)
                                                       * N_Initial_Normal_Spacing
                                                       * sizeof (DOUBLE_1D));
      *BL_Thickness = (DOUBLE_1D *) ug_malloc (&ierr,
                                               ((*Number_of_Nodes)+1)
                                             * N_BL_Thickness
                                             * sizeof (DOUBLE_1D));

      if (ierr)
      {
        if (strcmp (File_Type, ".ugrid") == 0) ug_fclose (Grid_File);
        ug_error_message ("*** ERROR 100213 : unable to malloc grid data arrays ***");
        return (100213);
      }
    }

    // set initial values for boundary data

    if (Read_Task_Flag == 1)
    {
      if (M_BL_Thickness)
        ug_set_double (1, *Number_of_Nodes, dc0, *BL_Thickness);

      if (M_Initial_Normal_Spacing)
        ug_set_double (1, *Number_of_Nodes, dc0, *Initial_Normal_Spacing);

      if (M_Bnd_Edge_Grid_BC_Flag)
        ug_set_int (1, *Number_of_Bnd_Edges, -1, *Bnd_Edge_Grid_BC_Flag);
    }

    ++Read_Task_Flag;
  }
  while (Read_Task_Flag <= 2);

  // remove interior nodes from grid data if option to read boundary only is set

  if (Bnd_Only_Flag)
  {
    ierr = ug2_bnd_nodes (1, &Number_of_Bnd_Nodes, *Number_of_Nodes,
                                *Number_of_Bnd_Edges,
                                *Bnd_Edge_Connectivity, &Map,
                                *BL_Thickness, *Initial_Normal_Spacing,
                                *Coordinates);

    ug_free (Map);

    if (ierr)
      return (ierr);

    *Number_of_Nodes = Number_of_Bnd_Nodes;

    if (Message_Flag >= 1 && File_Mode == UG_FIO_STANDARD_FILE_MODE)
    {
      snprintf (Text, sizeof(Text), "UG2      : Nodes             =%10i", (int) *Number_of_Nodes);
      ug_message (Text);
    }
  }

  // Convert surface grid quads to trias.

  if (No_Quad_Flag && *Number_of_Quads)
  {
    ierr = ug2_qtria (Number_of_Trias, Number_of_Quads,
                            Tria_Connectivity, *Quad_Connectivity,
                            *Coordinates);

    if (ierr)
      return (ierr);

    if (Message_Flag >= 1 && File_Mode == UG_FIO_STANDARD_FILE_MODE)
    {
      snprintf (Text, sizeof(Text), "UG2      : Quad Elements     =%10i", (int) *Number_of_Quads);
      ug_message (Text);
      snprintf (Text, sizeof(Text), "UG2      : Tria Elements     =%10i", (int) *Number_of_Trias);
      ug_message (Text);
    }
  }

  // output tailer

  if (Message_Flag >= 1 && File_Mode == UG_FIO_STANDARD_FILE_MODE)
    ug_message (" ");

  return (0);

}
