#include "UG_LIB.h"

INT_ ug_rerun (INT_ mmsg, const int argc, char *const argv[])
{

/*
 * Rerun with a new executable and the same options.
 * 
 * UG LIB : Unstructured Grid - General Purpose Routine Library
 * $Id: ug_rerun.c,v 1.35 2022/11/21 01:10:49 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

  char **argv2;
  char *path_list;

  CHAR_UG_MAX *items = NULL;

  CHAR_UG_MAX base_dir, def_exe_ext, exe_dir, exe_drive, exe_name, exe_ext,
              name, new_ver_dir, path, text;

  int argc2, i, j;
  int n = 0;
  INT_ err, N;

  // default

  strcpy (new_ver_dir, "");
  strcpy (def_exe_ext, ".exe");

  // loop over input argument vectors

  j = 0;

  for (i = 1; i < argc; ++i)
  {
    // check for system version directory path option

    if (strcmp (argv[i], "v") == 0 || strcmp (argv[i], "-v") == 0)
    {
      ++i;

      if (i < argc)
        strcpy (new_ver_dir, argv[i]);
      else
      {
        ug_error_message ("*** ERROR 416 : command line argument v or -v require a version directory path name ***");
        return (416);
      }
    }

    // count all arguments not related to rerun with a new executable

    else
      ++j;
  }

  // set number of entries in new argument vector

  argc2 = j+1;

  // if system directory path name was not set then return without error

  if (strcmp (new_ver_dir, "") == 0)
    return (0);

  // allocate a new argument vector
  // will be of the same length or less than the input argument vector

  err = 0;

  argv2 = (char **) ug_malloc (&err, (argc2+1) * sizeof (*argv));

  if (err)
  {
    ug_error_message ("*** ERROR 100411 : unable to allocate space for new argument vector ***");
    return (100411);
  }

  // loop over input argument vectors

  j = 0;

  for (i = 1; i < argc; ++i)
  {
    // skip system version directory path option

    if (strcmp (argv[i], "v") == 0 || strcmp (argv[i], "-v") == 0)
      ++i;

    // save all arguments not related to rerun with a new executable

    else
    {
      ++j;

      argv2[j] = argv[i];
    }
  }

  // split the path name for the current executable into separate strings for
  // the drive (Windows only), directory, executable name, and extension

  strcpy (path, argv[0]);

  ug_splitpath (path, exe_drive, exe_dir, exe_name, exe_ext);

  // if directory path name is empty then set it to the full path name of the
  // current directory

  if (strcmp (exe_dir, "") == 0)
  {
    strcpy (path, ug_get_cwd ());

    ug_splitpath (path, exe_drive, exe_dir, NULL, NULL);
  }

  // build derived path

  strcpy (path, exe_drive);
  strcat (path, exe_dir);
  strcat (path, UG_PATH_SEP);
  strcat (path, exe_name);
  strcat (path, exe_ext);

  // if the running executable file does not exist at the derived path then
  // check if it exists in a directory included in the PATH environment
  // variable

  if (ug_check_file_isreg (path))
  {
    strcpy (name, exe_name);
    strcat (name, exe_ext);

    path_list = ug_getenv ("PATH");

    err = ug_find_file_in_path_list (name, path_list, path);

    if (err && strcmp (exe_ext, "") == 0)
    {
      strcat (name, def_exe_ext);

      err = ug_find_file_in_path_list (name, path_list, path);
    }

    if (err)
    {
      ug_free (argv2);
      ug_error_message ("*** ERROR 417 : unable to find current executable ***");
      return (417);
    }

    ug_splitpath (path, exe_drive, exe_dir, exe_name, exe_ext);
  }

  // if extension is empty then also try an extension in the later checks

  if (strcmp (exe_ext, "") == 0)
    strcpy (exe_ext, def_exe_ext);

  // initialize text string for error message

  strcpy (text, "");

  // check if new version directory is correct

  strcpy (path, new_ver_dir);

  strcat (text, "         : ");
  strcat (text, path);
  strcat (text, "\n");

  strcat (path, UG_PATH_SEP);
  strcat (path, exe_name);

  if (ug_check_file_isreg (path))
    strcat (path, exe_ext);

  // check if new version directory + bin is correct

  if (ug_check_file_isreg (path))
  {
    strcpy (path, new_ver_dir);
    strcat (path, UG_PATH_SEP);
    strcat (path, "bin");

    strcat (text, "         : ");
    strcat (text, path);
    strcat (text, "\n");

    strcat (path, UG_PATH_SEP);
    strcat (path, exe_name);

    if (ug_check_file_isreg (path))
      strcat (path, exe_ext);
  }

  // check if current base + new version directory is correct

  if (ug_check_file_isreg (path))
  {
    // get a list of items from the executable directory path that represent
    // each subdirectory name

    err = ug_get_items_from_list (exe_dir, UG_PATH_SEP, &items, &N);

    n = (int) N;

    if (err)
    {
      ug_free (argv2);
      ug_free (items);
      if (err < 0)
      {
        ug_error_message ("*** ERROR 418 : current executable directory path does not follow standard naming ***");
        return (418);
      }
      else
      {
        snprintf (text, sizeof(text), "*** ERROR %i: error parsing current executable directory path ***", (int) err);
        ug_error_message (text);
        return (err);
      }
    }

    // continue if the current base directory is not the root

    if (n)
    {
      if (strcmp (items[n-1], "bin") == 0)
        j = 3;
      else
        j = 2;

      strcpy (base_dir, "");

      for (i = 0; i <= n-j; ++i)
      {
        strcat (base_dir, UG_PATH_SEP);
        strcat (base_dir, items[i]);
      }

      // free list of subdirectories

      ug_free (items);

      // check if current base + new version directory is correct

      strcpy (path, base_dir);
      strcat (path, UG_PATH_SEP);
      strcat (path, new_ver_dir);

      strcat (text, "         : ");
      strcat (text, path);
      strcat (text, "\n");

      strcat (path, UG_PATH_SEP);
      strcat (path, exe_name);

      if (ug_check_file_isreg (path))
        strcat (path, exe_ext);
    }
  }

  // check if current base + new version + bin directory is correct

  if (ug_check_file_isreg (path) && n)
  {
    strcpy (path, base_dir);
    strcat (path, UG_PATH_SEP);
    strcat (path, new_ver_dir);
    strcat (path, UG_PATH_SEP);
    strcat (path, "bin");

    strcat (text, "         : ");
    strcat (text, path);
    strcat (text, "\n");

    strcat (path, UG_PATH_SEP);
    strcat (path, exe_name);

    if (ug_check_file_isreg (path))
      strcat (path, exe_ext);
  }

  // check if HOME + new version directory is correct

  if (ug_check_file_isreg (path))
  {
    strcpy (path, ug_get_home_dir ());
    strcat (path, UG_PATH_SEP);
    strcat (path, new_ver_dir);

    strcat (text, "         : ");
    strcat (text, path);
    strcat (text, "\n");

    strcat (path, UG_PATH_SEP);
    strcat (path, exe_name);

    if (ug_check_file_isreg (path))
      strcat (path, exe_ext);
  }

  // check if HOME + new version + bin directory is correct

  if (ug_check_file_isreg (path))
  {
    strcpy (path, ug_get_home_dir ());
    strcat (path, UG_PATH_SEP);
    strcat (path, new_ver_dir);
    strcat (path, UG_PATH_SEP);
    strcat (path, "bin");

    strcat (text, "         : ");
    strcat (text, path);
    strcat (text, "\n");

    strcat (path, UG_PATH_SEP);
    strcat (path, exe_name);

    if (ug_check_file_isreg (path))
      strcat (path, exe_ext);
  }

  // return with error if new exectuable does not exist

  if (ug_check_file_isreg (path))
  {
    ug_free (argv2);
    ug_error_message ("UG RERUN : unable to find executable in any of the following directories");
    ug_error_message (text);
    ug_error_message ("*** ERROR 419 : unable to find new executbale ***");
    return (419);
  }

  // set first argument of new argument vector to the full path name of the new
  // executable

  argv2[0] = &path[0];

  // set last argument of new argument vector to NULL to signify the end

  argv2[argc2] = NULL;

  // output message

  if (mmsg)
  {
    strcpy (text, "UG RERUN :");

    for (j = 0; j < argc2; ++j)
    {
      strcat (text, " ");
      strcat (text, argv2[j]);
    }

    ug_message (text);
  }

  // rerun the executable with new argument vector

  ug_execv (path, argv2);

  // if no error occurrs then this code is never returned to
  // if an error occurred then free new argument vector and return with error

  ug_free (argv2);
  ug_error_message ("*** ERROR 420 : error attempting to run new executable ***");
  return (420);
}
