#include "UG_LIB.h"

/*
 * Parallel processing mode routines.
 *
 * mpmode = 0 = Serial processing mode.
 * mpmode = 1 = Parallel processing mode.
 * 
 * UG LIB : Unstructured Grid - General Purpose Routine Library
 * $Id: ug_mp.c,v 1.44 2022/11/21 01:10:49 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

// Local variables.

static CHAR_UG_MAX UG_MP_TMP_File_Dir_ = "";

static INT_1D *proc_status = NULL;

static INT_ isync_ = 0;
static INT_ iproc_ = 0;
static INT_ nproc_ = 0;
static INT_ mpmode_ = 0;

// Local headers.

static INT_ ug_mp_check_file (char Label[], INT_ iproc, INT_ isync, INT_ ntry);
static void ug_mp_remove_file (char Label[], INT_ iproc, INT_ isync);
static void ug_mp_remove_files (INT_ iproc);
static INT_ ug_mp_set_tmp_file_dir (INT_ jid);
static INT_ ug_mp_write_file (char Label[], INT_ iproc, INT_ isync);

INT_ ug_mp_setup (INT_ iproc, INT_ nproc, INT_ jid)
{
  // Setup parallel processing mode.

  INT_ Error_Flag = 0;

  iproc_ = 0;
  nproc_ = 0;

  mpmode_ = 0;

  if (iproc < 0 || iproc >= nproc)
  {
    ug_error_message ("*** ERROR 402 : invalid processor number ***");
    return (402);
  }

  if (nproc <= 1)
  {
    ug_error_message ("*** ERROR 403 : invalid number of processors ***");
    return (403);
  }

  Error_Flag = ug_mp_set_tmp_file_dir (jid);

  if (Error_Flag > 0)
    return (Error_Flag);

  iproc_ = iproc;
  nproc_ = nproc;

  mpmode_ = 1;

  proc_status = (INT_1D *) ug_malloc (&Error_Flag, nproc * sizeof (INT_1D));

  if (Error_Flag > 0)
  {
    ug_error_message ("*** ERROR 100403 : unable to allocate space for processor status list ***");
    return (100403);
  }

  for (iproc = 0; iproc < nproc_; ++iproc)
  {
    proc_status[iproc] = 0;
  }

  if (ug_mp_write_file ("active", iproc_, 1))
  {
    ug_error_message ("*** ERROR 404 : fatal error writing active-state file ***");
    return (404);
  }

  return (0);
}

INT_ ug_mp (char Task[])
{
  // Perform parallel processing task.

  INT_ iproc, iproc1, iproc2, nwait, pass;
  INT_ Error_Flag = 0;
  INT_ npass = 1000000;

  if (strcmp (Task, "iproc") == 0)
    return (iproc_);

  else if (strcmp (Task, "nproc") == 0)
    return (nproc_);

  else if (mpmode_ == 0)
    return (0);

  else if ((strcmp (Task, "wait") == 0 || strcmp (Task, "wait+") == 0))
  {
    ++isync_;

    if (ug_mp_write_file ("wait", iproc_, isync_))
    {
      ug_error_message ("*** ERROR 405 : fatal error writing wait-state file ***");
      return (405);
    }

    if (strcmp (Task, "wait") == 0) npass = 0;

    if (iproc_ == 0)
    {
      iproc1 = 1;
      iproc2 = nproc_-1;
    }
    else
    {
      iproc1 = 0;
      iproc2 = 0;
    }

    proc_status[iproc_] = 1;

    pass = 0;

    nwait = iproc2 - iproc1 + 1;

    while (nwait > 0)
    {
      ++pass;

      nwait = iproc2 - iproc1 + 1;

      pass = MIN (pass, npass+1);

      for (iproc = iproc1; iproc <= iproc2; ++iproc)
      {
        if (proc_status[iproc] == 0)
        {
          if (iproc == 0 && ug_mp_check_file ("active", iproc, 0, 1) == 0)
            return (-1);

          Error_Flag = ug_mp_check_file ("active", iproc, 1, 100);

          if (Error_Flag && pass > npass)
          {
            ug_error_message ("*** ERROR 426 : fatal error reading another process active-state file ***");
            return (426);
          }

          if (ug_mp_check_file ("wait", iproc, isync_, 1) == 0)
          {
            proc_status[iproc] = 1;

            --nwait;
          }
        }
        else
          --nwait;
      }
    }

    if (ug_mp_write_file ("done", iproc_, isync_))
    {
      ug_error_message ("*** ERROR 427 : fatal error writing done-state file ***");
      return (427);
    }

    proc_status[iproc_] = 0;

    nwait = iproc2 - iproc1 + 1;

    while (nwait > 0)
    {
      nwait = iproc2 - iproc1 + 1;

      for (iproc = iproc1; iproc <= iproc2; ++iproc)
      {
        if (proc_status[iproc] == 1)
        {
          if (iproc == 0 && ug_mp_check_file ("active", iproc, 0, 1) == 0)
            return (-1);

          Error_Flag = ug_mp_check_file ("active", iproc, 1, 1);

          if (Error_Flag)
          {
            ug_error_message ("*** ERROR 428 : fatal error reading another process active-state file ***");
            return (428);
          }

          if (ug_mp_check_file ("done", iproc, isync_, 1) == 0)
          {
            proc_status[iproc] = 0;

            --nwait;
          }
        }
        else
          --nwait;
      }
    }
  }

  else if (strcmp (Task, "end") == 0 && iproc_ == 0)
  {
    if (ug_mp_write_file ("active", iproc_, 0))
    {
      ug_error_message ("*** ERROR 429 : fatal error writing done-state file ***");
      return (429);
    }

    for (iproc = 1; iproc < nproc_; ++iproc)
    {
      proc_status[iproc] = 1;
    }

    nwait = nproc_-1;

    while (nwait > 0)
    {
      for (iproc = 1; iproc < nproc_; ++iproc)
      {
        if (proc_status[iproc] == 1 && ug_mp_check_file ("active", iproc, 1, 1))
        {
          proc_status[iproc] = 0;

          --nwait;
        }
      }
    }
  }

  else if (((iproc_ == 0 && strcmp (Task, "cleanup") == 0) || strcmp (Task, "cleanup_all") == 0))
  {
    for (iproc = 0; iproc <= nproc_-1; ++iproc)
    {
      ug_mp_remove_files (iproc);
    }

    ug_remove_dir_all (UG_MP_TMP_File_Dir_);

    strcpy (UG_MP_TMP_File_Dir_, "");

    ug_free (proc_status);

    proc_status = NULL;

    iproc_ = 0;
    nproc_ = 1;
    mpmode_ = 0;
  }

  else if (iproc_ && strcmp (Task, "cleanup") == 0)
    ug_mp_remove_files (iproc_);

  return (0);
}

void ug_mp_get_tmp_file_name (char File_Name[], char UG_MP_TMP_File_Name[])
{
  // Set binary UG_MP TMP file name including directory path.

  strcpy (UG_MP_TMP_File_Name, UG_MP_TMP_File_Dir_);

  if (strcmp (File_Name, ""))
  {
    strcat (UG_MP_TMP_File_Name, UG_PATH_SEP);
    strcat (UG_MP_TMP_File_Name, File_Name);
  }

  return;
}

static INT_ ug_mp_check_file (char Label[], INT_ iproc, INT_ isync, INT_ ntry)
{
  CHAR_UG_MAX File_Name, UG_MP_TMP_File_Name;

  INT_ m;

  snprintf (File_Name, sizeof(File_Name), "UG_MP_%s_%i_%i", Label, (int) iproc, (int) isync);

  ug_mp_get_tmp_file_name (File_Name, UG_MP_TMP_File_Name);

  m = (ntry <= 1) ? 0: ntry;

  if (ug_check_file (UG_MP_TMP_File_Name, m) != 0)
    return (1);

  return (0);
}

static void ug_mp_remove_files (INT_ iproc)
{
  INT_ isync;

  ug_mp_remove_file ("active", iproc, 0);
  ug_mp_remove_file ("active", iproc, 1);

  for (isync = 0; isync <= isync_; ++isync)
  {
    ug_mp_remove_file ("done", iproc, isync);
    ug_mp_remove_file ("wait", iproc, isync);
  }

  return;
}

static void ug_mp_remove_file (char Label[], INT_ iproc, INT_ isync)
{
  CHAR_UG_MAX File_Name, UG_MP_TMP_File_Name;

  snprintf (File_Name, sizeof(File_Name), "UG_MP_%s_%i_%i", Label, (int) iproc, (int) isync);

  ug_mp_get_tmp_file_name (File_Name, UG_MP_TMP_File_Name);

  ug_remove_file (UG_MP_TMP_File_Name);

  return;
}

static INT_ ug_mp_set_tmp_file_dir (INT_ jid)
{
  // Set UG_MP binary TMP file directory and create it.

  INT_ mode;
  INT_ Error_Flag = -1;

  snprintf (UG_MP_TMP_File_Dir_, sizeof (UG_MP_TMP_File_Dir_), ".UG_MP_%i", (int) jid);

  Error_Flag = ug_check_file (UG_MP_TMP_File_Dir_, 0);

  if (Error_Flag)
  {
    mode = UG_MODE_IRWXU;

    Error_Flag = ug_mkdir (UG_MP_TMP_File_Dir_, mode);
  }

  if (Error_Flag)
  {
    ug_error_message ("*** ERROR 430 : unable to find or create UG_MP TMP directory ***");
    return (430);
  }

  return (0);
}

static INT_ ug_mp_write_file (char Label[], INT_ iproc, INT_ isync)
{
  FILE *Data_File = NULL;

  CHAR_UG_MAX File_Name, UG_MP_TMP_File_Name;

  INT_ iwrite;

  snprintf (File_Name, sizeof(File_Name), "UG_MP_%s_%i_%i", Label, (int) iproc, (int) isync);

  ug_mp_get_tmp_file_name (File_Name, UG_MP_TMP_File_Name);

  Data_File = ug_fopen (UG_MP_TMP_File_Name, "w");

  if (Data_File == NULL)
  {
    ug_error_message ("*** ERROR 431 : error opening mode-state file ***");
    return (431);
  }

  iwrite = ug_fwrite (&isync, sizeof (INT_), 1, Data_File) - 1;

  ug_fclose (Data_File);

  if (iwrite < 0)
  {
    ug_error_message ("*** ERROR 432 : error writing mode-state file ***");
    return (432);
  }

  return (0);
}
