#include "UG_LIB.h"

/* 
 * UG LIB : Unstructured Grid - General Purpose Routine Library
 * $Id: ug_initialize_aflr_param.c,v 1.120 2023/11/20 01:38:31 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

INT_ ug_initialize_aflr_param (INT_ code, UG_Param_Struct * Param_Struct_Ptr)
{
  // Set the name, flag, default value, minimum allowable value, maximum
  // allowable value, and description for common AFLR/BLOOM3 related parameters.

  //    code=2 for AFLR2C
  //    code=3 for AFLR3 with ANBL3
  //    code=4 for AFLR4
  //    code=5 for BLOOM3
  //    code=44 for AFLR4 part of AFLR43
  //    code=43 for AFLR3 part of AFLR43
  //	code=-3 for AFLR3 using AFLR3t interface

  INT_ Index, Message_Flag,
       nelemdm, nelpnn, nelpnni, ninlpp, ninsmax, nmnrealloc, nqual, rank,
       Set_Vol_ID_Flag_def;
  INT_ ierr = 0;

  double cbidx, cdf, cdfn2, cinlpp, cnnpnni, csmin;

  // set combined params

  if (code == 2 || code == 3) {
    ierr = ug_set_char_param_struct ("-backgen", "",
     "Generate background grid and length-scale function files.",
     "\
Generate only an initial triangulation of the boundary points and sources (if \
any) without a recovered boundary grid and write background grid and \
length-scale function files using internal file writers. No other mesh \
generation is done. The output files will have a .back suffix and be of type \
UGRID and SFUNC if mw_func_type=1 or MESH and SOL if mw_func_type=2. ; \
Note that if there is an existing background grid, it is ignored (same as \
-no_back). \
     ",
     "mbackgen=1",
     2, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_char_param_struct ("-backgen3", "",
     "Generate background grid and length-scale function data.",
     "\
Generate only an initial triangulation of the boundary points and sources (if \
any) without a recovered boundary grid and return background grid and \
length-scale function data through the API. No other mesh \
generation is done. This option is only for internal use. \
     ",
     "mbackgen=3",
     -1, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  if (code == 2 || code == 3 || code == -3 || code == 43)
    ierr = ug_set_char_param_struct ("-backgen2", "",
     "Generate a background grid and length-scale function internally.",
     "\
Generate an internal background grid and length-scale function using the input \
surface boundary points and sources (if any). After background grid generation \
is complete it is used to generate the final volume grid. ; \
Do not use -backgen2 or mbackgen=2 with a mlsr=1. Instead generate the \
background grid separately with -backgen or mbackgen=1. \
Note that if there is an existing background grid, it is ignored (same as \
-no_back). \
     ",
     "mbackgen=2",
     2, 2, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code != 5) {

    ierr = ug_set_char_param_struct ("-met2", "",
     "Use metric space with advancing-front point placement.",
     "\
Use a metric space to determine all geometric quantities \
and use advancing-front point placement (equiangular type elements). \
     ",
     "mmet=1 mpp=2",
     2, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_char_param_struct ("-met3", "",
     "Use metric space with advancing-point point placement.",
     "\
Use a metric space to determine all geometric quantities \
and use advancing-point point placement (right angle type elements). \
     ",
     "mmet=1 mpp=3",
     2, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_char_param_struct ("-ext", "",
     "Use external sizing routine.",
     "\
Use external sizing routine with or without a background grid. \
If a site dependent specific routine is coded, compiled and linked with the \
executable, then this option is useful. \
If a background grid file and a background length-scale function file are \
found, then they \
are read for use in the external sizing evaluation routine. \
The background length-scale function file may contain either an isotropic \
sizing or a metric for directional sizing. Use whichever is allowed \
with for the external routine. \
By default, the built-in external sizing evaluation routine does not use \
a background grid. It is useful only for testing.  \
     ",
     "meval=1",
     2, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  if (code != 5) {

    ierr = ug_set_char_param_struct ("-grow", "",
     "Use specified growth in element size.",
     "\
Use specified growth in element size in a direction normal to the boundaries. \
     ",
     "mdf=2 cdfs=1.0 cdfr=",
     4, 3, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_char_param_struct ("-grow1", "",
     "Use moderate growth in element size.",
     "\
Use moderate growth in element size in a direction normal to the boundaries. \
     ",
     "mdf=2 cdfs=1.0 cdfr=1.2",
     4, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_char_param_struct ("-grow2", "",
     "Use high growth in element size.",
     "\
Use high growth in element size in a direction normal to the boundaries. \
     ",
     "mdf=2 cdfs=0.5 cdfr=1.5",
     4, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_char_param_struct ("-grow3", "",
     "Use very high growth in element size.",
     "\
Use very high growth in element size in a direction normal to the boundaries. \
     ",
     "mdf=2 cdfs=0.0 cdfr=2.0",
     4, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  if (code == 2)
    ierr = ug_set_char_param_struct ("-igen", "",
   "Generate an initial triangulation only.",
   "\
Generate only an initial triangulation of the boundary points with a \
fully recovered boundary grid. \
   ",
   "ngen=0 nqual=0",
   2, 2, Param_Struct_Ptr);
  else if (code == 3)
    ierr = ug_set_char_param_struct ("-igen", "",
   "Generate an initial triangulation only.",
   "\
Generate only an initial triangulation of the boundary points with a \
fully recovered boundary grid. \
   ",
   "ngen=0 nqual=0 mqrgen=0",
   2, 2, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2 || code == 3)
    ierr = ug_set_char_param_struct ("-no_source", "",
     "Do not use sources.",
     "\
Do not use sources even if there are source node files. \
This flag has no effect if there are no source node files.\
     ",
     "msource=0",
     2, 2, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2 || code == 4 || code == 44) {

    ierr = ug_set_char_param_struct ("-quad", "",
     "Generate a mixed quad/tria-face grid.",
     "\
With this option advancing-point \
point placement is used to generate right-angle tria-faces and, then \
tria-face pairs are combined to form quad-faces. \
     ",
     "mquad=1 mpp=3",
     1, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  if (code == 2 || code == 3 || code == 5 || code == -3 || code == 43) {

    ierr = ug_set_char_param_struct ("-no_vol_id", "-no_vid",
     "Do not set volume element ID flag.",
     "\
If selected then do not set the volume element ID. \
 ",
     "Set_Vol_ID_Flag=0",
     2, 2, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_char_param_struct ("-vid1", "",
     "Set volume element region ID flag.",
     "\
If selected then allocate and set the volume element ID to a unique \
value for each region in the domain. BL region elements are set to region IDs. \
     ",
     "Set_Vol_ID_Flag=1",
     2, 2, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_char_param_struct ("-vid2", "",
     "Set volume element region and BL layer ID flag.",
     "\
If selected then allocate and set the volume element ID to a unique \
value for each region in the domain and set each layer of the BL region, \
if any. The BL region IDs are set to negative the BL layer number. \
     ",
     "Set_Vol_ID_Flag=2",
     2, 2, Param_Struct_Ptr);

    if (ierr) return ierr;
  }

  // set int params

  if (code != 5)
    ierr = ug_set_int_param_struct ("malign", "",
     "Metric alignment flag.",
     "\
If malign = 0, then do not use metric alignment. ; \
If malign = 1, then use metric alignment to create candidate nodes with \
advancing-point (mpp=3) and advancing-front (mpp=2) point placement. ; \
If malign = 2, then use metric alignment to create candidate nodes \
and for each candidate select either advancing-point or advancing-front \
point placement. This option is only applicable if the base point placement \
is advancing-front (mpp=2). Otherwise it is the same as malign=1. ; \
Only applicable if external sizing evaluation is used (meval!=0) or \
if the advancing-point point placement option is selected (mpp=3). \
Only applicable if the metric space option is on (mmet=1 or 2). \
     ",
     2, 0, 1, 0, 2, Param_Struct_Ptr);
  else if (code == 4 || code == 44)
    ierr = ug_set_int_param_struct ("malign", "",
     "", "", 0, 0, 0, 0, 2, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2 || code == 3 || code == -3 || code == 43)
    ierr = ug_set_int_param_struct ("mbackgen", "",
     "Generate a background grid.",
     "\
If mbackgen = 0, then do not generate a background grid. ; \
If mbackgen = 1, then \
generate only an initial triangulation of the boundary points and sources (if \
any) without a recovered boundary grid and write background grid and \
length-scale function files using internal file writers. No other mesh \
generation is done. The background grid output files will be named \
case_name.back.*. See mw_func_type for file type. \
Note that if there is an existing background grid, it is ignored (as with \
-no_back). ; \
If mbackgen = 2, then \
generate an internal background grid and length-scale function using the input \
surface boundary points and sources (if any). After the background grid is \
generated use it to generate the final volume grid. \
Do not use -backgen2 or mbackgen=2 with a mlsr=1. Instead generate the \
background grid separately with -backgen or mbackgen=1. \
Note that if there is an existing background grid, it is ignored (same as \
-no_back). ; \
If mbackgen = 3, then \
generate only an initial triangulation of the boundary points and sources (if \
any) without a recovered boundary grid and return background grid and \
length-scale function data through the API. No other mesh \
generation is done. This option is only for internal use. ; \
If mbackgen = 4, then \
operations are exactly the same as with mbackgen=2. With mbackgen=4 also \
return the element neighbor connectivity for the final volume grid (this is \
the only difference between mbackgen=2 and mbackgen=4). ; \
If mbackgen = 5, then \
generate a background grid and length-scale function using the input \
surface boundary points and sources (if any). After the background grid is \
generated use it to generate the final volume grid. \
This option is only for internal use. \
     ",
     5, 0, 0, 0, 5, Param_Struct_Ptr);
  else if (code == 4 || code == 44)
    ierr = ug_set_int_param_struct ("mbackgen", "",
     "", "", 0, 0, 0, 0, 5, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code != 5)
    ierr = ug_set_int_param_struct ("mdf", "",
     "Distribution function flag.",
     "\
If mdf = 1, then interpolate the node distribution function for new nodes \
from the containing element. ; \
If mdf = 2, then use geometric growth from the boundaries to determine the \
distribution function for new nodes. \
     ",
     4, 0, 1, 1, 2, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2)
    ierr = ug_set_int_param_struct ("mdse", "",
   "Small edge deletion flag.",
"\
If mdse = 0, then do not delete small edges. ; \
If mdse = 1, then delete small edges. \
   ",
   4, 0, 1, 0, 1, Param_Struct_Ptr);
  else if (code == 3 || code == -3 || code == 43)
    ierr = ug_set_int_param_struct ("mdse", "",
   "Small edge deletion flag.",
"\
If mdse = 0, then do not delete small edges. ; \
If mdse = 1, then delete small edges. ; \
Note that small edge deletion is automatically turned on if \
candidate to node checking is off (mfchkn=0). \
   ",
   4, 0, 0, 0, 1, Param_Struct_Ptr);
  else if (code == 4 || code == 44)
    ierr = ug_set_int_param_struct ("mdse", "",
     "", "", 0, 0, 1, 0, 1, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code != 5)
    ierr = ug_set_int_param_struct ("melem", "",
     "Maximum number of elements.",
     "\
All dimensions key off this parameter. \
If melem = 0, then the code estimates the required dimensions. ; \
If melem > 0, then the code estimates are ignored, and all dimensions \
are initially set based on the value of melem. ; \
In either case the code will reallocate memory as needed during grid generation. \
     ",
     4, 0, 0, 0, 2000000000, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == -3 || code == 43) {
    ierr = ug_set_int_param_struct ("merge_vol_mesh", "",
     "Set option to merge duplicate boundary nodes and reorder.",
     "\
If merge_vol_mesh = 0 then do not merge duplicate nodes. ; \
If merge_vol_mesh = 1 then merge duplicate nodes and reorder. ; \
If merge_vol_mesh = 2 then merge duplicate nodes, reorder, set grid BC \
to transparent for the 1st matching face of each matching pair, and remove \
the 2nd matching face of each pair. ; \
If merge_vol_mesh = 3 then merge duplicate nodes, reorder, and remove both \
matching faces of each matching pair. ; \
Not applicable if there are no zones. \
     ",
     3, 0, 3, 0, 3, Param_Struct_Ptr);

    if (ierr) return ierr;
  }

  if (code == 2 || code == 3 || code == -3 || code == 43) {

    ierr = ug_set_int_param_struct ("mmet", "",
     "Metric space flag.",
     "\
If mmet = 0, then do not use a metric space. ; \
If mmet = 1, then use a metric space to determine all geometric quantities. ; \
If mmet = 2, then use a metric space to determine all geometric quantities \
along with geometric growth defined by parameters cdfrsrc and cdfssrc. \
Growth is not applicable with meval>=1 or a background grid. ; \
The metric field may be set from an externally defined sizing function \
(meval!=0) or the use of BL blending (mmetblbl=1 or 2). \
     ",
     2, 0, 0, 0, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_int_param_struct ("mw_met_hist", "",
     "Output flag for metric space analysis information.",
     "\
If mw_met_hist = 0, then do not output metric space analysis information. ; \
If mw_met_hist = 0, then output metric space analysis information. ; \
Applicable only if the metric space option is on (mmet=1 or 2). \
     ",
     2, 0, 0, 0, 1, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }
  else if (code == 4 || code == 44)
  {
    ierr = ug_set_int_param_struct ("mmet", "",
     "", "", 0, 0, 0, 0, 2, Param_Struct_Ptr);
    if (ierr) return (ierr);
    ierr = ug_set_int_param_struct ("mw_met_hist", "",
     "", "", 0, 0, 0, 0, 1, Param_Struct_Ptr);
    if (ierr) return (ierr);
  }

  if (code == 2 || code == 4 || code == 44) {

    ierr = ug_set_int_param_struct ("mquad", "",
     "Quad face combination flag.",
     "\
If mquad = 0, then do not combine \
tria faces to form quad faces. ; \
If mquad = 1, then combine tria faces \
to form quad faces. This option automatically sets \
the advancing-point placement option (mpp=3). \
     ",
     4, 0, 0, 0, 1, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  if (code == 4 || code == 44) {

    ierr = ug_set_int_param_struct ("mpp", "",
     "Point placement flag.",
     "\
If mpp = 2, then use advancing-front point placement. The produces optimal quality tria-faces. ; \
If mpp = 3, then use advancing-point point placement. The produces isotropic \
right-angle tria-faces. This option is intended for use with the quad element \
combination option (mquad=1). \
     ",
     4, 0, 2, 2, 3, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  else if (code == 2) {

    ierr = ug_set_int_param_struct ("mpp", "",
     "Point placement flag.",
     "\
If mpp = 1, then use centroid point placement. Centroid placement is very \
efficient. However, element quality is not optimal. ; \
If mpp = 2, then use advancing-front point placement. The produces optimal quality elements. ; \
If mpp = 3, then use advancing-point point placement. The produces isotropic \
right-angle elements. This option is intended for use with the quad face \
combination option (mquad=1). \
     ",
     4, 0, 2, 1, 3, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  else if (code != 5) {

    ierr = ug_set_int_param_struct ("mpp", "",
     "Point placement flag.",
     "\
If mpp = 1, then use centroid point placement. Centroid placement is very \
efficient. However, element quality is not optimal. ; \
If mpp = 2, then use advancing-front point placement. The produces optimal quality elements. ; \
If mpp = 3, then use advancing-point point placement. The produces isotropic \
right-angle elements. \
     ",
     4, 0, 2, 1, 3, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  if (code == 2)
    ierr = ug_set_int_param_struct ("meval", "",
     "External evaluation option flag.",
     "\
If meval = -2, then use interpolation from sources for \
sizing evaluation. This option is set internally if sources are \
input and meval=0. ; \
If meval = -1, then use interpolation from an input background grid for \
sizing evaluation. This option is set internally if a background grid is \
input and meval=0. ; \
If meval = 0, then do not use external sizing evaluation routines. ; \
If meval >= 1, then use externally defined sizing evaluation routine \
to determine the distribution function and metric for \
new nodes. If a site dependent specific routine is coded, compiled and \
linked with the executable, then this option is useful. \
An external routine may or may not use a background grid (if any) and/or \
source nodes (if any) that are input to AFLR. \
AFLR includes a built-in external sizing evaluation routine. By default, the \
built-in external routine is useful only for testing. The built-in external \
routine provides 10 different test cases. ; \
   meval = 1 is a diagonal ; \
   meval = 2 is a constant field ; \
   meval = 3 is a blended cross ; \
   meval = 4 is a circle ; \
   meval = 5 is a is a quarter circle ; \
     ",
     4, 0, 0, -1, 100, Param_Struct_Ptr);
  else if (code == 3)
    ierr = ug_set_int_param_struct ("meval", "",
     "External evaluation option flag.",
     "\
If meval = -2, then use interpolation from sources for \
sizing evaluation. This option is set internally if sources are \
input and meval=0. ; \
If meval = -1, then use interpolation from an input background grid for \
sizing evaluation. This option is set internally if a background grid is \
input and meval=0. ; \
If meval = 0, then do not use external sizing evaluation routines. ; \
If meval >= 1, then use externally defined sizing evaluation routine \
to determine the distribution function and metric for \
new nodes. If a site dependent specific routine is coded, compiled and \
linked with the executable, then this option is useful. \
An external routine may or may not use a background grid (if any) and/or \
source nodes (if any) that are input to AFLR. \
AFLR includes a built-in external sizing evaluation routine. By default, the \
built-in external routine is useful only for testing. The built-in external \
routine provides 10 different test cases. ; \
   meval = 1 is a centered source ; \
   meval = 2 is a centered source with no x-variation ; \
   meval = 3 is a centered source with no y-variation ; \
   meval = 4 is a centered source with no z-variation ; \
   meval = 5 is a horizontal sheet ; \
   meval = 6 is a blended cross ; \
   meval = 7 is a circle ; \
   meval = 8 is a horizontal sheet that requires boundary surface adaptation ; \
   meval = 9 is a blended cross that requires boundary surface adaptation ; \
   meval = 10 is a quarter circle that requires boundary surface adaptation ; \
Note that meval >= 5 is only \
only applicable if the metric space option is on (mmet=1 or 2). \
     ",
     4, 0, 0, -1, 100, Param_Struct_Ptr);
  else if (code == 4 || code == -3 || code == 43 || code == 44)
    ierr = ug_set_int_param_struct ("meval", "",
     "", "", 0, 0, 0, -1, 100, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2 || code == 3 || code == -3 || code == 43)
    ierr = ug_set_int_param_struct ("msource", "",
     "Source option flag.",
     "\
If msource = 0, then do not use sources. ; \
If msource = 1 and sources are input, then insert the sources into the initial grid. ; \
If msource = 2 and meval = 0 (on input) or meval = -2 (after reset) and sources are input, then create an oct-tree of the source \
nodes for interpolating the local distribution function and metric \
during grid generation. ; \
If msource = 2 and meval = -1 (after reset), then msource is automatically reset to msource=0. ; \
If msource = 2 and meval >= 1, then msource is automatically reset to msource=3. ; \
If msource = 2 and meval >= 1, then msource is automatically reset to msource=3. ; \
If msource = 3 and meval >= 1, then use externally defined evaluation routines to determine \
the distribution function and metric for new nodes. \
If the external routines do not use source nodes, \
then presence of source nodes will have no effect. ; \
Sources modify the point distribution function and metric \
and the resulting element size and shape in the region near a source node. The \
msource option has no effect if there are not sources nodes. \
     ",
     2, 0, 1, 0, 3, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2 || code == 3 || code == -3 || code == 43) {

    ierr = ug_set_int_param_struct ("mw_func", "",
     "Write BG function file flag.",
     "\
If mw_func = 0, then do not write grid function file. ; \
If mw_func = 1, then write grid function file with scalar \
isotropic length scale and/or directional length scale metric. The values will \
be those that exist at the end of mesh generation. \
The output file will be case_name.out.b8.sfunc if mw_func_type=1 or \
case_name.out.solb if mw_func_type=2. \
     ",
     4, 0, 0, 0, 1, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_int_param_struct ("mw_func_type", "",
     "Write BG function file flag.",
     "\
If mw_func_type = 1, then write grid function files using UGRID and SFUNC types. ; \
If mw_func_type = 2, then write grid function files using MESHB and SOLB types. ;\
be those that exist at the end of mesh generation. \
     ",
     4, 0, 1, 1, 2, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }
  else if (code == 4 || code == 44)
  {
    ierr = ug_set_int_param_struct ("msource", "",
     "", "", 0, 0, 1, 0, 3, Param_Struct_Ptr);
    if (ierr) return (ierr);
    ierr = ug_set_int_param_struct ("mw_func", "",
     "", "", 0, 0, 0, 0, 1, Param_Struct_Ptr);
    if (ierr) return (ierr);
    ierr = ug_set_int_param_struct ("mw_func_type", "",
     "", "", 0, 0, 1, 1, 2, Param_Struct_Ptr);
    if (ierr) return (ierr);
  }

  if (code == 2 || code == 3 || code == -3 || code == 43)
    ierr = ug_set_int_param_struct ("m_debug_mode", "",
     "Debug mode flag.",
"\
If m_debug_mode = 0, then operate in normal mode. ; \
If m_debug_mode = 1, then operate in debug mode. Note that not all codes \
have debug mode implemented. ; \
     ",
     5, 0, 0, 0, 1, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code != 5) {

    nelemdm = (code == 2 || code == 4 || code == 44) ? 100: 1000;

    ierr = ug_set_int_param_struct ("nelemdm", "",
     "Minimum number of elements to allocate.",
     "\
Minimum value for number of elements to allocate space for. \
     ",
     5, 0, nelemdm, 100, 10000000, Param_Struct_Ptr);

    if (ierr) return (ierr);

    nelpnn = (code == 2 || code == 4 || code == 44) ? 2: 5;

    ierr = ug_set_int_param_struct ("nelpnn", "",
     "Number of elements per node.",
     "\
The maximum number of nodes allocated is set to the maximum of either the    \
number of elements allocated divided by nelpnn or the number of initial nodes \
multiplied by nnpnni. \
     ",
     5, 0, nelpnn, 2, 7, Param_Struct_Ptr);

    if (ierr) return (ierr);

    nelpnni = (code == 2 || code == 4 || code == 44) ? 3: 5;

    ierr = ug_set_int_param_struct ("nelpnni", "",
     "Number of initial elements per node.",
     "\
The maximum number of initial elements allocated is set to the number of nodes \
in the boundary grid multiplied by nelpnni. This is used only if there \
is not an initial triangulation and it is used only to generate the initial triangulation. \
     ",
     5, 0, nelpnni, 2, 100, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ninsmax = (code == 2 || code == 4 || code == 44) ? 100: 3;

    ierr = ug_set_int_param_struct ("ninsmax", "",
     "Maximum element subdivisions.",
     "\
Maximum number of element subdivisions during direct boundary node insertion. \
     ",
     5, 0, ninsmax, 1, 10000000, Param_Struct_Ptr);

    if (ierr) return (ierr);

    nmnrealloc = (code == 2 || code == 4 || code == 44) ? 1000: 10000;

    ierr = ug_set_int_param_struct ("nmnrealloc", "",
     "Minimum array elements to reallocate.",
     "\
Minimum number of new array elements to add if the initial estimate for maximum \
number of elements is too low. \
     ",
     5, 0, nmnrealloc, 100, 10000000, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_int_param_struct ("ngen", "",
     "Maximum number of grid passes.",
     "",
     5, 0, 10000, 0, 10000000, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ninlpp = (code == 2 || code == 4 || code == 44) ? 0: 4;

    ierr = ug_set_int_param_struct ("ninlpp", "",
     "Maximum initial point placement passes.",
     "\
During grid generation large elements that have all edges larger than the \
distribution function (local point spacing) are subdivided using centroid \
point placement. This is repeated for ninlpp passes or less if there are no \
elements to subdivide. Standard point placement (as specified by mpp) is then \
used. A value of ninlpp=0 will turn off initial point placement and standard \
point placement will be used from the start. \
     ",
     5, 0, ninlpp, 0, 10000000, Param_Struct_Ptr);

    if (ierr) return (ierr);
 
    nqual = (code == 2 || code == 4 || code == 44) ? 1: 2;

    ierr = ug_set_int_param_struct ("nqual", "",
     "Number of quality improvement passes.",
"\
If nqual = 0, then all quality improvement is skipped. ; \
     ",
     5, 0, nqual, 0, 10, Param_Struct_Ptr);

    if (ierr) return (ierr);
 
    ierr = ug_set_int_param_struct ("nsmth", "",
     "Number of smoothing iterations.",
     "",
     5, 0, 3, 0, 10, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }

  if (code == 2 || code == 3 || code == -3 || code == 43)
    ierr = ug_set_int_param_struct ("nlsrpp", "",
     "Maximum LSR point placement passes.",
     "\
Initial length scale reduction point placement is used with advancing-type \
point placement (mpp=2,3) for initial elements that have length scales smaller \
than lsrpplim. Standard advancing-type point placement is used after the \
initial element length scales are reduced. A value of nlsrpp=0 will turn \
off initial LSR point placement. \
     ",
     4, 0, 0, 0, 10000000, Param_Struct_Ptr);
  else if (code == 4 || code == 44)
    ierr = ug_set_int_param_struct ("nlsrpp", "",
     "", "", 0, 0, 0, 0, 10000000, Param_Struct_Ptr);

  if (ierr) return (ierr);

  ierr = ug_set_int_param_struct ("nnpbchk", "",
   "Quad-/Oct-tree bin checking node limit.",
   "\
Node limit target used to generate the quad-/oct-tree for \
checking data sets of nodes. \
The node limit target for final bins is set to nnpbchk boundary nodes. \
   ",
   5, 0, 100, 10, 10000, Param_Struct_Ptr);

  if (ierr) return (ierr);

  ierr = ug_set_int_param_struct ("nnpbeval", "",
   "Quad-/Oct-tree bin evaluation node limit.",
   "\
Node limit target used to generate the quad-/oct-tree for \
evaluation of data sets with functions specified at nodes. \
The node limit target for final bins is set to nnpbeval source nodes. \
   ",
   5, 0, 10, 10, 10000, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code != 5)
    ierr = ug_set_int_param_struct ("nnpnni", "",
     "Number of nodes per initial node.",
     "\
The maximum number of nodes allocated is set to the maximum of either the  \
number of elements allocated divided by by nelpnn or the number of initial nodes \
multiplied by nnpnni. \
     ",
     5, 0, 2, 1, 10, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2 || code == 4 || code == 44) {

    ierr = ug_set_int_param_struct ("nquadc", "",
     "Number of quad combination passes.",
     "\
The maximum quad face angles are \
varied between initial and final values \
over nquadc passes. During each pass \
quality improvement is performed. Only \
applicable if quad face combination \
flag is on (mquad>=1). \
     ",
     5, 0, 4, 1, 10, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }
 
  ierr = ug_set_int_param_struct ("cpu_timer", "",
   "Routine level CPU timer flag.",
   "\
If cpu_timer = 0, then do not use routine level CPU timer. ; \
If cpu_timer = 1, then use routine level CPU timer. ; \
   ",
   5, 0, 0, 0, 1, Param_Struct_Ptr);

  if (ierr) return (ierr);
 
  Set_Vol_ID_Flag_def = (code == 5) ? 2: 1;

  if (code == 3 || code == 5 || code == -3 || code == 43)
    ierr = ug_set_int_param_struct ("Set_Vol_ID_Flag", "",
     "Set volume element ID flag.",
     "\
If Set_Vol_ID_Flag = 0 then do not allocate or \
set the volume element ID. ; \
If Set_Vol_ID_Flag = 1 then allocate and set the volume element ID to a unique \
value for each region in the domain. ; \
If Set_Vol_ID_Flag = 2 then allocate and set the volume element ID to a unique \
value for each region in the domain and set each layer of the BL region. \
The BL region IDs are set to negative the BL layer number. \
     ",
     4, 0, Set_Vol_ID_Flag_def, 0, 2, Param_Struct_Ptr);

  if (ierr) return ierr;

  // set double params

  ierr = ug_set_double_param_struct ("angdbe", "",
   "Discontinuous boundary edge angle.",
   "\
Angle between two adjacent boundary \
edge vectors used to identify edge \
discontinuities.",
   4, 0, 30.0, 0.0, 179.9, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 2 || code == 4 || code == 44) {

    ierr = ug_set_double_param_struct ("angquad1", "",
     "Maximum aligned quad face angle.",
     "\
Quads are first formed by combining \
aligned face pairs starting at a \
boundary edge. Tria face combination is \
locally terminated if the maximum quad \
face angle exceeds angquad1. \
Only applicable if the quad face \
combination flag is on (mquad>=1). \
     ",
     4, 0, 120.0, 100.0, 179.9, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("angquad1i", "",
     "Initial maximum aligned quad angle.",
     "\
The initial maximum aligned quad angle \
is varied between angquad1i and angquad1 \
over nquadc passes. Only applicable if \
quad face combination flag is on \
(mquad>=1). \
     ",
     5, 0, 130.0, 100.0, 179.9, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("angquad2", "",
     "Maximum non-aligned quad face angle.",
     "\
Quads are also formed by combining \
non-aligned face pairs after all \
possible aligned quads are formed. \
Tria faces are not combined if the maximum \
quad face angle exceeds angquad2. \
Only applicable if the quad face \
combination flag is on (mquad>=1). \
     ",
     4, 0, 140.0, 100.0, 179.9, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("angquad2i", "",
     "Initial maximum non-aligned quad angle.",
     "\
The initial maximum non-aligned quad \
angle is varied between angquad2i and \
angquad2 over nquadc passes. Only \
applicable if the quad face \
combination flag is on (mquad>=1). \
     ",
     5, 0, 170.0, 100.0, 179.9, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("angquad3", "",
     "Final maximum quad face angle.",
     "\
One additional pass beyond the nquadc \
quad formation passes is performed with \
the maximum non-aligned quad face \
angle, angquad2, set to the final \
maximum, angquad3. Only applicable if \
the quad face combination flag is on \
(mquad>=1). \
     ",
     4, 0, 140.0, 100.0, 179.9, Param_Struct_Ptr);

    if (ierr) return (ierr);
  }
 
  if (code == 3 || code == 5 || code == -3 || code == 43) {

    ierr = ug_set_double_param_struct ("angbd", "",
     "Discontinuous surface angle.",
     "\
Dihedral angle between two adjacent faces used to identify discontinuous \
edges. Edges are considered discontinuous if the \
dihedral angle between adjacent faces is less than angbd. \
     ",
     5, 0, 140.0, 90.0, 179.9, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_double_param_struct ("angqbf", "",
     "Maximum planar surface angle.",
"\
Planar surface angle for boundary \
triangular faces used to check the \
boundary surface triangulation. The \
boundary surface triangulation is not \
considered valid if the planar angle for \
a vertex of any face is greater than \
angqbf. To turn this check off use option angqbf=180. \
     ",
     4, 0, 179.5, 120.0, 180.0, Param_Struct_Ptr);

    if (ierr) return ierr;
 
    ierr = ug_set_double_param_struct ("angqbfmin", "",
     "Minimum planar surface angle.",
"\
Planar surface angle for boundary \
triangular faces used to check the \
boundary surface triangulation. The \
boundary surface triangulation is not \
considered valid if the planar angle for \
a vertex of any face is less than \
angqbfmin. This typically means there are co-located nodes. \
It could also be intentional with very high-aspect-ratio boundary faces. \
To turn this check off use option angqbfmin=0. \
     ",
     4, 0, 1.0e-4, 0.0, 180.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("angrbfsd", "",
     "Small dihedral surface angle.",
     "\
Dihedral angle between two adjacent \
faces used to check the boundary surface \
triangulation. The boundary surface \
triangulation is not considered valid if \
the dihedral angle (within the domain) \
between two faces is less than angrbfsd. \
     ",
     5, 0, 2.0, 0.0, 10.0, Param_Struct_Ptr);

    if (ierr) return ierr;
  }

  ierr = ug_set_double_param_struct ("bdfmchk", "",
   "Quad-/Oct-tree bin checking size factor.",
   "\
Factor used to generate the quad-/oct-tree for checking data sets of nodes. \
The size of the smallest quad-/oct-tree bin is set to the local length scale \
multiplied by bdfmchk. \
   ",
   5, 0, 1.0, 1.0, 100000.0, Param_Struct_Ptr);

  if (ierr) return (ierr);

  ierr = ug_set_double_param_struct ("bdfmeval", "",
   "Quad-/Oct-tree bin evaluation size factor.",
   "\
Factor used to generate the quad-/oct-tree for evaluation of data sets \
with functions specified at nodes. \
The size of the smallest quad-/oct-tree bin is set to the local length scale \
multiplied by bdfmeval. \
   ",
   5, 0, 1.0, 1.0, 100000.0, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 3 || code == 5 || code == -3 || code == 43) {

    ierr = ug_set_double_param_struct ("cblchkbb", "",
     "Intersection checking box size factor.",
     "\
The size of the bounding box for checking BL interface edge-face \
intersections is increased in all directions by the local normal spacing \
multiplied by the sum of the factors cdfn and cblchkbb. Increasing \
cblchkbb will increase the number of faces and edges that are checked for \
intersections. \
     ",
     5, 0, 0.0, 0.0, 1000.0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_double_param_struct ("cblsrchbb", "",
     "Nearby node search bounding box size factor.",
     "\
The size of the bounding box for determining if a BL node is nearby is \
set equal to the local length scale multiplied by the factor cblsrchbb. \
These nearby BL nodes are used for checking for possible face \
intersections within the BL region. ; \
Only applicable with an open method (BLOOM3_Mode_Flag=2). \
     ",
     5, 0, 3.0, 1.0, 1000.0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_double_param_struct ("cblsrchbb2", "",
     "Nearby surf search bounding box size factor.",
     "\
The size of the bounding box for determining if a non-BL surface node is \
nearby a BL node is set equal to the local length scale multiplied by the \
factor cblsrchbb2. These nearby surface nodes are used for checking for possible \
intersections with the nearby BL region. \
     ",
     5, 0, 1.2, 1.0, 1000.0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_double_param_struct ("sminj", "",
     "Face intersection location tolerance.",
     "\
Relative tolerance for checking if BL interface faces intersect near rebuild \
surfaces that intersect the BL region and for checking if interface faces are \
too close. ; \
Only applicable with an open method (BLOOM3_Mode_Flag=2). \
     ",
     5, 0, 0.3, 0.0, 1.0e+6, Param_Struct_Ptr);

    if (ierr) return ierr;
  }

  if (code != 5) {

    cdf = (code == 2 || code == 4 || code == 44) ? 1.0: 1.2;
    rank = (code == 4 || code == 44) ? 5: 3;

    ierr = ug_set_double_param_struct ("cdf", "",
     "Distribution function multiplier.",
     "\
The distribution function is used to specify desired element size. The \
distribution function originally determined from the average of the \
surrounding boundary edge lengths is multiplied by cdf. Increasing cdf will \
increase the total number of grid nodes generated. Decreasing cdf will decrease \
the total number of grid nodes generated. Deviation of cdf far from a \
value of 1.0 will degrade the element quality next to the boundaries. A cdf \
value above 1.0 will create elements adjacent to boundaries that are \
elongated normal to the boundary. In the rest of the field the elements \
should be relatively isotropic. \
     ",
     rank, 0, cdf, 0.5, 3.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("cdff", "",
     "Satisfied edge length multiplier.",
     "\
An edge is considered satisfied if its length divided by cdff is less than the \
average of the node distribution function at the edge end-nodes. \
     ",
     5, 0, 1.5, 1.1, 3.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("cdff2", "",
     "Satisfied edge length multiplier #2.",
     "\
An edge is considered satisfied if its length divided by cdff is less than the \
average of the node distribution function at the edge end-nodes. \
The value of the satisfied edge length multiplier, cdff, is replaced with the \
value of multiplier #2, cdff2, if \
external sizing evaluation is used (meval!=0) or \
if the advancing-point point placement option is selected (mpp=3). \
     ",
     5, 0, 2.0, 1.1, 3.0, Param_Struct_Ptr);

    if (ierr) return (ierr);
 
    if (code == 2 || code == 3 || code == -3 || code == 43)
      ierr = ug_set_double_param_struct ("cdfn_se", "",
       "Small edge factor.",
       "\
Small edges are deleted if they have a ratio \
of actual edge length to local length \
scale of cdfn_se or less. ; \
Only applicable \
if small edge deletion is on (mdse=1). \
       ",
       4, 0, 0.5, 0.1, 0.7, Param_Struct_Ptr);
    else
      ierr = ug_set_double_param_struct ("cdfn_se", "",
       "", "", 0, 0, 0.5, 0.1, 0.7, Param_Struct_Ptr);

    if (ierr) return (ierr);

    cbidx = (code == 2 || code == 4 || code == 44) ? 1.0: 2.0;

    ierr = ug_set_double_param_struct ("cbidx", "",
     "Initial bounding box size factor.",
     "\
The initial bounding box used for insertion of all boundary nodes is sized \
to be equal to the size of the actual domain plus two times cbidx multiplied \
by the size of the actual domain. \
     ",
     5, 0, cbidx, 1.0, 100000.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    if (code == 2 || code == 3 || code == -3 || code == 43)
      ierr = ug_set_double_param_struct ("cdfm", "",
       "Distribution function weighting factor.",
       "\
The node distribution function for new nodes is averaged with the minimum \
nearby node distribution functions. This factor is the weighting for the minimum \
contribution. Increasing cdfm above 0.0 will in general reduce the growth of \
element size from small to larger elements and increase the total number \
of grid nodes generated. It will have little or no effect if all the boundary \
edges are nearly the same size. \
       ",
       4, 0, 0.0, 0.0, 1.0, Param_Struct_Ptr);
    else
      ierr = ug_set_double_param_struct ("cdfm", "",
       "", "", 0, 0, 0.0, 0.0, 1.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    cdfn2 = (code == 3 || code == -3 || code == 43) ? 0.65: 0.7;

    ierr = ug_set_double_param_struct ("cdfn2", "",
     "Nearby node factor #2.",
     "\
Nodes are considered too close if the distance between them is less than their \
average node distribution functions multiplied by cdfn. \
The value of the nearby node factor, cdfn, is replaced with the \
value of factor #2, cdfn2, if \
if the advancing-point point placement option is selected (mpp=3). \
     ",
     5, 0, cdfn2, 0.5, 0.9, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("cdfr", "",
     "Maximum geometric growth rate.",
     "\
Used as the advancing-front growth limit. The element size for new nodes is \
limited to be less than the physical size of the local front advanced from    \
multiplied by cdfr. Also used as the geometric growth rate for the node \
distribution function with the growth option (mdf=2). A cdfr value just above \
1.0 will produce a grid with optimal element quality. A value of cdfr well \
above a value of 1.0 will decrease the number of grid nodes generated and \
potentially decrease the element quality. \
     ",
     4, 0, 1.1, 1.0, 3.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    cinlpp = (code == 2 || code == 4 || code == 44) ? 0.01: 0.1;

    ierr = ug_set_double_param_struct ("cinlpp", "",
     "Initial point placement factor.",
     "\
During grid generation large elements that have all edges larger than the \
distribution function (local point spacing) are subdivided using centroid \
point placement. Elements that have a ratio of local spacing over minimum edge \
length less than cinlpp are subdivided. A value of cinlpp=0 will turn off \
initial point placement and standard point placement will be used from the start. \
     ",
     5, 0, cinlpp, 0.0, 0.5, Param_Struct_Ptr);

    if (ierr) return (ierr);

    cnnpnni = (code == 2 || code == 4 || code == 44) ? 1.5: 1.25;

    ierr = ug_set_double_param_struct ("cnnpnni", "",
     "Number of initial nodes multiplier.",
     "\
The number of nodes allocated for the initial triangulation is set to \
the number of initial nodes multiplied by cnnpnni. Additional nodes beyond \
those in the initial boundary grid may be required to complete boundary recovery \
and obtain an initial triangulation. \
     ",
     5, 0, cnnpnni, 1.0, 3.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("crealloc", "",
     "Reallocation multiplier.",
     "\
The maximum number of elements and nodes is increased, and all arrays are \
reallocated if more elements or nodes are required to complete the grid. The \
allocations for each array are increased by crealloc multiplied by their current \
value. The new array size is limited by the maximum number of new array elements \
set in parameter nmnrealloc. \
     ",
     5, 0, 1.25, 1.05, 2.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("csmth", "",
     "Smoothing coefficient.",
     "",
     5, 0, 0.5, 0.0, 1.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    if (code == 2 || code == 3 || code == -3 || code == 43)
    {
      ierr = ug_set_double_param_struct ("csrcw", "",
       "Source weight.",
       "\
Weighting factor used to determine distribution function and metric \
from sources. The distribution function and metric \
from sources are weighted by csrcw and that from the \
local grid is weighted by 1-csrcw. ; \
Only applicable if the source interpolation option flag is on (msource=2). \
       ",
       4, 0, 0.8, 0.0, 1.0, Param_Struct_Ptr);

      if (ierr) return (ierr);

      ierr = ug_set_double_param_struct ("dfmax", "",
       "Maximum distribution function.",
       "\
The distribution function specifies the point spacing in the field. \
If dfmax < 0, then do not limit the maximum distribution function. ; \
If dfmax = 0, then limit the maximum distribution function to the maximum \
value determined from the boundary surface grid. ; \
If dfmax > 0, then limit the maximum distribution function to dfmax. \
       ",
       4, 0, -1.0, -1.0, 1.0e+19, Param_Struct_Ptr);

      if (ierr) return (ierr);

      ierr = ug_set_double_param_struct ("lsrpplim", "",
       "LSR point placement limit factor.",
       "\
If initial length scale reduction (LSR) point placement is specified (nlsrpp>0), \
then initial length scale reduction point placement is used with advancing-type point placement \
(mpp=2,3) for initial elements that have length scales smaller than lsrpplim. Standard \
point placement is used after the initial element length scales are \
reduced. A value of lsrpplim=0 will turn off initial LSR point placement. \
       ",
       4, 0, 0.01, 0.0, 1.0, Param_Struct_Ptr);
    
      if (ierr) return (ierr);
    }
    else {

      ierr = ug_set_double_param_struct ("csrcw", "",
       "", "", 0, 0, 0.8, 0.0, 1.0, Param_Struct_Ptr);
      if (ierr) return (ierr);
      ierr = ug_set_double_param_struct ("dfmax", "",
       "", "", 0, 0, -1.0, -1.0, 1.0e+19, Param_Struct_Ptr);
      if (ierr) return (ierr);
      ierr = ug_set_double_param_struct ("lsrpplim", "",
       "", "", 0, 0, 0.01, 0.0, 1.0, Param_Struct_Ptr);
      if (ierr) return (ierr);
    }

    ierr = ug_set_double_param_struct ("relem0", "",
     "Re-numbering limit.",
     "\
Elements are not re-numbered if the ratio of elements to be moved to total \
elements is less than relem0. \
     ",
     5, 0, 0.1, 0.0, 1.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_double_param_struct ("vsmthb", "",
     "Smoothing reduction factor.",
     "\
Smoothing coefficient reduction factor for all nodes adjacent to a boundary. \
     ",
     5, 0, 0.5, 0.0, 1.0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    if (code == 2 || code == 3 || code == -3 || code == 43) {

      ierr = ug_set_double_param_struct ("cdfrsrc", "",
       "Maximum geometric source growth rate.",
       "\
Used as the geometric growth rate for the node distribution function and \
metric from sources. \
       ",
       4, 0, 1.2, 1.0, 3.0, Param_Struct_Ptr);

      if (ierr) return (ierr);

      ierr = ug_set_double_param_struct ("cdfssrc", "",
       "Source distribution function exclusion zone.",
       "\
The source distribution function and metric \
remain constant near a source node for a distance equal to the source \
distribution function multiplied by cdfssrc. \
       ",
       4, 0, 2.0, 0.0, 100.0, Param_Struct_Ptr);

      if (ierr) return (ierr);

      ierr = ug_set_double_param_struct ("cdfs", "",
       "Distribution function exclusion zone.",
       "\
Distribution function growth exclusion zone factor used with growth option \
(mdf=2). The node distribution function and element size remain constant near a \
boundary node for a distance equal to the node distribution function at the \
boundary multiplied by cdfs. \
       ",
       3, 0, 1.0, 0.0, 10.0, Param_Struct_Ptr);

      if (ierr) return (ierr);
    }
    else if (code == 4 || code == 44) {

      ierr = ug_set_double_param_struct ("cdfrsrc", "",
       "", "", 0, 0, 1.2, 1.0, 3.0, Param_Struct_Ptr);
      if (ierr) return (ierr);
      ierr = ug_set_double_param_struct ("cdfssrc", "",
       "", "", 0, 0, 2.0, 0.0, 100.0, Param_Struct_Ptr);
      if (ierr) return (ierr);
      ierr = ug_set_double_param_struct ("cdfs", "",
       "", "", 0, 0, 1.0, 0.0, 10.0, Param_Struct_Ptr);
      if (ierr) return (ierr);
    }
  }

  ierr = ug_set_double_param_struct ("cdfn", "",
   "Nearby node factor.",
   "\
Nodes are considered too close if the distance between them is less than their \
average node distribution functions multiplied by cdfn. \
   ",
   5, 0, 0.7, 0.5, 0.9, Param_Struct_Ptr);

  if (ierr) return (ierr);

  csmin = (code == 2 || code == 4 || code == 44) ? 0.5: 0.8;

  ierr = ug_set_double_param_struct ("csmin", "",
   "Tolerance exponent for searching.",
   "",
   5, 0, csmin, 0.4, 0.9, Param_Struct_Ptr);

  if (ierr) return (ierr);

  ierr = ug_set_double_param_struct ("csmini", "",
   "Initial tolerance exponent.",
   "\
Tolerance exponent for searching during initial triangulation. \
   ",
   5, 0, 0.25, 0.125, 0.9, Param_Struct_Ptr);

  if (ierr) return (ierr);

  ierr = ug_set_double_param_struct ("ctol", "",
   "Overall tolerance exponent.",
   "", 
   5, 0, 0.94, 0.8, 0.94, Param_Struct_Ptr);

  if (ierr) return (ierr);

  if (code == 3 || code == 5 || code == -3 || code == 43)
    ierr = ug_set_double_param_struct ("ctolm", "",
     "BL grid volume tolerance exponent.",
     "",
     5, 0, 0.88, 0.5, 0.94, Param_Struct_Ptr);

  if (ierr) return ierr;

  // set BC parameters

  if (code != 2) {

    ierr = ug_set_int_param_struct ("BC_IDs", "bc_ids",
     "List of IDs to set grid BC parameters.",
     "\
If the BC_IDs and the Grid_BC_Flag parameter vectors are set \
then the grid BC flag is set for all faces with corresponding surface IDs. \
Only those IDs listed will have their corresponding values set. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     2, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;
 
    ierr = ug_set_int_param_struct ("Grid_BC_Flag", "bc_list",
     "List of grid BC flags to set.",
     "\
If the BC_IDs and the Grid_BC_Flag parameter vectors are set \
then the grid BC flag is set for all faces with corresponding surface IDs. \
Values for the Grid_BC_Flag \
and their meaning are listed below. ; \
; \
 0 = FarField surface ; \
 1 = Solid surface ; \
-1 = BL Generating Solid surface ; \
 2 = BL Intersecting surface ; \
 3 = Source Transparent surface converted to source nodes ; \
 4 = BL Intersecting Transparent surface ; \
 5 = Transparent surface ; \
-5 = BL Generating Transparent surface ; \
 6 = Internal Transparent surface ; \
-6 = BL Generating Internal Transparent surface ; \
 7 = Fixed Surface with BL region that intersects BL region ; \
; \
Note that 1) FarField and Solid surfaces behave the same during volume mesh ; \
generation, 2) all Transparent surfaces will have a volume mesh on both sides, \
and 3) Internal Transparent surfaces are converted to internal faces and \
their boundary face connectivity is not included in the final mesh. ; \
Commas are required between the grid BC flags in the list. \
If only one grid BC flag is specified then it must be terminated with a comma. \
     ",
     2, 1, 0, 0, 0, Param_Struct_Ptr);

    ierr = ug_set_int_param_struct ("Trnsp_IDs", "-trnsp_ids",
     "List of IDs for Transparent surfaces.",
     "\
If the vector Trnsp_IDs is set then the grid BC flag is set to a \
value specifying a Transparent surface for \
all faces with a surface ID in the list. \
For example, if the vector Trnsp_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to Transparent surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_int_param_struct ("Trnsp_Src_IDs", "-trnsp_src_ids",
     "List of IDs for Transparent Source surfaces.",
     "\
If the vector Trnsp_Src_IDs is set then the grid BC flag is set to a \
value specifying a Transparent Source surface for \
all faces with a surface ID in the list. \
For example, if the vector Trnsp_Src_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to Transparent Source surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_int_param_struct ("Trnsp_Intl_IDs", "-trnsp_intl_ids",
     "List of IDs for Internal Transparent surfaces.",
     "\
If the vector Trnsp_Intl_IDs is set then the grid BC flag is set to a \
value specifying a Internal Transparent surface for \
all faces with a surface ID in the list. \
For example, if the vector Trnsp_Intl_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to Internal Transparent surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_int_param_struct ("BL_IDs", "-bl_ids",
     "List of IDs for BL Generating Solid surfaces.",
     "\
If the vector BL_IDs is set then the grid BC flag is set to a \
value specifying a BL Generating Solid surface for all faces with a surface ID \
in the list. \
For example, if the vector BL_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to BL Generating Solid surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_int_param_struct ("BL_Int_IDs", "-bl_ints -bl_int_ids -ints -int_ids",
     "List of IDs for BL Intersecting surfaces.",
     "\
If the vector BL_Int_IDs is set then the grid BC flag is set to a \
value specifying a BL Intersecting surface for all faces with a surface ID \
in the list. \
The surface mesh for BL Intersecting surfaces is regenerated to match the BL \
region. \
For example, if the vector BL_Int_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to BL Intersecting surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;
 
    if (code == 3)
      ierr = ug_set_int_param_struct ("Fixed_BL_Int_IDs", "-fints -fbl_ints -fbl_int_ids",
       "List of IDs for BL Intersecting surfaces.",
       "\
If the vector BL_Int_IDs is set then the grid BC flag is set to a \
value specifying a BL Intersecting surface for all faces with a surface ID \
in the list. \
The surface mesh for BL Intersecting surfaces is regenerated to match the BL \
region. \
For example, if the vector BL_Int_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to BL Intersecting surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
       ",
       1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;

    ierr = ug_set_int_param_struct ("FF_IDs", "-ffs -ff_ids",
     "List of IDs to for FarField surfaces.",
     "\
If the vector FF_IDs is set then the grid BC flag is set to a \
value specifying a FarField surface for all faces with a surface ID \
in the list. \
For example, if the vector FF_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to FarField surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return (ierr);
 
    ierr = ug_set_int_param_struct ("Rec_Flag", "",
     "List of reconnection flags to reset.",
"\
If the BC_IDs and Rec_Flag parameter \
vectors are set then the reconnection \
flag is reset to the values in the \
Rec_Flag vector for all faces with \
corresponding surface IDs. Reconnection of surface faces \
is used to assist boundary recovery and eliminate boundary \
sliver elements. Values for the Rec_Flag \
and their meaning for tria faces are listed below. \
However only the values 0 (reconnection allowed) or \
7 (reconnection not allowed) are practical in general usage. \
The intermediate values are typically set internally based on \
curvature, quality and neighbor ID. \
For quad faces the Rec_Flag is ignored and the outer \
edges of the quad faces are always preserved. The neighbor \
faces referenced below are for a tria face connectivity \
ordered node 1, 2, 3. ; \
; \
0 = reconnection allowed with any neighbor ; \
1 = reconnection not allowed with neighbor opposite node 1 ; \
2 = reconnection not allowed with neighbor opposite node 2 ; \
3 = reconnection not allowed with neighbor opposite node 1 or 2 ; \
4 = reconnection not allowed with neighbor opposite node 3 ; \
5 = reconnection not allowed with neighbor opposite node 3 or 1 ; \
6 = reconnection not allowed with neighbor opposite node 3 or 2 ; \
7 = reconnection not allowed with any neighbor ; \
; \
Note that regardless of the Rec_Flag value no reconnection is \
allowed between faces of differing surface IDs or between faces \
that would result in discontinuity or poor quality as defined by \
angrbfdd, angrbfdd2, and angrbfmxp. Typically the \
reconnection flag is set to 0 or 7. \
The Rec_Flag should be set to 0 unless a value of 7 is required \
to match another domain. \
Commas are required between the reconnection flags in the list. \
If only one reconnection flag is specified then it must be terminated with a comma. \
The default is 0 if not set within the input surface grid. \
   ",
     4, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return (ierr);
 
    ierr = ug_set_int_param_struct ("Rec_IDs", "-rec_ids",
     "List of IDs to reset Reconnection flag.",
"\
If the Rec_IDs parameter vector is set \
then the reconnection flag is set to on, \
allowing reconnection in all directions, \
for all faces with a surface ID in the \
vector and it is set to off, allowing no \
reconnection, for all faces with a \
surface ID not in the vector. \
For example, if Rec_IDs is set to the \
vector 3,5,6 then there are 3 vector \
entries and all faces with surface IDs \
3, 5, or 6 will be turned on and all \
others will be turned off. \
Note that this parameter is a simplified form of \
using the BC_IDs and Rec_Flag parameters. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
   ",
     3, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return (ierr);

    ierr = ug_set_int_param_struct ("Std_IDs", "-stds -std_ids",
     "List of IDs to for Solid surfaces.",
     "\
If the vector STD_IDs is set then the grid BC flag is set to a \
value specifying a Solid surface for all faces with a surface ID \
in the list. \
For example, if the vector STD_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to Solid surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return (ierr);
 
    ierr = ug_set_int_param_struct ("Trnsp_BL_IDs", "-trnsp_bl_ids",
     "List of IDs for BL Generating Transparent surfaces.",
     "\
If the vector Trnsp_BL_IDs is set then the grid BC flag is set to a \
value specifying a BL Generating Transparent surface for \
all faces with a surface ID in the list. \
For example, if the vector Trnsp_BL_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to Transparent BL Generating surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;
 
    if (code == 3 || code == 4 || code == -3 || code == 43 || code == 44)
      ierr = ug_set_int_param_struct ("Trnsp_BL_Int_IDs", "-trnsp_bl_int_ids",
       "List of IDs for BL Intersecting Transparent surfaces.",
       "\
If the vector Trnsp_BL_Int_IDs is set then the grid BC flag is set to a \
value specifying a BL Intersecting Transparent surface for \
all faces with a surface ID in the list. \
For example, if the vector Trnsp_BL_Int_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to Transparent BL Intersecting surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
       ",
       1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;
 
    ierr = ug_set_int_param_struct ("Trnsp_Intl_BL_IDs", "-trnsp_intl_bl_ids",
     "List of IDs for BL Generating Internal Transparent surfaces.",
     "\
If the vector Trnsp_Intl_BL_IDs is set then the grid BC flag is set to a \
value specifying a BL Generating Internal Transparent surface for \
all faces with a surface ID in the list. \
For example, if the vector Trnsp_Intl_BL_IDs is set to \
2,5,6 then there are 3 entries in the vector and all faces with surface \
IDs 2, 5 or 6 will be set to BL Generating Internal Transparent surfaces. \
Commas are required between the surface IDs in the list. \
If only one ID is specified then it must be terminated with a comma. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

    if (ierr) return ierr;
  }

  if (code == 3 || code == 5 || code == -3 || code == 43)
    ierr = ug_set_int_param_struct ("PS_IDs", "-psids",
     "ID pairs for periodic surfaces.",
"\
For each periodic surface pair the parent and child surface IDs must be \
specified in the list of periodic surface IDs, PS_IDs. The list PS_IDs \
contains the list of IDs \
for each parent and child surface. For example, if the list PS_IDs is set to \
5,4,12,2 then there are 4 entries in the vector (must be an even number) and \
two periodic surface pairs. The first pair includes the surface faces with IDs \
5 and 4 and the second includes the surface faces with IDs 12 and 2. \
Commas are required between the surface IDs in the list. \
All IDs set as periodic must have their Grid BC set to a BL Intersecting \
surfaces using BL_Int_IDs or Grid_BC_Flag and BC_IDs. \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

  if (ierr) return ierr;

  if (code == 5)
    ierr = ug_set_int_param_struct ("PS_Nodes", "-psns",
     "Node pairs for periodic surfaces.",
"\
For each periodic surface pair a boundary edge vertex/node pair must be \
specified in the list of periodic vertex pairs, PS_Nodes. Each vertex/node \
pair must be for a set of vertices that are on a respective surface boundary \
edge and represent a pair that would be an identical match after \
periodic translation and rotation. \
The list PS_Nodes contains a list of periodic nodes \
for each parent and child surface. For example, if the list PS_Nodes is set to \
3,7,18,32 then there are 4 entries in the vector (must be an even number) and \
two periodic surface pairs. The first pair includes the surface faces with \
nodes 3 and 7 and the second includes the surface faces with Nodes 18 and 32. \
If the surface IDs were set with the example used in the description for \
PS_IDs then node 3 is on surface ID 5, node 7 is on surface ID 4, \
node 18 is on surface ID 12, and node 32 is on surface ID 2. \
Commas are required between the surface Nodes in the list. ; \
     ",
     1, 1, 0, 0, 0, Param_Struct_Ptr);

  if (ierr) return ierr;

  // set new minimum and default values for the message flag for AFLR4 

  if (code == 4 || code == 44) {

    Index = ug_get_int_param_ ("Message_Flag", &Message_Flag, Param_Struct_Ptr);

    if (Index >= 0)
    {
      Param_Struct_Ptr->Int_Param[Index] = -1;
      Param_Struct_Ptr->Def_Int_Param[Index] = -1;
      Param_Struct_Ptr->Min_Int_Param[Index] = -1;
    }
  }

  ug_set_param_def (Param_Struct_Ptr);

  return (0);

}
