#include "UG_LIB.h"

/*
 * Routines to add arguments.
 *
 * UG LIB : Unstructured Grid - General Purpose Routine Library
 * $Id: ug_add_arg.c,v 1.16 2022/11/21 01:10:49 marcum Exp $
 * Copyright 1994-2021, David L. Marcum
 */

INT_ ug_add_char_arg
 (char string[],
  int *argc,
  char ***argv)
{
  // Add a type char argument.

  return (ug_add_flag_arg (string, argc, argv));
}

INT_ ug_add_double_arg
 (double value,
  int *argc,
  char ***argv)
{
  // Add a type double argument.

  CHAR_UG_MAX text;
  snprintf (text, sizeof(text), "%.15g", value);
  ++(*argc);
  return (ug_add_new_arg (argv, text));
}

INT_ ug_add_double_vector_arg
 (INT_ n,
  DOUBLE_1D *vector,
  int *argc,
  char ***argv)
{
  // Add a type double vector argument.

  char *new_arg = NULL;
  CHAR_UG_MAX text;
  INT_ i;
  INT_ ierr = 0;
  INT_ nc = 30;

  new_arg = (char *) ug_malloc ((INT_ *) &ierr, (n*nc + 1) * sizeof (char));

  if (ierr)
  {
    ug_free (new_arg);
    ug_error_message ("*** ERROR 100428 : unable to allocate space for double vector argument ***");
    return (100428);
  }

  strcpy (new_arg, "");

  for (i = 0; i < n; ++i)
  {
    snprintf (text, sizeof(text), "%.15g", vector[i]);

    strcat (new_arg, text);

    if (i < n-1 || n == 1) strcat (new_arg, ",");
  }

  ++(*argc);

  ierr = ug_add_new_arg (argv, new_arg);

  ug_free (new_arg);

  return (ierr);
}

INT_ ug_add_flag_arg
 (char flag[],
  int *argc,
  char ***argv)
{
  // Add a flag argument.

  ++(*argc);
  return (ug_add_new_arg (argv, flag));
}

INT_ ug_add_int_arg
 (INT_ value,
  int *argc,
  char ***argv)
{
  // Add a type int argument.

  CHAR_UG_MAX text;
  snprintf (text, sizeof(text), "%i", (int) value);
  ++(*argc);
  return (ug_add_new_arg (argv, text));
}

INT_ ug_add_int_vector_arg
 (INT_ n,
  INT_1D *vector,
  int *argc,
  char ***argv)
{
  // Add a type int vector argument.

  char *new_arg = NULL;

  CHAR_UG_MAX text;

  INT_ i;
  INT_ ierr = 0;
  INT_ nc = 30;

  new_arg = (char *) ug_malloc ((INT_ *) &ierr, (n*nc + 1) * sizeof (char));

  if (ierr)
  {
    ug_free (new_arg);
    ug_error_message ("*** ERROR 100429 : unable to allocate required memory ***");
    return (100429);
  }

  strcpy (new_arg, "");

  for (i = 0; i < n; ++i)
  {
    snprintf (text, sizeof(text), "%i", (int) vector[i]);

    strcat (new_arg, text);

    if (i < n-1 || n == 1) strcat (new_arg, ",");
  }

  ++(*argc);

  ierr = ug_add_new_arg (argv, new_arg);

  ug_free (new_arg);

  return (ierr);
}

INT_ ug_add_char_param_arg
 (char name[],
  char value[],
  int *argc,
  char ***argv)
{
  // Add a type char parameter argument.

  CHAR_UG_MAX new_arg;
  ++(*argc);
  snprintf (new_arg, sizeof(new_arg), "%s=%s", name, value);
  return (ug_add_new_arg (argv, new_arg));
}

INT_ ug_add_double_param_arg
 (char name[],
  double value,
  int *argc,
  char ***argv)
{
  // Add a type double parameter argument.

  CHAR_UG_MAX new_arg;
  ++(*argc);
  snprintf (new_arg, sizeof(new_arg), "%s=%.15g", name, value);
  return (ug_add_new_arg (argv, new_arg));
}

INT_ ug_add_double_vector_param_arg
 (char name[],
  INT_ n,
  DOUBLE_1D *vector,
  int *argc,
  char ***argv)
{
  // Add a type double vector parameter argument.

  CHAR_UG_MAX new_arg, string;
  INT_ i;
  ++(*argc);
  snprintf (new_arg, sizeof(new_arg), "%s=", name);
  for (i = 0; i < n; ++i) {
    snprintf (string, sizeof(string), "%.15g", vector[i]);
    strcat (new_arg, string);
    if (i < n-1 || n == 1) strcat (new_arg, ",");
  }
  return (ug_add_new_arg (argv, new_arg));
}

INT_ ug_add_flag_param_arg
 (char flag[],
  int *argc,
  char ***argv)
{
  // Add a flag parameter argument.

  ++(*argc);
  return (ug_add_new_arg (argv, flag));
}

INT_ ug_add_int_param_arg
 (char name[],
  INT_ value,
  int *argc,
  char ***argv)
{
  // Add a type int parameter argument.

  CHAR_UG_MAX new_arg;
  ++(*argc);
  snprintf (new_arg, sizeof(new_arg), "%s=%i", name, (int) value);
  return (ug_add_new_arg (argv, new_arg));
}

INT_ ug_add_int_vector_param_arg
 (char name[],
  INT_ n,
  INT_1D *vector,
  int *argc,
  char ***argv)
{
  // Add a type int vector parameter argument.

  CHAR_UG_MAX new_arg, string;
  INT_ i;
  ++(*argc);
  snprintf (new_arg, sizeof(new_arg), "%s=", name);
  for (i = 0; i < n; ++i) {
    snprintf (string, sizeof(string), "%i", (int) vector[i]);
    strcat (new_arg, string);
    if (i < n-1 || n == 1) strcat (new_arg, ",");
  }
  return (ug_add_new_arg (argv, new_arg));
}

INT_ ug_add_new_arg
 (char ***argv,
  char *new_arg)
{
  // Add a new argument to the argument vector.

  int argc;
  INT_ ierr = 0;
  
  if (strcmp (new_arg, "") == 0)
    return (0);

  if (strcmp (new_arg, "allocate_and_initialize_argv") == 0)
  {
    argc = 1;

    *argv = (char **) ug_malloc ((INT_ *) &ierr, (argc+1) * sizeof (char *));

    if (ierr)
    {
      ug_free (*argv);
      *argv = NULL;
      ug_error_message ("*** ERROR 100427 : unable to allocate required memory ***");
      return (100427);
    }

    (*argv)[argc-1] = (char *) ug_malloc ((INT_ *) &ierr, sizeof (char));

    if (ierr)
    {
      ug_free ((*argv)[0]);
      ug_free (*argv);
      *argv = NULL;
      ug_error_message ("*** ERROR 100427 : unable to allocate required memory ***");
      return (100427);
    }

    strcpy ((*argv)[0], "\0");

    (*argv)[argc] = NULL;

    return (0);
  }

  for (argc = 0; (*argv)[argc] != NULL; argc++);
  
  ++argc;

  *argv = (char **) ug_realloc ((INT_ *) &ierr, *argv, (argc+1) * sizeof (char *));

  if (ierr)
  {
    ug_error_message ("*** ERROR 100423 : unable to allocate required memory ***");
    return (100423);
  }
 
  (*argv)[argc-1] = (char *) ug_malloc ((INT_ *) &ierr, (strlen (new_arg) + 1) * sizeof (char));

  if (ierr)
  {
    (*argv)[argc-1] = NULL;
    ug_error_message ("*** ERROR 100424 : unable to allocate required memory ***");
    return (100424);
  }

  strcpy ((*argv)[argc-1], new_arg);

  (*argv)[argc] = NULL;

  return (0);
}

INT_ ug_add_list_arg
 (char list[],
  int *argc,
  char ***argv)
{
  // Add argument entries from a list of arguments.

  char *entry;
  char *sep = " ";
  INT_ ierr = 0;
  if (*argc == 0) {
    ierr = ug_add_new_arg (argv, "allocate_and_initialize_argv");
    if (ierr) return (ierr);
    *argc = 1;
  }
  if (list) {
    for (entry = strtok (list, sep); entry && ierr == 0; entry = strtok (NULL, sep)) {
      ierr = ug_add_flag_arg (entry, argc, argv);
    }
  }
  return (ierr);
}
