#include "mesh_ug_io.h"
#include "libmeshb7.h"

INT_ mesh_ug_io_write_func (
  char File_Name[],
  INT_ NDim,
  INT_ Number_of_Nodes,
  INT_ Number_of_U_Scalars,
  INT_ Number_of_U_Vectors,
  INT_ Number_of_U_Metrics,
  INT_ Number_of_U_SymMetrics,
  DOUBLE_1D * U_Scalars,
  DOUBLE_1D * U_Vectors,
  DOUBLE_1D * U_Metrics,
  DOUBLE_1D * U_SymMetrics)
{

/*
 * Write function data to a SOL solution file.
 * 
 * $Id: mesh_ug_io_write_func.c,v 1.9 2024/11/14 06:31:36 marcum Exp $
 */

  int TypTab[ GmfMaxTyp ];
  int FilVer;

  LONG_LONG_int Sol_File_Index;

  INT_ i, j, n, Index, Number_of_Functions;

  double bufDbl[9];
  double Size;
  double SizeLimit = 1.95;

  // set number of functions and type

  if (U_Scalars == NULL) Number_of_U_Scalars = 0;
  if (U_Vectors == NULL) Number_of_U_Vectors = 0;
  if (U_Metrics == NULL) Number_of_U_Metrics = 0;
  if (U_SymMetrics == NULL) Number_of_U_SymMetrics = 0;

  Number_of_Functions = Number_of_U_Scalars + Number_of_U_Vectors
                      + Number_of_U_Metrics + Number_of_U_SymMetrics;

  // check input data

  if (Number_of_Functions == 0 ) {
    ug_error_message ("*** ERROR : no function fields specified for SOL file ***");
    return 1;
  }

  if (Number_of_Nodes == 0 ) {
    ug_error_message ("*** ERROR : no vertices/nodes specified for SOL file ***");
    return 1;
  }

  // set function type table

  j = 0;

  for (i = j; i < j + (int) (Number_of_U_Scalars); i++) {
    TypTab[i] = GmfSca;  
  }

  j = j + (int) (Number_of_U_Scalars);

  for (i = j; i < j + (int) (Number_of_U_Vectors); i++) {
    TypTab[i] = GmfVec;  
  }

  j = j + (int) (Number_of_U_Vectors);

  for (i = j; i < j + (int) (Number_of_U_Metrics); i++) {
    TypTab[i] = GmfMat;  
  }

  j = j + (int) (Number_of_U_Metrics);
  
  for (i = j; i < j + (int) (Number_of_U_SymMetrics); i++) {
    TypTab[i] = GmfSymMat;  
  }

  j = j + (int) (Number_of_U_SymMetrics);

  // set file version

  Size = ((double) ((sizeof (DOUBLE_1D)) * Number_of_U_Scalars * Number_of_Nodes
                  + (sizeof (DOUBLE_3D)) * Number_of_U_Vectors * Number_of_Nodes
                  + (sizeof (DOUBLE_9D)) * Number_of_U_Metrics * Number_of_Nodes
                  + (sizeof (DOUBLE_6D)) * Number_of_U_SymMetrics * Number_of_Nodes))
       / ((double) (1024 * 1024 * 1024));

  FilVer = (sizeof (INT_) == 8) ? 4: 
           (Size > SizeLimit) ? 3: 2;

  // write function data

  Sol_File_Index = GmfOpenMesh(File_Name, GmfWrite, FilVer, (int) NDim); 
  
  if ( Sol_File_Index <= 0 ) {
    ug_error_message ("*** ERROR : error opening SOL file ***");
    return 1;
  }
  
  GmfSetKwd(Sol_File_Index, GmfSolAtVertices, Number_of_Nodes, Number_of_Functions, TypTab);

  for (Index = 1; Index <= Number_of_Nodes; Index++) {

    for (i = 0; i < Number_of_U_Scalars; i++)
      bufDbl[i] = U_Scalars[Index+i*Number_of_Nodes];

    GmfSetLin(Sol_File_Index, GmfSolAtVertices, bufDbl);
  }

  n = NDim;

  for (Index = 1; Index <= Number_of_Nodes; Index++) {

    for (i = 0; i < Number_of_U_Vectors; i++) {
      for (j = 0; j < n; j++)
        bufDbl[j+i*n] = U_Vectors[1+j+(Index-1)*n+i*n*Number_of_Nodes];
    }

    GmfSetLin(Sol_File_Index, GmfSolAtVertices, bufDbl);
  }

  n = NDim*NDim;

  for (Index = 1; Index <= Number_of_Nodes; Index++) {

    for (i = 0; i < Number_of_U_Metrics; i++) {
      for (j = 0; j < n; j++)
        bufDbl[j+i*n] = U_Metrics[1+j+(Index-1)*n+i*n*Number_of_Nodes];
    }

    GmfSetLin(Sol_File_Index, GmfSolAtVertices, bufDbl);
  }

  n = 3*(NDim-1);

  for (Index = 1; Index <= Number_of_Nodes; Index++) {

    for (i = 0; i < Number_of_U_SymMetrics; i++) {

      for (j = 0; j < n; j++)
        bufDbl[j+i*n] = U_SymMetrics[1+j+(Index-1)*n+i*n*Number_of_Nodes];

      // WARNING: Gamma convention switch 2 and 3

      if (NDim == 3)
      {
        bufDbl[2+i*n] = U_SymMetrics[1+3+(Index-1)*n+i*n*Number_of_Nodes];
        bufDbl[3+i*n] = U_SymMetrics[1+2+(Index-1)*n+i*n*Number_of_Nodes];
        bufDbl[2] = U_SymMetrics[3+Index*n];
        bufDbl[3] = U_SymMetrics[2+Index*n];
      }
    }

    GmfSetLin(Sol_File_Index, GmfSolAtVertices, bufDbl);
  }

  GmfCloseMesh(Sol_File_Index);

  return 0;
}
