#include "cgns_ug_io.h"
#include "cgnslib.h"

INT_ cgns_ug_io_read_grid
 (char Grid_File_Name[],
  INT_ Read_Task_Flag,
  INT_ *Number_of_BL_Vol_Tets,
  INT_ *Number_of_Bnd_Nodes,
  INT_ *Number_of_Nodes,
  INT_ *Number_of_Surf_Quads,
  INT_ *Number_of_Surf_Trias,
  INT_ *Number_of_Vol_Hexs,
  INT_ *Number_of_Vol_Pents_5,
  INT_ *Number_of_Vol_Pents_6,
  INT_ *Number_of_Vol_Tets,
  INT_1D * Surf_Grid_BC_Flag,
  INT_1D * Surf_ID_Flag,
  INT_4D * Surf_Quad_Connectivity,
  INT_1D * Surf_Reconnection_Flag,
  INT_3D * Surf_Tria_Connectivity,
  INT_8D * Vol_Hex_Connectivity,
  INT_1D * Vol_ID_Flag,
  INT_5D * Vol_Pent_5_Connectivity,
  INT_6D * Vol_Pent_6_Connectivity,
  INT_4D * Vol_Tet_Connectivity,
  DOUBLE_3D * Coordinates,
  DOUBLE_1D * Initial_Normal_Spacing,
  DOUBLE_1D * BL_Thickness)

{

/*
 * Read grid data from a CGNS file.
 * 
 * UG_IO LIB : Unstructured Grid - Input/Output Routine Library
 * $Id: cgns_ug_io_read_grid.c,v 1.7 2016/02/19 05:45:25 marcum Exp $
 */

  char Label[33];

  INT_ Index, Index_1, Index_2, Index_3, Index_4, Index_5, Index_Surf_Quads,
       Index_Surf_Trias, Index_Vol_Hexs, Index_Vol_Pents_5, Index_Vol_Pents_6,
       Index_Vol_Tets, Number_of_Elems_Faces, Number_of_Zone_Elems_Faces,
       Surf_ID;
  
  INT_ Error_Flag = 0;

  int *Connectivity = NULL;
  int *Parent_Data = NULL;

  int Zone_Size[3][3];

  int Array_Size, Base_Index, Cell_Dim, End_Index, File_Index, max_Index,
      min_Index, Number_of_Bases, Number_of_Boundary_Items, Number_of_Sections,
      Number_of_Zones, Parent_Flag, Phys_Dim, Section_Index, Start_Index,
      Zone_Index;

  ElementType_t Element_Type;

  ZoneType_t Zone_Type;

  double *Coordinate = NULL;

  Error_Flag = cg_open (Grid_File_Name, CG_MODE_READ, &File_Index);

  if (Error_Flag > 0 || File_Index < 0)
  {
    ug_error_message ("*** ERROR : unable to open CGNS grid file ***");
    return (1);
  }

  Error_Flag = cg_nbases (File_Index, &Number_of_Bases);

  if (Error_Flag > 0)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : error reading CGNS grid file ***");
    return (1);
  }

  if (Number_of_Bases != 1)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : only one base allowed in CGNS grid file ***");
    return (1);
  }

  Base_Index = 1;

  Error_Flag = cg_base_read (File_Index, Base_Index,
                             Label, &Cell_Dim, &Phys_Dim);

  if (Error_Flag > 0)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : error reading CGNS grid file ***");
    return (1);
  }

  if (Cell_Dim != 3 || Phys_Dim != 3)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : only 3D cell and physical dimensions are allowed in CGNS file ***");
  }

  Error_Flag = cg_nzones (File_Index, Base_Index, &Number_of_Zones);

  if (Error_Flag > 0)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : error reading CGNS grid file ***");
    return (1);
  }

  if (Number_of_Zones != 1)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : only one zone is allowed in CGNS grid file ***");
    return (1);
  }

  Zone_Index = 1;

  Error_Flag = cg_zone_type (File_Index, Base_Index, Zone_Index, &Zone_Type);

  if (Error_Flag > 0)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : error reading CGNS grid file ***");
    return (1);
  }

  if (Zone_Type != Unstructured)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : only an unstructured zone type is allowed in CGNS grid file ***");
    return (1);
  }

  Error_Flag = cg_zone_read (File_Index, Base_Index, Zone_Index, Label,
                             *Zone_Size);

  if (Error_Flag > 0)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : error reading CGNS grid file ***");
    return (1);
  }

  *Number_of_Nodes = (INT_) Zone_Size[0][0];

  if (*Number_of_Nodes < 1)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : no coordinates nodes specified in CGNS grid file ***");
    return (1);
  }

  Error_Flag = cg_nsections (File_Index, Base_Index, Zone_Index,
                             &Number_of_Sections);

  if (Error_Flag > 0)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : error reading CGNS grid file ***");
    return (1);
  }

  if (Number_of_Sections <= 0)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : no element sections are specified in CGNS grid file ***");
    return (1);
  }

  *Number_of_Surf_Trias = 0;
  *Number_of_Surf_Quads = 0;
  *Number_of_Vol_Tets = 0;
  *Number_of_Vol_Pents_5 = 0;
  *Number_of_Vol_Pents_6 = 0;
  *Number_of_Vol_Hexs = 0;

  for (Section_Index = 1; Section_Index <= Number_of_Sections; ++Section_Index)
  {
    Error_Flag = cg_section_read (File_Index, Base_Index, Zone_Index,
                                  Section_Index, Label, &Element_Type,
                                  &Start_Index, &End_Index,
                                  &Number_of_Boundary_Items, &Parent_Flag);

    if (Error_Flag > 0)
    {
      cg_close(File_Index);
      ug_error_message ("*** ERROR : error reading CGNS grid file ***");
      return (1);
    }

    if (Parent_Flag != 0)
    {
      cg_close(File_Index);
      ug_error_message ("*** ERROR : parent data not allowed in CGNS grid file ***");
      return (1);
    }

    if (Element_Type == TRI_3)
      *Number_of_Surf_Trias = *Number_of_Surf_Trias
                            + End_Index - Start_Index + 1;

    else if (Element_Type == QUAD_4)
      *Number_of_Surf_Quads = *Number_of_Surf_Quads
                            + End_Index - Start_Index + 1;

    else if (Element_Type == TETRA_4)
      *Number_of_Vol_Tets = *Number_of_Vol_Tets
                          + End_Index - Start_Index + 1;

    else if (Element_Type == PYRA_5)
      *Number_of_Vol_Pents_5 = *Number_of_Vol_Pents_5
                             + End_Index - Start_Index + 1;

    else if (Element_Type == PENTA_6)
      *Number_of_Vol_Pents_6 = *Number_of_Vol_Pents_6
                             + End_Index - Start_Index + 1;

    else if (Element_Type == HEXA_8)
      *Number_of_Vol_Hexs = *Number_of_Vol_Hexs
                          + End_Index - Start_Index + 1;

    else
    {
      cg_close(File_Index);
      ug_error_message ("*** ERROR : unknown element type found in CGNS grid file ***");
      return (1);
    }
  }

  Number_of_Elems_Faces = *Number_of_Vol_Tets
                        + *Number_of_Vol_Pents_5 + *Number_of_Vol_Pents_6
                        + *Number_of_Vol_Hexs
                        + *Number_of_Surf_Trias + *Number_of_Surf_Quads;

  Number_of_Zone_Elems_Faces = (INT_) Zone_Size[0][1];

  if (Number_of_Elems_Faces != Number_of_Zone_Elems_Faces)
  {
    cg_close(File_Index);
    ug_error_message ("*** ERROR : number of elements and faces in zone not equal to number in CGNS grid file ***");
    return (1);
  }

  if (Read_Task_Flag == 1 && *Number_of_Bnd_Nodes == -1)
  {
    Connectivity = (int *) ug_malloc (&Error_Flag,
                                      MAX (3*(*Number_of_Surf_Trias),
                                           4*(*Number_of_Surf_Quads))
                                    * sizeof (int));

    if (Error_Flag > 0)
    {
      ug_free (Connectivity);
      cg_close(File_Index);
      ug_error_message ("*** ERROR : unable to malloc CGNS I/O work array ***");
      return (1);
    }

    for (Section_Index = 1;
         Section_Index <= Number_of_Sections; ++Section_Index)
    {
      Error_Flag = cg_section_read (File_Index, Base_Index, Zone_Index,
                                    Section_Index, Label, &Element_Type,
                                    &Start_Index, &End_Index,
                                    &Number_of_Boundary_Items, &Parent_Flag);

      if (Error_Flag > 0)
      {
        ug_free (Connectivity);
        cg_close(File_Index);
        ug_error_message ("*** ERROR : error reading CGNS grid file ***");
        return (1);
      }

      if (Element_Type == TRI_3)
      {
        Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                       Section_Index,
                                       Connectivity, Parent_Data);

        if (Error_Flag > 0)
        {
          ug_free (Connectivity);
          cg_close(File_Index);
          ug_error_message ("*** ERROR : error reading CGNS grid file ***");
          return (1);
        }

        Index_Surf_Trias = 0;

        for (Index = 1; Index <= End_Index-Start_Index+1; ++Index)
        {
          *Number_of_Bnd_Nodes = MAX (*Number_of_Bnd_Nodes,
                                      Connectivity[Index_Surf_Trias]);
          *Number_of_Bnd_Nodes = MAX (*Number_of_Bnd_Nodes,
                                      Connectivity[Index_Surf_Trias+1]);
          *Number_of_Bnd_Nodes = MAX (*Number_of_Bnd_Nodes,
                                      Connectivity[Index_Surf_Trias+2]);

          Index_Surf_Trias = Index_Surf_Trias+3;
        }
      }

      else if (Element_Type == QUAD_4)
      {
        Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                       Section_Index,
                                       Connectivity, Parent_Data);

        if (Error_Flag > 0)
        {
          ug_free (Connectivity);
          cg_close(File_Index);
          ug_error_message ("*** ERROR : error reading CGNS grid file ***");
          return (1);
        }

        Index_Surf_Quads = 0;

        for (Index = 1; Index <= End_Index-Start_Index+1; ++Index)
        {
          *Number_of_Bnd_Nodes = MAX (*Number_of_Bnd_Nodes,
                                      Connectivity[Index_Surf_Quads]);
          *Number_of_Bnd_Nodes = MAX (*Number_of_Bnd_Nodes,
                                      Connectivity[Index_Surf_Quads+1]);
          *Number_of_Bnd_Nodes = MAX (*Number_of_Bnd_Nodes,
                                      Connectivity[Index_Surf_Quads+2]);
          *Number_of_Bnd_Nodes = MAX (*Number_of_Bnd_Nodes,
                                      Connectivity[Index_Surf_Quads+3]);

          Index_Surf_Quads = Index_Surf_Quads+4;
        }
      }
    }

    ug_free (Connectivity);
  }

  if (*Number_of_Bnd_Nodes > 0)
    *Number_of_Nodes = *Number_of_Bnd_Nodes;

  if (Read_Task_Flag == 2)
  {
    Error_Flag = 0;

    Array_Size = (*Number_of_Nodes) * (INT_) sizeof (double);

    Coordinate = (double *) ug_malloc (&Error_Flag, Array_Size);

    if (Error_Flag > 0)
    {
      ug_free (Coordinate);
      cg_close(File_Index);
      ug_error_message ("*** ERROR : unable to malloc CGNS I/O work array ***");
      return (1);
    }

    min_Index = 1;
    max_Index = (int) *Number_of_Nodes;
 
    Error_Flag = cg_coord_read (File_Index, Base_Index, Zone_Index,
                                "CoordinateX", RealDouble,
                                &min_Index, &max_Index, Coordinate);

    if (Error_Flag > 0)
    {
      ug_free (Coordinate);
      cg_close(File_Index);
      ug_error_message ("*** ERROR : error reading CGNS grid file ***");
      return (1);
    }
 
    for (Index = 1; Index <= *Number_of_Nodes; ++Index)
    {
      Coordinates[Index][0] = Coordinate[Index-1];
    }  

    min_Index = 1;
    max_Index = (int) *Number_of_Nodes;

    Error_Flag = cg_coord_read (File_Index, Base_Index, Zone_Index,
                                "CoordinateY", RealDouble,
                                &min_Index, &max_Index, Coordinate);

    if (Error_Flag > 0)
    {
      ug_free (Coordinate);
      cg_close(File_Index);
      ug_error_message ("*** ERROR : error reading CGNS grid file ***");
      return (1);
    }
 
    for (Index = 1; Index <= *Number_of_Nodes; ++Index)
    {
      Coordinates[Index][1] = Coordinate[Index-1];
    }  

    min_Index = 1;
    max_Index = (int) *Number_of_Nodes;

    Error_Flag = cg_coord_read (File_Index, Base_Index, Zone_Index,
                                "CoordinateZ", RealDouble,
                                &min_Index, &max_Index, Coordinate);

    if (Error_Flag > 0)
    {
      ug_free (Coordinate);
      cg_close(File_Index);
      ug_error_message ("*** ERROR : error reading CGNS grid file ***");
      return (1);
    }
 
    for (Index = 1; Index <= *Number_of_Nodes; ++Index)
    {
      Coordinates[Index][2] = Coordinate[Index-1];
    }  
 
    ug_free (Coordinate);

    Index_Surf_Trias = 0;
    Index_Surf_Quads = 0;
    Index_Vol_Tets = 0;
    Index_Vol_Pents_5 = 0;
    Index_Vol_Pents_6 = 0;
    Index_Vol_Hexs = 0;

    Surf_ID = 0;

    for (Section_Index = 1;
         Section_Index <= Number_of_Sections; ++Section_Index)
    {
      Error_Flag = cg_section_read (File_Index, Base_Index, Zone_Index,
                                    Section_Index, Label, &Element_Type,
                                    &Start_Index, &End_Index,
                                    &Number_of_Boundary_Items, &Parent_Flag);

      if (Error_Flag > 0)
      {
        cg_close(File_Index);
        ug_error_message ("*** ERROR : error reading CGNS grid file ***");
        return (1);
      }

      if (Element_Type == TRI_3)
      {
        Connectivity = &(Surf_Tria_Connectivity[Index_Surf_Trias+1][0]);

        Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                       Section_Index,
                                       Connectivity, Parent_Data);

        if (Error_Flag > 0)
        {
          cg_close(File_Index);
          ug_error_message ("*** ERROR : error reading CGNS grid file ***");
          return (1);
        }

        ++Surf_ID;

        Index_1 = Index_Surf_Trias + 1;
        Index_2 = Index_Surf_Trias + End_Index - Start_Index + 1;

        for (Index = Index_1; Index <= Index_2; ++Index)
        {
          Surf_ID_Flag[Index] = Surf_ID;
        }

        Index_Surf_Trias = Index_2;
      }

      else if (Element_Type == QUAD_4)
      {
        Connectivity = &(Surf_Quad_Connectivity[Index_Surf_Quads+1][0]);

        Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                       Section_Index,
                                       Connectivity, Parent_Data);

        if (Error_Flag > 0)
        {
          cg_close(File_Index);
          ug_error_message ("*** ERROR : error reading CGNS grid file ***");
          return (1);
        }

        ++Surf_ID;

        Index_1 = Index_Surf_Quads + 1;
        Index_2 = Index_Surf_Quads + End_Index - Start_Index + 1;

        for (Index = Index_1; Index <= Index_2; ++Index)
        {
          Surf_ID_Flag[*Number_of_Surf_Trias+Index] = Surf_ID;
        }

        Index_Surf_Quads = Index_2;
      }

      else if (Element_Type == TETRA_4)
      {
        if (*Number_of_Vol_Tets > 0 && *Number_of_Bnd_Nodes == 0)
        {
          Connectivity = &(Vol_Tet_Connectivity[Index_Vol_Tets+1][0]);

          Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                         Section_Index,
                                         Connectivity, Parent_Data);

          if (Error_Flag > 0)
          {
            cg_close(File_Index);
            ug_error_message ("*** ERROR : error reading CGNS grid file ***");
            return (1);
          }

          Index_Vol_Tets = Index_Vol_Tets + End_Index - Start_Index + 1;
        }
      }

      else if (Element_Type == PYRA_5)
      {
        if (*Number_of_Vol_Pents_5 > 0 && *Number_of_Bnd_Nodes == 0)
        {
          Connectivity = &(Vol_Pent_5_Connectivity[Index_Vol_Pents_5+1][0]);

          Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                         Section_Index,
                                         Connectivity, Parent_Data);

          if (Error_Flag > 0)
          {
            cg_close(File_Index);
            ug_error_message ("*** ERROR : error reading CGNS grid file ***");
            return (1);
          }

          Index_1 = Index_Vol_Pents_5 + 1;
          Index_2 = Index_Vol_Pents_5 + End_Index - Start_Index + 1;

          for (Index = Index_1; Index <= Index_2; ++Index)
          {
            Index_1 = Vol_Pent_5_Connectivity[Index][0];
            Index_2 = Vol_Pent_5_Connectivity[Index][1];
            Index_3 = Vol_Pent_5_Connectivity[Index][2];
            Index_4 = Vol_Pent_5_Connectivity[Index][3];
            Index_5 = Vol_Pent_5_Connectivity[Index][4];

            Vol_Pent_5_Connectivity[Index][0] = Index_2;
            Vol_Pent_5_Connectivity[Index][1] = Index_1;
            Vol_Pent_5_Connectivity[Index][2] = Index_5;
            Vol_Pent_5_Connectivity[Index][3] = Index_3;
            Vol_Pent_5_Connectivity[Index][4] = Index_4;
          }

          Index_Vol_Pents_5 = Index_Vol_Pents_5 + End_Index - Start_Index + 1;
        }
      }

      else if (Element_Type == PENTA_6)
      {
        if (*Number_of_Vol_Pents_6 > 0 && *Number_of_Bnd_Nodes == 0)
        {
          Connectivity = &(Vol_Pent_6_Connectivity[Index_Vol_Pents_6+1][0]);

          Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                         Section_Index,
                                         Connectivity, Parent_Data);

          if (Error_Flag > 0)
          {
            cg_close(File_Index);
            ug_error_message ("*** ERROR : error reading CGNS grid file ***");
            return (1);
          }

          Index_Vol_Pents_6 = Index_Vol_Pents_6 + End_Index - Start_Index + 1;
        }
      }

      else if (Element_Type == HEXA_8)
      {
        if (*Number_of_Vol_Hexs > 0 && *Number_of_Bnd_Nodes == 0)
        {
          Connectivity = &(Vol_Hex_Connectivity[Index_Vol_Hexs+1][0]);

          Error_Flag = cg_elements_read (File_Index, Base_Index, Zone_Index,
                                         Section_Index,
                                         Connectivity, Parent_Data);

          if (Error_Flag > 0)
          {
            cg_close(File_Index);
            ug_error_message ("*** ERROR : error reading CGNS grid file ***");
            return (1);
          }

          Index_Vol_Hexs = Index_Vol_Hexs + End_Index - Start_Index + 1;
        }
      }

      else
      {
        cg_close(File_Index);
        ug_error_message ("*** ERROR : error reading CGNS grid file ***");
        return (1);
      }
    }
  }
  
  Error_Flag = cg_close(File_Index);

  return (0);

}
